
\name{mset_user}

\alias{mset_user}

\title{Generates Clustering Methods Settings for a Prototype Methodology
Provided by the User}

\description{
  The function generates a software abstraction of a list of clustering
  models implemented through the a set of  tuned methods and algorithms.
  The \emph{base} clustering methodology is provided via a user-defined
  function. The latter prototype is exapanded in a list  of fucntions
  each  combining tuning parameters and other algorithmic settings. 
  The generated functions are ready to be called on the data set.
}

\usage{
mset_user(fname, .packages = NULL, .export = NULL, ...)
}

\arguments{
  \item{fname}{
    a function implementing a user-defined clustering method. It clusters
    a data set and outputs cluster parameters. \code{fname}
    must fulfill certain requirements detailed below in the
    \emph{Details}. 
  }
  \item{.packages}{
    character vector of packages that the tasks in \code{fname} depend
    on (see \emph{Details}).
  }
  \item{.export}{
    character vector of variables to export that are needed by
    \code{fname} and that are not defined in the current environment
    (see \emph{Details}).

  }
  \item{...}{
    parameters passed to \code{fname}. If a given parameter is included as
    a vector/list each of its members is to obtain the final collection
    of \code{fname} specifications (see \emph{Details} and \emph{Examples}).
  }
}


\details{
  The function produces functions implementing competing clustering methods
  based on a \emph{prototype} methodology implemented by the user via
  the input argument  \code{fname}. 
  In particular, it builds a list of \code{fname}-type  functions each
  corresponding to a specific setup in terms of
  hyper-parameters (\emph{e.g.} the number of clusters) and algorithm's
  control parameters (\emph{e.g.}  initialization).

  \strong{Requirements for \code{fname}.} 
  \code{fname} is a function implementing the base clustering method of
  interest. It must have the following input argument
  %%---%%
  \itemize{
    \item \code{data:}
    a numeric vector, matrix, or data frame of observations. Rows
    correspond to observations and columns correspond to
    variables/features.
    Categorical variables and \code{NA} values are not allowed.
  }
  %%---%%
  Additionally, \code{fname} can have any other input parameter controlling
  the underlying clustering model/method/algorithm. All this additional
  parameters  are passed to \code{mset_user} via \code{...}
  (see \emph{Arguments}).

  
  The output of \code{fname} must contain a list named \code{params} 
  with cluster parameters describing size, centrality and scatter. 
  Let \code{P=}\emph{number of variable/features} and \code{K=}\emph{number of clusters}. 
  The elements  of \code{params} are as follows:
  %%---%%
  \itemize{
    \item \code{prop:} a vector of clusters' proportions; 
    \item \code{mean:} a matrix of dimension \code{(P x K)} containing the clusters' mean 
    parameters;
    \item \code{cov:} an array of size \code{(P x P x K)} containing the clusters'
    covariance matrices.
  }
  %%---%%
  Note that \code{params} can be easily obtained from a vector of cluster labels
  using \code{\link{clust2params}}.



  \strong{\code{packages} and \code{export}.} The user does not
  normally need to specify \code{packages} and \code{export}.
  These arguments are not needed if the functions generated by \code{mset_user}
  will  be called from an environment containing all variables and
  functions needed to execute \code{fname}.
  Functions like \code{\link{bqs}} will call the functions 
  by {\code{mset_user}} within a parallel infrastructure
  using \code{\link[foreach]{foreach}}. If the user specifies
  \code{packages} and \code{export}, they will be passed to the 
  \code{.packages} and \code{.export} arguments of 
  \code{\link[foreach]{foreach}}.  


  
  Finally, note that the package already contains specialized versions of \code{mset_user}
  generating methods settings for some popular algorithms
  (see \code{\link{mset_gmix}},  \code{\link{mset_kmeans}}, \code{\link{mset_pam}})
  }
 %%---%%


\value{
  An S3 object of class \code{'qcmethod'}. Each element of the list
  represents a competing method containing the following objects
  
  \item{fullname}{
    a string identifying the setup.
  }
  \item{callargs}{
    a list with arguments that are passed to the base function.
  }
  \item{fn}{
    the function implementing the specified setting. This \code{fn}
    function can be executed on the data set.
    It has two arguments: \code{data} and \code{only_params}.
    \code{data} is a data matrix or data.frame
    \code{only_params} is logical.
    If \code{only_params==FALSE} (default), \code{fn} will return the
    object returned by the \code{fname}.
    If \code{only_params==TRUE} (default) \code{fn} will return only cluster 
    parameters (proportions, mean, and cov, see \link{clust2params}).
  }
}





\references{
  Coraggio, Luca, and Pietro Coretto (2023).
  Selecting the Number of Clusters, Clustering Models, and Algorithms.
  A Unifying Approach Based on the Quadratic Discriminant Score.
  \emph{Journal of Multivariate Analysis}, Vol. 196(105181), pp. 1-20,
  \doi{10.1016/j.jmva.2023.105181}
}




\seealso{
  \code{\link{clust2params}}, \code{\link{mset_gmix}},  \code{\link{mset_kmeans}}, \code{\link{mset_pam}}
}







\examples{
# load data
data("banknote")
dat  <- banknote[-1]



# EXAMPLE 1: generate Hierarchical Clustering settings
# ----------------------------------------------------

# wrapper for the popular stats::hclust() for Hierarchical Clustering
# Note the usee:
#   of the optional arguments '...' passed to the underling clustering function
#   the use of 'clust2params' to add cluster parameters to the output 
hc_wrapper <- function(data, K, ...){ 
    dm  <- dist(data, method = "euclidean")
    ## ... = hc parameters 
    hc  <- hclust(dm, ...)
    cl  <- cutree(hc, k = K)
    ## output with params 
    res          <- list()
    res$cluster  <- cl
    res$params   <- clust2params(data, cluster = cl)
    return(res)
}

# generate settings for Hierarchical Clustering with varying
# number of clusters K={3,4},  agglomeration method = {ward.D, median}
# see help('stats::hclust')
A <- mset_user(fname="hc_wrapper", K = c(2,3), method = c("ward.D", "complete"))

# get the setting with K=2 and method = "complete"
ma <- A[[4]]
ma

# cluster data with M[[3]]
fit_a1 <- ma$fn(dat)
fit_a1

## if only cluster parameters are needed 
fit_a2 <- ma$fn(dat, only_params = TRUE)
fit_a2


 
\dontrun{
# EXAMPLE 2: generate 'mclust' model settings 
# -------------------------------------------
# mclust is popular package for performing model based clustering based on
# Gaussian mixture. Please visit
# https://cran.r-project.org/web/packages/mclust/vignettes/mclust.html
require(mclust)

# wrapper for the popular stats::hclust() for Hierarchical Clustering
# Notes:
#  * optional arguments '...' are passed to the underling
#    'mclust' clustering function
#  * 'mclust' fits Gaussian Mixture models so cluster parameters are 
#     contained in the mclust object  
mc_wrapper <- function(data, K, ...){
    y <- Mclust(data, G = K, ...)
    y[["params"]] <- list(proportion = y$parameters$pro,
                          mean = y$parameters$mean,
                          cov = y$parameters$variance$sigma)
    return(y)
    }

# generate 'mclust' model settings by varying the number of clusters and
# covariance matrix models (see help('mclust::mclustModelNames'))
B <- mset_user(fname = "mc_wrapper", K = c(2,3), modelNames = c("EEI", "VVV"))

    
# get the setting with K=3 and covariance model "EEI"
mb <- B[[2]]
mb

# cluster data with M[[3]]
fit_b <- mb$fn(dat)
fit_b ## class(fit_b) = "Mclust"

   
# if needed one can make sure that 'mclust' package is always available
# by setting the argument 'packages'
B <- mset_user(fname = "mc_wrapper", K = c(2,3), modelNames = c("EEI","VVV"),
               packages=c("mclust"))
}


\dontrun{
# EXAMPLE 3: generate 'dbscan' settings 
# -------------------------------------
# DBSCAN is popular nonparametric method for discovering clusters of
# arbitrary shapes with noise. The number of clusters is implicitly
# determined via two crucial tunings usually called 'eps' and 'minPts'
# See https://en.wikipedia.org/wiki/DBSCAN
require(dbscan)

# wrapper for dbscan::dbscan
db_wrap <- function(data, ...) {
  cl <- dbscan(data, borderPoints = TRUE, ...)$cluster
  return(params = clust2params(data, cl))
}

D  <- mset_user(fname = "db_wrap", eps = c(0.5, 1), minPts=c(5,10))
md    <- D[[2]]
fit_d <- md$fn(dat)
fit_d
class(fit_d)
}
}

