% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/map.R
\name{map}
\alias{map}
\alias{map_if}
\alias{map_at}
\alias{map_lgl}
\alias{map_chr}
\alias{map_int}
\alias{map_dbl}
\alias{map_dfr}
\alias{map_df}
\alias{map_dfc}
\alias{walk}
\title{Apply a function to each element of a vector}
\usage{
map(.x, .f, ...)

map_if(.x, .p, .f, ...)

map_at(.x, .at, .f, ...)

map_lgl(.x, .f, ...)

map_chr(.x, .f, ...)

map_int(.x, .f, ...)

map_dbl(.x, .f, ...)

map_dfr(.x, .f, ..., .id = NULL)

map_dfc(.x, .f, ...)

walk(.x, .f, ...)
}
\arguments{
\item{.x}{A list or atomic vector.}

\item{.f}{A function, formula, or atomic vector.

If a \strong{function}, it is used as is.

If a \strong{formula}, e.g. \code{~ .x + 2}, it is converted to a function. There
are three ways to refer to the arguments:
\itemize{
\item For a single argument function, use \code{.}
\item For a two argument function, use \code{.x} and \code{.y}
\item For more arguments, use \code{..1}, \code{..2}, \code{..3} etc
}

This syntax allows you to create very compact anonymous functions.

If \strong{character vector}, \strong{numeric vector}, or \strong{list}, it
is converted to an extractor function. Character vectors index by name
and numeric vectors index by position; use a list to index by position
and name at different levels. Within a list, wrap strings in \code{get_attr()}
to extract named attributes. If a component is not present, the value of
\code{.default} will be returned.}

\item{...}{Additional arguments passed on to \code{.f}.}

\item{.p}{A single predicate function, a formula describing such a
predicate function, or a logical vector of the same length as \code{.x}.
Alternatively, if the elements of \code{.x} are themselves lists of
objects, a string indicating the name of a logical element in the
inner lists. Only those elements where \code{.p} evaluates to
\code{TRUE} will be modified.}

\item{.at}{A character vector of names or a numeric vector of
positions. Only those elements corresponding to \code{.at} will be
modified.}

\item{.id}{If not \code{NULL} a variable with this name will be created
giving either the name or the index of the data frame.}
}
\value{
All functions return a vector the same length as \code{.x}.

\code{map()} returns a list, \code{map_lgl()} a logical vector, \code{map_int()} an
integer vector, \code{map_dbl()} a double vector, and \code{map_chr()} a character
vector. The output of \code{.f} will be automatically typed upwards,
e.g. logical -> integer -> double -> character.

\code{walk()} returns the input \code{.x} (invisibly). This makes it easy to
use in pipe.
}
\description{
The map functions transform their input by applying a function to
each element and returning a vector the same length as the input.
\itemize{
\item \code{map()}, \code{map_if()} and \code{map_at()} always return a list. See the
\code{\link[=modify]{modify()}} family for versions that return an object of the same
type as the input.

The \code{_if} and \code{_at} variants take a predicate function \code{.p} that
determines which elements of \code{.x} are transformed with \code{.f}.
transform.
\item \code{map_lgl()}, \code{map_int()}, \code{map_dbl()} and \code{map_chr()} return
vectors of the corresponding type (or die trying).
\item \code{map_dfr()} and \code{map_dfc()} return data frames created by
row-binding and column-binding respectively. They require dplyr
to be installed.
\item \code{walk()} calls \code{.f} for its side-effect and returns the input \code{.x}.
}
}
\examples{
1:10 \%>\%
  map(rnorm, n = 10) \%>\%
  map_dbl(mean)

# Or use an anonymous function
1:10 \%>\%
  map(function(x) rnorm(10, x))

# Or a formula
1:10 \%>\%
  map(~ rnorm(10, .x))

# Extract by name or position
# .default specifies value for elements that are missing or NULL
l1 <- list(list(a = 1L), list(a = NULL, b = 2L), list(b = 3L))
l1 \%>\% map("a", .default = "???")
l1 \%>\% map_int("b", .default = NA)
l1 \%>\% map_int(2, .default = NA)

# Supply multiple values to index deeply into a list
l2 <- list(
  list(num = 1:3,     letters[1:3]),
  list(num = 101:103, letters[4:6]),
  list()
)
l2 \%>\% map(c(2, 2))

# Use a list to build an extractor that mixes numeric indices and names,
# and .default to provide a default value if the element does not exist
l2 \%>\% map(list("num", 3))
l2 \%>\% map_int(list("num", 3), .default = NA)

# A more realistic example: split a data frame into pieces, fit a
# model to each piece, summarise and extract R^2
mtcars \%>\%
  split(.$cyl) \%>\%
  map(~ lm(mpg ~ wt, data = .x)) \%>\%
  map(summary) \%>\%
  map_dbl("r.squared")

# Use map_lgl(), map_dbl(), etc to reduce to a vector.
# * list
mtcars \%>\% map(sum)
# * vector
mtcars \%>\% map_dbl(sum)

# If each element of the output is a data frame, use
# map_df to row-bind them together:
mtcars \%>\%
  split(.$cyl) \%>\%
  map(~ lm(mpg ~ wt, data = .x)) \%>\%
  map_df(~ as.data.frame(t(as.matrix(coef(.)))))
# (if you also want to preserve the variable names see
# the broom package)
}
\seealso{
Other map variants: \code{\link{imap}},
  \code{\link{invoke}}, \code{\link{lmap}},
  \code{\link{map2}}, \code{\link{modify}}
}
