% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/methods.r
\docType{methods}
\name{mlg.filter}
\alias{mlg.filter}
\alias{mlg.filter,genclone-method}
\alias{mlg.filter,snpclone-method}
\alias{mlg.filter,genind-method}
\alias{mlg.filter,genlight-method}
\alias{mlg.filter<-}
\alias{mlg.filter<-,genclone-method}
\alias{mlg.filter<-,genind-method}
\alias{mlg.filter<-,snpclone-method}
\alias{mlg.filter<-,genlight-method}
\title{MLG definitions based on genetic distance}
\usage{
mlg.filter(pop, threshold = 0, missing = "asis", memory = FALSE,
  algorithm = "farthest_neighbor", distance = "diss.dist",
  threads = 1L, stats = "MLGs", ...)

mlg.filter(pop, missing = "asis", memory = FALSE,
  algorithm = "farthest_neighbor", distance = "diss.dist",
  threads = 1L, ...) <- value
}
\arguments{
\item{pop}{a \code{\linkS4class{genclone}}, \code{\linkS4class{snpclone}}, or
\code{\linkS4class{genind}} object.}

\item{threshold}{a number indicating the minimum distance two MLGs must be
separated by to be considered different. Defaults to 0, which will reflect
the original (naive) MLG definition.}

\item{missing}{any method to be used by \code{\link{missingno}}: "mean", 
"zero", "loci", "genotype", or "asis" (default).}

\item{memory}{whether this function should remember the last distance matrix 
it generated. TRUE will attempt to reuse the last distance matrix if the 
other parameters are the same. (default) FALSE will ignore any stored 
matrices and not store any it generates.}

\item{algorithm}{determines the type of clustering to be done. 
\describe{
  \item{"farthest_neighbor"}{\emph{ (default) }merges clusters based on the 
  maximum distance between points in either cluster. This is the strictest of
  the three.}
  \item{"nearest_neighbor"}{ merges clusters based on the minimum distance
  between points in either cluster. This is the loosest of the three.}
  \item{"average_neighbor"}{ merges clusters based on the average distance
  between every pair of points between clusters.}
}}

\item{distance}{a character or function defining the distance to be applied 
to pop. Defaults to \code{\link{diss.dist}} for genclone objects and
\code{\link{bitwise.dist}} for snpclone objects. A matrix or table
containing distances between individuals (such as the output of 
\code{\link{rogers.dist}}) is also accepted for this parameter.}

\item{threads}{(unused) Previously, this was the maximum number of parallel 
threads to be used within this function. Default is 1 indicating that this
function will run serially. Any other number will result in a warning.}

\item{stats}{a character vector specifying which statistics should be
returned (details below). Choices are "MLG", "THRESHOLDS", "DISTANCES",
"SIZES", or "ALL". If choosing "ALL" or more than one, a named list will be
returned.}

\item{...}{any parameters to be passed off to the distance method.}

\item{value}{the threshold at which genotypes should be collapsed.}
}
\value{
Default, a vector of collapsed multilocus genotypes. Otherwise, any
  combination of the following:
\subsection{MLGs}{
  a numeric vector defining the multilocus genotype cluster of each
  individual in the dataset. Each genotype cluster is separated from every
  other genotype cluster by at least the defined threshold value, as 
  calculated by the selected algorithm.
}
\subsection{THRESHOLDS}{
  A numeric vector representing the thresholds \strong{beyond} which clusters
  of multilocus genotypes were collapsed. 
}
\subsection{DISTANCES}{
  A square matrix representing the distances between each cluster.
}
\subsection{SIZES}{
 The sizes of the multilocus genotype clusters in order. 
}
}
\description{
Multilocus genotypes are initially defined by naive string matching, but this
definition does not take into account missing data or genotyping error,
casting these as unique genotypes. Defining multilocus genotypes by genetic
distance allows you to incorporate genotypes that have missing data o
genotyping error into their parent clusters.
}
\details{
This function will take in any distance matrix or function and
collapse multilocus genotypes below a given threshold. If you use this
function as the assignment method (mlg.filter(myData, distance = myDist) <-
0.5), the distance function or matrix will be remembered by the object. This
means that if you define your own distance matrix or function, you must keep
it in memory to further utilize mlg.filter.
}
\note{
\code{mlg.vector} makes use of \code{mlg.vector} grouping prior to 
  applying the given threshold. Genotype numbers returned by
  \code{mlg.vector} represent the lowest numbered genotype (as returned by
  \code{mlg.vector}) in in each new multilocus genotype. Therefore
  \strong{\code{mlg.filter} and \code{mlg.vector} return the same vector when
  threshold is set to 0 or less}.
}
\examples{

data(partial_clone)
pc <- as.genclone(partial_clone, threads = 1L) # convert to genclone object

# Basic Use ---------------------------------------------------------------


# Show MLGs at threshold 0.05
mlg.filter(pc, threshold = 0.05, distance = "nei.dist", threads = 1L)
pc # 26 mlgs

# Set MLGs at threshold 0.05
mlg.filter(pc, distance = "nei.dist", threads = 1L) <- 0.05
pc # 25 mlgs

\dontrun{

# The distance definition is persistant
mlg.filter(pc) <- 0.1
pc # 24 mlgs

# But you can still change the definition
mlg.filter(pc, distance = "diss.dist", percent = TRUE) <- 0.1
pc

# Choosing a threshold ----------------------------------------------------


# Thresholds for collapsing multilocus genotypes should not be arbitrary. It
# is important to consider what threshold is suitable. One method of choosing
# a threshold is to find a gap in the distance distribution that represents
# clonal groups. You can look at this by analyzing the distribution of all
# possible thresholds with the function "cutoff_predictor".

# For this example, we'll use Bruvo's distance to predict the cutoff for
# P. infestans.

data(Pinf)
Pinf
# Repeat lengths are necessary for Bruvo's distance
(pinfreps <- fix_replen(Pinf, c(2, 2, 6, 2, 2, 2, 2, 2, 3, 3, 2)))

# Now we can collect information of the thresholds. We can set threshold = 1
# because we know that this will capture the maximum possible distance:
(thresholds <- mlg.filter(Pinf, distance = bruvo.dist, stats = "THRESHOLDS",
                          replen = pinfreps, threshold = 1))
# We can use these thresholds to find an appropriate cutoff
(pcut <- cutoff_predictor(thresholds))
mlg.filter(Pinf, distance = bruvo.dist, replen = pinfreps) <- pcut
Pinf

# This can also be visualized with the "filter_stats" function.

# Special case: threshold = 0 ---------------------------------------------


# It's important to remember that a threshold of 0 is equal to the original
# MLG definition. This example will show a data set that contains genotypes
# with missing data that share all alleles with other genotypes except for 
# the missing one.

data(monpop)
monpop # 264 mlg
mlg.filter(monpop) <- 0
nmll(monpop) # 264 mlg

# In order to merge these genotypes with missing data, we should set the 
# threshold to be slightly higher than 0. We will use the smallest fraction 
# the computer can store.

mlg.filter(monpop) <- .Machine$double.eps ^ 0.5
nmll(monpop) # 236 mlg

# Custom distance ---------------------------------------------------------

# Custom genetic distances can be used either in functions from other
# packages or user-defined functions

data(Pinf)
Pinf
mlg.filter(Pinf, distance = function(x) dist(tab(x))) <- 3
Pinf
mlg.filter(Pinf) <- 4
Pinf

# genlight / snpclone objects ---------------------------------------------


set.seed(999)
gc <- as.snpclone(glSim(100, 0, n.snp.struc = 1e3, ploidy = 2))
gc # 100 mlgs
mlg.filter(gc) <- 0.25
gc # 82 mlgs

}
}
\seealso{
\code{\link{filter_stats}}, 
  \code{\link{cutoff_predictor}}, 
  \code{\link{mll}}, 
  \code{\link{genclone}}, 
  \code{\link{snpclone}}, 
  \code{\link{diss.dist}}, 
  \code{\link{bruvo.dist}}
}
