\name{ Kreiss }
\alias{ Kreiss }
\title{ Calculate Kreiss bounds }
\description{
    Calculate the upper or lower Kreiss bound for a specified 
    population projection matrix (PPM).
}
\usage{
    Kreiss(A, bound=NULL, return.r=FALSE, theta=1,
           rlimit=100, step1=1e-3, step2=1e-6)
}
\arguments{
  \item{A}{
    a square, irreducible, non-negative numeric matrix of any dimension
  }
  \item{bound}{
    (optional) specifies whether an upper or lower bound should be calculated.
  }
  \item{return.r}{
    (optional) specifies whether the value of r at which the Kreiss bound is achieved should
    be returned (see details).
  }
  \item{theta}{
    the value to which the Kriess bound is to be assessed relative to (see details).
  }
  \item{rlimit}{
    the maximum value of r that may be reached before the code breaks (see details).    
  }
  \item{step1,step2}{
    determine the iterative process in calculating the Kreiss bound (see details).
  }
}
\details{
    \code{Kreiss} by default returns a standardised Kreiss bound relative to both asymptotic growth/decline
    and initial population density (Townley & Hodgson 2008; Stott et al. 2011).  It uses an iterative 
    process that evaluates a function of the resolvent of \code{A} over a range of values r where 
    r>\code{theta}.  This iterative process finds the maximum/minimum of the function for the upper/lower 
    bounds respectively.  The process is determined using \code{step1} and \code{step2}: in order to 
    increase accuracy but keep computation time low, the function is evaluated forward in steps equal to 
    \code{step1} until the maximum/minimum is passed and then backward in steps of \code{step2} to more 
    accurately find the maximum/minimum itself.  Therefore, \code{step1} should be larger than \code{step2}.
    The balance between both will determine computation time, whilst accuracy is determined almost solely by
    \code{step2}.  The defaults should be sufficient for most matrices.\cr

    \code{theta} defaults to 1, which means the Kriess bound is assessed relative to both asymptotic
    growth and initial population size.  Sometimes, the maximum/minimum of the function occurs at
    r-->\code{theta}, in which case r is equal to \code{theta+step2}.  Setting \code{return.r=TRUE} tells 
    the function to return the value of r where the maximum/minimum occurs alongside the value of
    the Kreiss bound.  r may not exceed \code{rlimit}.\cr

    \code{Kreiss} will not work with reducible matrices, and returns a warning for imprimitive matrices.
}
\value{
    The upper or lower Kreiss bound of \code{A}.\cr
    If \code{return.r=TRUE}, a list with components:
  \item{bound}{
    the upper or lower Kriess bound
  }
  \item{r}{
    the value of r at which the function is minimised/maximised.
  }
}
\references{
    Stott et al. (2011) Ecol. Lett., 14, 959-970.\cr
    Townley & Hodgson (2008) J. Appl. Ecol., 45, 1836-1839.
}
\author{
    Stott, I., Hodgson, D. J., Townley, S.
}
\examples{
    # Create a 3x3 PPM
    A <- matrix(c(0,1,2,0.5,0.1,0,0,0.6,0.6), byrow=TRUE, ncol=3)
    A

    # Calculate the upper Kreiss bound of A
    Kreiss(A, bound="upper")

    # Calculate the lower Kreiss bound of A
    Kreiss(A, bound="lower")

    # Calculate the upper Kreiss bound of A and return 
    # the value of r at which the function is maximised
    Kreiss(A, bound="upper", return.r=TRUE)
}
\keyword{ Kreiss bound }
\keyword{ ecology }
\keyword{ demography }
\keyword{ transient dynamics }
\keyword{ population projection matrix }
\keyword{ PPM }