## pomp object encoding the "SIRS model with seasonal reservoir" of
##   King, A. A., Ionides, E. L., Pascual, M., & Bouma, M. J.
##   Inapparent infections and cholera dynamics.
##   Nature 454:877-880 (2008)

## Alternative formulation by Matteo Fasiolo.

## Data are cholera deaths and decadal census figures from the Dacca district of Bengal province, 1891-1941.
##
## Data courtesy of Menno J. Bouma, London School of Tropical Medicine & Hygiene

require(pomp)
require(reshape2)
require(plyr)
require(magrittr)

mle <- c(
         gamma=20.8,eps=19.1,rho=0,
         delta=0.02, deltaI=0.06, clin=1, alpha=1,
         beta_trend=-0.00498,
         log_beta1=0.747, log_beta2=6.38, log_beta3=-3.44,
         log_beta4=4.23, log_beta5=3.33, log_beta6=4.55,
         log_omega1=log(0.184), log_omega2=log(0.0786), log_omega3=log(0.0584),
         log_omega4=log(0.00917), log_omega5=log(0.000208), log_omega6=log(0.0124),
         sd_beta=3.13, tau=0.23,
         S_0=0.621, I_0=0.378, Rs_0=0, R1_0=0.000843, R2_0=0.000972, R3_0=1.16e-07
         )

read.csv(text="
year,census
1891,2420656
1901,2649522
1911,2960402
1921,3125967
1931,3432577
1941,4222142
") -> census

read.csv(text="
year,month,cholera_deaths
1891,1,2641
1891,2,939
1891,3,905
1891,4,1219
1891,5,368
1891,6,78
1891,7,29
1891,8,12
1891,9,30
1891,10,44
1891,11,270
1891,12,1149
1892,1,633
1892,2,501
1892,3,855
1892,4,1271
1892,5,666
1892,6,101
1892,7,62
1892,8,23
1892,9,20
1892,10,28
1892,11,461
1892,12,892
1893,1,751
1893,2,170
1893,3,253
1893,4,906
1893,5,700
1893,6,98
1893,7,57
1893,8,72
1893,9,471
1893,10,4217
1893,11,5168
1893,12,4747
1894,1,2380
1894,2,852
1894,3,1166
1894,4,2122
1894,5,576
1894,6,60
1894,7,53
1894,8,62
1894,9,241
1894,10,403
1894,11,551
1894,12,739
1895,1,862
1895,2,348
1895,3,490
1895,4,5596
1895,5,1180
1895,6,142
1895,7,41
1895,8,28
1895,9,39
1895,10,748
1895,11,3934
1895,12,3562
1896,1,587
1896,2,311
1896,3,1639
1896,4,1903
1896,5,601
1896,6,110
1896,7,32
1896,8,19
1896,9,82
1896,10,420
1896,11,1014
1896,12,1073
1897,1,416
1897,2,168
1897,3,909
1897,4,1355
1897,5,447
1897,6,59
1897,7,13
1897,8,21
1897,9,43
1897,10,109
1897,11,338
1897,12,470
1898,1,489
1898,2,394
1898,3,483
1898,4,842
1898,5,356
1898,6,29
1898,7,17
1898,8,16
1898,9,57
1898,10,110
1898,11,488
1898,12,1727
1899,1,1253
1899,2,359
1899,3,245
1899,4,549
1899,5,215
1899,6,9
1899,7,7
1899,8,31
1899,9,236
1899,10,279
1899,11,819
1899,12,1728
1900,1,1942
1900,2,1251
1900,3,3521
1900,4,3412
1900,5,290
1900,6,46
1900,7,35
1900,8,14
1900,9,79
1900,10,852
1900,11,2951
1900,12,2656
1901,1,607
1901,2,172
1901,3,325
1901,4,2191
1901,5,584
1901,6,58
1901,7,38
1901,8,8
1901,9,22
1901,10,50
1901,11,380
1901,12,2059
1902,1,938
1902,2,389
1902,3,767
1902,4,1882
1902,5,286
1902,6,94
1902,7,61
1902,8,10
1902,9,106
1902,10,281
1902,11,357
1902,12,1388
1903,1,810
1903,2,306
1903,3,381
1903,4,1308
1903,5,702
1903,6,87
1903,7,9
1903,8,14
1903,9,36
1903,10,46
1903,11,553
1903,12,1302
1904,1,618
1904,2,147
1904,3,414
1904,4,768
1904,5,373
1904,6,39
1904,7,10
1904,8,36
1904,9,151
1904,10,1130
1904,11,3437
1904,12,4041
1905,1,1415
1905,2,207
1905,3,92
1905,4,128
1905,5,147
1905,6,32
1905,7,7
1905,8,59
1905,9,426
1905,10,2644
1905,11,2891
1905,12,4249
1906,1,2291
1906,2,797
1906,3,680
1906,4,1036
1906,5,404
1906,6,41
1906,7,19
1906,8,12
1906,9,10
1906,10,121
1906,11,931
1906,12,2158
1907,1,1886
1907,2,803
1907,3,397
1907,4,613
1907,5,132
1907,6,48
1907,7,17
1907,8,22
1907,9,26
1907,10,34
1907,11,344
1907,12,657
1908,1,117
1908,2,75
1908,3,443
1908,4,972
1908,5,646
1908,6,107
1908,7,18
1908,8,6
1908,9,9
1908,10,5
1908,11,12
1908,12,142
1909,1,133
1909,2,189
1909,3,1715
1909,4,3115
1909,5,1412
1909,6,182
1909,7,50
1909,8,37
1909,9,77
1909,10,475
1909,11,1730
1909,12,1489
1910,1,620
1910,2,190
1910,3,571
1910,4,1558
1910,5,440
1910,6,27
1910,7,7
1910,8,14
1910,9,93
1910,10,1462
1910,11,2467
1910,12,1703
1911,1,1262
1911,2,458
1911,3,453
1911,4,717
1911,5,232
1911,6,26
1911,7,16
1911,8,18
1911,9,9
1911,10,78
1911,11,353
1911,12,897
1912,1,777
1912,2,404
1912,3,799
1912,4,2067
1912,5,613
1912,6,98
1912,7,19
1912,8,26
1912,9,47
1912,10,171
1912,11,767
1912,12,1896
1913,1,887
1913,2,325
1913,3,816
1913,4,1653
1913,5,355
1913,6,85
1913,7,54
1913,8,88
1913,9,609
1913,10,882
1913,11,1363
1913,12,2178
1914,1,580
1914,2,396
1914,3,1493
1914,4,2154
1914,5,683
1914,6,78
1914,7,19
1914,8,10
1914,9,27
1914,10,88
1914,11,1178
1914,12,1862
1915,1,611
1915,2,478
1915,3,2697
1915,4,3395
1915,5,520
1915,6,67
1915,7,41
1915,8,36
1915,9,209
1915,10,559
1915,11,971
1915,12,2144
1916,1,1099
1916,2,494
1916,3,586
1916,4,508
1916,5,269
1916,6,27
1916,7,19
1916,8,21
1916,9,12
1916,10,22
1916,11,333
1916,12,676
1917,1,487
1917,2,262
1917,3,535
1917,4,979
1917,5,170
1917,6,25
1917,7,9
1917,8,19
1917,9,13
1917,10,45
1917,11,229
1917,12,673
1918,1,432
1918,2,107
1918,3,373
1918,4,1126
1918,5,339
1918,6,19
1918,7,11
1918,8,3
1918,9,15
1918,10,101
1918,11,539
1918,12,709
1919,1,200
1919,2,208
1919,3,926
1919,4,1783
1919,5,831
1919,6,103
1919,7,37
1919,8,17
1919,9,33
1919,10,179
1919,11,426
1919,12,795
1920,1,481
1920,2,491
1920,3,773
1920,4,936
1920,5,325
1920,6,101
1920,7,22
1920,8,25
1920,9,24
1920,10,88
1920,11,633
1920,12,513
1921,1,298
1921,2,93
1921,3,687
1921,4,1750
1921,5,356
1921,6,33
1921,7,2
1921,8,18
1921,9,70
1921,10,648
1921,11,2471
1921,12,1270
1922,1,616
1922,2,193
1922,3,706
1922,4,1372
1922,5,668
1922,6,107
1922,7,58
1922,8,21
1922,9,23
1922,10,93
1922,11,318
1922,12,867
1923,1,332
1923,2,118
1923,3,437
1923,4,2233
1923,5,491
1923,6,27
1923,7,7
1923,8,21
1923,9,96
1923,10,360
1923,11,783
1923,12,1492
1924,1,550
1924,2,176
1924,3,633
1924,4,922
1924,5,267
1924,6,91
1924,7,42
1924,8,4
1924,9,10
1924,10,7
1924,11,43
1924,12,377
1925,1,563
1925,2,284
1925,3,298
1925,4,625
1925,5,131
1925,6,35
1925,7,12
1925,8,8
1925,9,9
1925,10,83
1925,11,502
1925,12,551
1926,1,256
1926,2,198
1926,3,664
1926,4,1701
1926,5,425
1926,6,76
1926,7,17
1926,8,9
1926,9,16
1926,10,5
1926,11,141
1926,12,806
1927,1,1603
1927,2,587
1927,3,530
1927,4,771
1927,5,511
1927,6,97
1927,7,35
1927,8,39
1927,9,156
1927,10,1097
1927,11,1233
1927,12,1418
1928,1,1125
1928,2,420
1928,3,1592
1928,4,4169
1928,5,1535
1928,6,371
1928,7,139
1928,8,55
1928,9,85
1928,10,538
1928,11,1676
1928,12,1435
1929,1,804
1929,2,370
1929,3,477
1929,4,394
1929,5,306
1929,6,132
1929,7,84
1929,8,87
1929,9,53
1929,10,391
1929,11,1541
1929,12,1859
1930,1,894
1930,2,326
1930,3,853
1930,4,1891
1930,5,1009
1930,6,131
1930,7,77
1930,8,63
1930,9,66
1930,10,33
1930,11,178
1930,12,1003
1931,1,1051
1931,2,488
1931,3,911
1931,4,1806
1931,5,837
1931,6,280
1931,7,132
1931,8,76
1931,9,381
1931,10,1328
1931,11,2639
1931,12,2164
1932,1,1082
1932,2,326
1932,3,254
1932,4,258
1932,5,119
1932,6,106
1932,7,93
1932,8,29
1932,9,17
1932,10,17
1932,11,17
1932,12,46
1933,1,79
1933,2,135
1933,3,1290
1933,4,2240
1933,5,561
1933,6,116
1933,7,24
1933,8,15
1933,9,33
1933,10,18
1933,11,16
1933,12,38
1934,1,26
1934,2,45
1934,3,151
1934,4,168
1934,5,57
1934,6,32
1934,7,29
1934,8,27
1934,9,20
1934,10,106
1934,11,1522
1934,12,2013
1935,1,434
1935,2,205
1935,3,528
1935,4,634
1935,5,195
1935,6,45
1935,7,33
1935,8,19
1935,9,20
1935,10,46
1935,11,107
1935,12,725
1936,1,572
1936,2,183
1936,3,2199
1936,4,4018
1936,5,428
1936,6,67
1936,7,31
1936,8,8
1936,9,44
1936,10,484
1936,11,1324
1936,12,2054
1937,1,467
1937,2,216
1937,3,673
1937,4,887
1937,5,353
1937,6,73
1937,7,46
1937,8,15
1937,9,20
1937,10,27
1937,11,25
1937,12,38
1938,1,158
1938,2,312
1938,3,1226
1938,4,1021
1938,5,222
1938,6,90
1938,7,31
1938,8,93
1938,9,368
1938,10,657
1938,11,2208
1938,12,2178
1939,1,702
1939,2,157
1939,3,317
1939,4,146
1939,5,63
1939,6,27
1939,7,22
1939,8,23
1939,9,28
1939,10,225
1939,11,483
1939,12,319
1940,1,120
1940,2,59
1940,3,274
1940,4,282
1940,5,155
1940,6,31
1940,7,16
1940,8,15
1940,9,12
1940,10,14
1940,11,14
1940,12,42
") %>%
  mutate(time=year+month/12) %>%
  subset(select=-c(year,month)) -> cholera

covar.dt <- 0.01
nbasis <- 6L
nrstage <- 3L
cholera.deathrate <- with(cholera,mean(cholera_deaths))/with(census,mean(census))*12
cholera.deathrate <- 0

t0 <- with(cholera,2*time[1]-time[2])
tcovar <- seq(from=t0,to=max(cholera$time)+2/12,by=covar.dt)
ssfit <- smooth.spline(x=census$year,y=census$census)
covartable <- data.frame(
                         time=tcovar,
                         seas=periodic.bspline.basis(tcovar-1/12,nbasis=nbasis,degree=3,period=1),
                         pop=predict(ssfit,x=tcovar,deriv=0)$y,
                         dpopdt=predict(ssfit,x=tcovar,deriv=1)$y,
                         trend=tcovar-mean(tcovar)
                         )

states <- c("S","I","Rs",sprintf("R%d",1:nrstage),"D","W")
params <- c("tau","gamma","eps","delta","deltaI","log_omega1","sd_beta",
            "beta_trend","log_beta1","alpha","rho","clin",
            "S_0","I_0","Rs_0","R1_0")

globals <- Csnippet(paste0("
  int nrstage = ",nrstage,";
  int nbasis = ",nbasis,";
  double cholera_deathrate = ",cholera.deathrate,";

  inline double out_flow (double init, double rate_by_dt) {
    return init*(1.0-exp(-rate_by_dt));
  }
"))

toEst <- Csnippet("
  Ttau = log(tau);
  Tgamma = log(gamma);
  Teps = log(eps);
  TdeltaI = log(deltaI);
  Tsd_beta = log(sd_beta);
  Talpha = log(alpha);
  Trho = log(rho);
  Tclin = logit(clin);
  to_log_barycentric(&TS_0,&S_0,3+nrstage);
")

fromEst <- Csnippet("
  Ttau = exp(tau);
  Tgamma = exp(gamma);
  Teps = exp(eps);
  TdeltaI = exp(deltaI);
  Tsd_beta = exp(sd_beta);
  Talpha = exp(alpha);
  Trho = exp(rho);
  Tclin = expit(clin);
  from_log_barycentric(&TS_0,&S_0,3+nrstage);
")

nbinomRmeasure <- Csnippet("
  double tol = 1.0e-18;
  if (!(R_FINITE(D))) {
    cholera_deaths = R_NaReal;
  } else {
    cholera_deaths = rnbinom_mu(1/(tau*tau), D+tol);
  }
")

nbinomDmeasure <- Csnippet("
  double tol = 1.0e-18;
  if (!(R_FINITE(D))) {
    lik = tol;
  } else {
    lik = dnbinom_mu(cholera_deaths+tol, 1/(tau*tau), D+tol, 0) + tol;
  }
  if (give_log) lik = log(lik);
")

normRmeasure <- Csnippet("
  double v, tol = 1.0e-18;
  v = D*tau;
  if (!(R_FINITE(v))) {
    cholera_deaths = R_NaReal;
  } else {
    cholera_deaths = rnorm(D,v+tol);
  }
")

normDmeasure <- Csnippet("
  double v, tol = 1.0e-18;
  v = D*tau;
  if (!(R_FINITE(v))) {
    lik = tol;
  } else {
    lik = dnorm(cholera_deaths,D,v+tol,0)+tol;
  }
  if (give_log) lik = log(lik);
")


choleraSim <- Csnippet("
  double *R = &R1;
  double R_o[nrstage];
  double k_eps = nrstage*eps;
  double gamma_delta_deltaI = gamma+delta+deltaI;
  double keps_div_keps_delta = k_eps/(k_eps+delta);

  double beta = exp(dot_product(nbasis,&seas_1,&log_beta1)+beta_trend*trend);
  double omega = exp(dot_product(nbasis,&seas_1,&log_omega1));
  double dw = rgammawn(sd_beta,dt);
  double lambda = omega+(beta*dw/dt)*I/pop;

  double S_o = out_flow(S,(lambda+delta)*dt);
  double I_o = out_flow(I,gamma_delta_deltaI*dt);
  double RS_o = out_flow(Rs,(rho+delta)*dt);
  double births = dpopdt + delta*pop + cholera_deathrate*pop;
  double new_infect = S_o*(lambda/(lambda+delta));

  int j;

  for (j = 0; j < nrstage; j++) R_o[j] = out_flow(R[j],(k_eps+delta)*dt);

  S += births*dt + R_o[nrstage-1]*keps_div_keps_delta;
  I += clin*new_infect - I_o;
  Rs += (1-clin)*new_infect - RS_o;
  R[0] += I_o*(gamma/gamma_delta_deltaI) - R_o[0];
  for (j = 1; j < nrstage; j++) {
    R[j] += R_o[j-1]*keps_div_keps_delta - R_o[j];
  }
  D += I_o*(deltaI/gamma_delta_deltaI);
  W += (dw-dt)/sd_beta;
")

rInit <- Csnippet("
  double *R = &R1;
  const double *R0 = &R1_0;
  double m = S_0+I_0+Rs_0;
  int j;
  for (j = 0; j < nrstage; j++) m += R0[j];
  m = pop/m;
  S = m*S_0;
  I = m*I_0;
  Rs = m*Rs_0;
  for (j = 0; j < nrstage; j++) R[j] = m*R0[j];
  D = 0;
  W = 0;
")

pomp(
     data=cholera,
     times='time',
     t0=t0,
     params=mle,
     globals=globals,
     rprocess = euler.sim(step.fun = choleraSim,delta.t=1/240),
     dmeasure = normDmeasure,
     rmeasure=normRmeasure,
     covar=covartable,
     tcovar='time',
     zeronames = c("D","W"),
     statenames = states,
     paramnames = params,
     fromEstimationScale=fromEst,
     toEstimationScale=toEst,
     initializer = rInit
     ) -> dacca2

c("dacca2")
