% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/major_mutate_variations.R
\name{mutate_cascade}
\alias{mutate_cascade}
\title{Perform mutate one time period at a time ('Cascading mutate')}
\usage{
mutate_cascade(
  .df,
  ...,
  .skip = TRUE,
  .backwards = FALSE,
  .group_i = TRUE,
  .i = NULL,
  .t = NULL,
  .d = NA,
  .uniqcheck = FALSE,
  .setpanel = TRUE
)
}
\arguments{
\item{.df}{Data frame or tibble.}

\item{...}{Specification to be passed to \code{mutate()}.}

\item{.skip}{Set to \code{TRUE} to skip the first period present in the data (or present within each group for grouped data) when applying \code{mutate()}. Since most uses of \code{mutate_cascade()} will involve a \code{lag()} or \code{tlag()}, this avoids creating an \code{NA} in the first period that then cascades down. By default this is TRUE. If you set this to FALSE you should probably have some method for avoiding a first-period \code{NA} in your \code{...} entry, perhaps using the \code{default} option in \code{dplyr::lag} or the \code{.default} option in \code{tlag}.}

\item{.backwards}{Set to \code{TRUE} to run \code{mutate_cascade()} from the last period to the first, rather than from the first to the last.}

\item{.group_i}{By default, if \code{.i} is specified or found in the data, \code{mutate_cascade} will group the data by \code{.i}, ignoring any grouping already implemented (although the original grouping structure will be returned at the end). Set \code{.group_i = FALSE} to avoid this.}

\item{.i}{Quoted or unquoted variables that identify the individual cases. Note that setting any one of \code{.i}, \code{.t}, or \code{.d} will override all three already applied to the data, and will return data that is \code{as_pibble()}d with all three, unless \code{.setpanel=FALSE}.}

\item{.t}{Quoted or unquoted variables indicating the time. \code{pmdplyr} accepts two kinds of time variables: numeric variables where a fixed distance \code{.d} will take you from one observation to the next, or, if \code{.d=0}, any standard variable type with an order. Consider using the \code{time_variable()} function to create the necessary variable if your data uses a \code{Date} variable for time.}

\item{.d}{Number indicating the gap in \code{.t} between one period and the next. For example, if \code{.t} indicates a single day but data is collected once a week, you might set \code{.d=7}. To ignore gap length and assume that "one period ago" is always the most recent prior observation in the data, set \code{.d=0}. The default \code{.d = NA} here will become \code{.d = 1} if either \code{.i} or \code{.t} are declared.}

\item{.uniqcheck}{Logical parameter. Set to TRUE to always check whether \code{.i} and \code{.t} uniquely identify observations in the data. By default this is set to FALSE and the check is only performed once per session, and only if at least one of \code{.i}, \code{.t}, or \code{.d} is set.}

\item{.setpanel}{Logical parameter. \code{TRUE} by default, and so if \code{.i}, \code{.t}, and/or \code{.d} are declared, will return a \code{pibble} set in that way.}
}
\description{
This function is a wrapper for \code{dplyr::mutate()} which performs \code{mutate} one time period at a time, allowing each period's calculation to complete before moving on to the next. This allows changes in one period to 'cascade down' to later periods. This is (number of time periods) slower than regular \code{mutate()} and, generally, is only used for mutations where an existing variable is being defined in terms of its own \code{lag()} or \code{tlag()}. This is similar in concept to (and also slower than) \code{cumsum} but is much more flexible, and works with data that has multiple observations per individual-period using \code{tlag()}. For example, this could be used to calculate the current value of a savings account given a variable with each period's deposits, withdrawals, and interest, or could calculate the cumulative number of credits a student has taken across all classes.
}
\details{
To apply \code{mutate_cascade()} to non-panel data and without any grouping (perhaps to mimic standard Stata \code{replace} functionality), add a variable to your data indicating the order you'd like \code{mutate} performed in (perhaps using \code{dplyr::row_number()}) and \code{.t} to that new variable.
}
\examples{

if(interactive()){
data(Scorecard)
# I'd like to build a decaying function that remembers previous earnings but at a declining rate
# Let's only use nonmissing earnings
# And let's say we're only interested in four-year colleges in Colorado
# (mutate_cascade + tlag can be very slow so we're working with a smaller sample)
Scorecard <- Scorecard \%>\%
  dplyr::filter(
    !is.na(earnings_med),
    pred_degree_awarded_ipeds == 3,
    state_abbr == "CO"
  ) \%>\%
  # And declare the panel structure
  as_pibble(.i = unitid, .t = year)
Scorecard <- Scorecard \%>\%
  # Almost all instances involve a variable being set to a function of a lag of itself
  # we don't want to overwrite so let's make another
  # Note that earnings_med is an integer -
  # but we're about to make non-integer decay function, so call it a double!
  dplyr::mutate(decay_earnings = as.double(earnings_med)) \%>\%
  # Now we can cascade
  mutate_cascade(
    decay_earnings = decay_earnings +
      .5 * tlag(decay_earnings, .quick = TRUE)
  )
  }
}
