% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plte_builder.R
\name{plte_builder}
\alias{plte_builder}
\title{Create simulations for long-term effects in models with temporal dependence}
\source{
Williams, Laron K. 2016. "Long-Term Effects in Models with Temporal
Dependence". Political Analysis: 24(2): 243-262.
}
\usage{
plte_builder(obj, obj_tvar, cf, cf_duration = "permanent", t_points,
  FUN = logistic_prob_FUN, ci = 0.95, nsim = 1000)
}
\arguments{
\item{obj}{a fitted model object.}

\item{obj_tvar}{character string specifying the name of the base time variable
in \code{obj}.}

\item{cf}{a data frame with the first row containing the counterfactual.
An optional second row could be supplied with values for the baseline
scenario. If not supplied then all values are set to zero for the baseline.
Columns should have names that match variables in \code{obj} and contain
fitted values to find quantities of interest for. Note, this should not
include your time variable as this is handled using \code{t_points}.}

\item{cf_duration}{a character string or numeric specifying the
counterfactual's duration. If \code{'permanent'} then the
counterfacutal lasts for the full time span in \code{t_points}. If
\code{'one-time'} then the counterfactual only lasts for one period.
If \code{cf_duration} is numeric then the number specifies the
number of time increments at which the counterfactual resets.}

\item{t_points}{a numeric vector with a minimum length of 2 and a maximum
lentgh of 3. The first and last values should be the time starting and ending
points for the simulatinos. The (optional) middle value can specify a point
between the first and last time points where a subsequent event occurs.}

\item{FUN}{a function for finding a quantity of interest from the linear
systematic component. See \code{\link{qi_builder}}. The default is a function
for finding the predicted probability from a logistic regression model.}

\item{ci}{the proportion of the central interval of the simulations to
return. Must be in (0, 1] or equivalently (0, 100].}

\item{nsim}{number of simulations to draw.}
}
\value{
A data frame with the medians and central intervals of the
simulated scenarios. Note that the column \code{scenario_name}
encodes scenarios where y = 0 as \code{baseline} and y = 1 as
\code{counterfactual}.
}
\description{
Create simulations for long-term effects in models with temporal dependence
}
\examples{
data('negative')

# BTSCS set the data
neg_set <- btscs(df = negative, event = 'y', t_var = 'tim',
                cs_unit = 'group', pad_ts = FALSE)

# Create temporal dependence variable
neg_set$t <- neg_set$spell + 1

m1 <- glm(y ~ x + t + I(t^2) + I(t^3),
          family = binomial(link = 'logit'), data = neg_set)

# Create fitted counterfactual
counterfactual <- data.frame(x = 0.5)

# Permanent counterfactual, one event
sim1 <- plte_builder(obj = m1, obj_tvar = 't',
                     cf = counterfactual, t_points = c(13, 25))

# Multiple events
sim2 <- plte_builder(obj = m1, obj_tvar = 't',
                     cf = counterfactual, t_points = c(13, 18, 25))

# One-time counterfactual
sim3 <- plte_builder(obj = m1, obj_tvar = 't',
                     cf = counterfactual, t_points = c(13, 25),
                     cf_duration = 'one-time')

# Temporary (4 period counterfactual)
sim4 <- plte_builder(obj = m1, obj_tvar = 't',
                     cf = counterfactual, t_points = c(13, 25),
                     cf_duration = 4)

# Custom baseline scenario
# Note: the second row is the custom baseline
counterfactual_baseline <- data.frame(x = c(1, 0.5))

sim5 <- plte_builder(obj = m1, obj_tvar = 't', cf_duration = 4,
                     cf = counterfactual_baseline, t_points = c(13, 25))

# Time splines
library(splines)
m2 <- glm(y ~ x + bs(t, degree = 3), family = binomial(link = 'logit'),
          data = neg_set)

sim6 <- plte_builder(obj = m2, obj_tvar = 't', cf_duration = 4,
                     cf = counterfactual, t_points = c(13, 25))

}
