% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/stability.selection.R
\name{stability.selection}
\alias{stability.selection}
\title{Stability selection procedure to select covariates for the sparse PLS, 
LOGIT-SPLS and multinomial-SPLS methods}
\usage{
stability.selection(stab.out, piThreshold = 0.9, rhoError = 10)
}
\arguments{
\item{stab.out}{the output of the functions \code{\link{spls.stab}}, 
\code{\link{logit.spls.stab}} or \code{\link{multinom.spls.stab}}.}

\item{piThreshold}{a value in (0,1], corresponding to the threshold 
probability used to select covariate (c.f. Durif et al., 2017).}

\item{rhoError}{a positive value used to restrict the grid of 
hyper-parameter candidate values (c.f. Durif et al., 2017).}
}
\value{
An object with the following attributes:
\item{selected.predictors}{The list of the name of covariates that 
are selected.}
\item{max.probs}{The corresponding estimated probabilities of selection for 
each covariate, i.e. the maximal values on the reduced grid of 
hyper-parameters.}
}
\description{
The function \code{stability.selection} returns the list of selected 
covariates, when following the stability selection procedure described in 
Durif et al. (2017). In particular, it selects covariates that are selected 
by most of the sparse PLS, the logit-SPLS or the multinomial-SPLS models 
when exploring the grid of hyper-parameter candidate values.
}
\details{
The procedure is described in Durif et al. (2017). The stability selection 
procedure can be summarize as follow (c.f. Meinshausen and Buhlmann, 2010).

(i) For each candidate values of hyper-parameters, a model is trained 
on \code{nresamp} resamplings of the data. Then, for each candidate value of
the hyper-parameters, the probability that a covariate 
(i.e. a column in \code{X}) is selected is computed among the resamplings.

The estimated probabilities can be visualized as a heatmap with the 
function \code{\link{stability.selection.heatmap}}.

(ii) Eventually, the set of "stable selected" variables corresponds to the 
set of covariates that were selected by most of the training among the 
grid of hyper-parameters candidate values, based on a threshold probability
\code{piThreshold} and a restriction of the grid of hyper-parameters based 
on \code{rhoError} (c.f. Durif et al., 2017, for details).

This function achieves the second step (ii) of the stability selection 
procedure. The first step (i) is achieved by the functions 
\code{\link{spls.stab}}, \code{\link{logit.spls.stab}} 
or \code{\link{multinom.spls.stab}}.
}
\examples{
\dontrun{
### load plsgenomics library
library(plsgenomics)

### generating data
n <- 100
p <- 100
sample1 <- sample.cont(n=n, p=p, kstar=10, lstar=2, 
                       beta.min=0.25, beta.max=0.75, mean.H=0.2, 
                       sigma.H=10, sigma.F=5, sigma.E=5)
                       
X <- sample1$X
Y <- sample1$Y

### hyper-parameters values to test
lambda.l1.range <- seq(0.05,0.95,by=0.1) # between 0 and 1
ncomp.range <- 1:10

### tuning the hyper-parameters
stab1 <- spls.stab(X=X, Y=Y, lambda.l1.range=lambda.l1.range, 
                   ncomp.range=ncomp.range, weight.mat=NULL, 
                   adapt=FALSE, center.X=TRUE, center.Y=TRUE, 
                   scale.X=TRUE, scale.Y=TRUE, weighted.center=FALSE, 
                   ncores=1, nresamp=100)
                       
str(stab1)

### selected covariates
stability.selection(stab1, piThreshold=0.6, rhoError=10)
}

}
\seealso{
\code{\link{spls.stab}}, \code{\link{logit.spls.stab}}, 
\code{\link{multinom.spls.stab}}, \code{\link{stability.selection.heatmap}}
}
\author{
Ghislain Durif (\url{http://thoth.inrialpes.fr/people/gdurif/}).
}
