

################################################################################


#' Enclose text in Rd commands
#'
#' Enclose text in Rd commands after escaping it appropriately.
#'
#' @inheritParams pack_desc
#' @param x Character vector of Rd command names (outermost first). It is an
#'   error if an entry in \code{x} contains special characters from Rd syntax,
#'   but empty character strings and \code{NA} values are just ignored.
#' @param text Character vector or convertible to such; text to be enclosed.
#'   Needs not be appropriately escaped beforehand. Must not contain \code{NA}.
#~ @export
#' @return Character vector of the same length than \code{text}.
#~ @family rd-functions
#~ @keywords character
#' @keywords internal
#' @seealso tools::toRd
#~ @examples
#~ (x <- rd_quote(c("code", "link"), c("fun.1", "fun.2")))
#~ stopifnot(is.character(x), length(x) == 2)
#~ stopifnot(sapply(c("fun.", "code", "link"), grepl, x = x, fixed = TRUE))
#'
rd_quote <- function(x, ...) UseMethod("rd_quote")

#' @rdname rd_quote
#' @method rd_quote character
#' @export
#'
rd_quote.character <- function(x, text, ...) {
  text <- tools::toRd(na.fail(text))
  x <- x[nzchar(na.fail(x))]
  if (length(bad <- x[x != tools::toRd(x)]))
    stop(sprintf("command name '%s' contains special characters", bad[1L]))
  for (item in rev(x))
    text <- sprintf("\\%s{%s}", item, text)
  text
}


################################################################################


#' Repair Rd files
#'
#' Repair some sometimes encountered, minor problems in Rd files generated by
#' \pkg{roxygen2} (but not those related to S4 methods), and run the checks
#' implemented in \code{\link{check_keywords}} and
#' \code{\link{check_examples}}.
#'
#' @inheritParams pkg_files
#' @param x Object of class \sQuote{Rd}, or name(s) of Rd file(s) or of one to
#'   several package directories. See \code{\link{pkg_files}} for the
#'   possible expansion of names.
#' @param remove.dups Logical scalar indicating whether (probably) duplicate
#'   links in the \sQuote{seealso} section shall be removed.
#' @export
#' @return The \sQuote{Rd} method returns an object of class \sQuote{Rd}, the
#'   character method a list of such objects, invisibly.
#' @family rd-functions
#' @keywords package
#' @examples
#' data(xyplot.docu)
#' summary(x <- repair_docu(xyplot.docu))
#' stopifnot(identical(x, xyplot.docu)) # nothing to repair here
#'
repair_docu <- function(x, ...) UseMethod("repair_docu")

#' @rdname repair_docu
#' @method repair_docu character
#' @export
#'
repair_docu.character <- function(x, ignore = NULL, ...) {
  do_repair <- function(x) {
    data <- repair_docu(tools::parse_Rd(file = x), ...)
    check_keywords(data, file = x, list.only = FALSE)
    check_examples(data, file = x)
    puts(data, file = x)
  }
  x <- pkg_files(x, what = "man", installed = FALSE, ignore = ignore)
  invisible(sapply(x, do_repair, simplify = FALSE))
}

#' @rdname repair_docu
#' @method repair_docu Rd
#' @export
#'
repair_docu.Rd <- function(x, remove.dups = FALSE, ...) {
  cum_parts <- function(x) {
    x <- strsplit(x, ".", fixed = TRUE)
    x <- x[vapply(x, length, integer(1L)) > 0L]
    unlist(lapply(x, function(y) {
      vapply(seq_along(y), function(i) {
        paste(y[seq.int(1L, i)], collapse = ".")
      }, character(1L))
    }))
  }
  function.names <- new.env(parent = emptyenv())
  removed <- FALSE
  do_repair <- function(x, parent.tags) {
    case(attr(x, "Rd_tag"),
      TEXT = {
        switch(parent.tags[1L],
          `\\keyword` = x[x == "dataset"] <- "datasets",
          `\\link` = if (remove.dups && "\\seealso" %in% parent.tags[-1L]) {
            for (part in cum_parts(x))
              if (exists(part, function.names)) {
                x <- NULL
                removed <<- TRUE
                break
              }
            if (!is.null(x)) {
              removed <<- FALSE
              function.names[[x]] <- NULL
            }
          },
          `\\seealso` = if (removed) {
            x <- NULL
            removed <<- FALSE
          }
        )
        x
      },
      COMMENT =,
      VERB = x,
      RCODE = {
        switch(parent.tags[1L],
          `\\usage` = if (grepl("\\s*<-\\s*value\\s*$", x, perl = TRUE))
            # Repair duplicate 'value' entries of replacement functions
            x <- sub(",\\s*value", "", x, perl = TRUE)
        )
        x
      }
    )
  }
  repair_recursively <- function(x, parent.tags) {
    if (!is.list(x))
      return(do_repair(x, parent.tags))
    if (!length(x))
      return(x) # keep lists as-is if they were already empty
    if (is.null(this.tag <- attr(x, "Rd_tag")))
      this.tag <- ".empty"
    y <- lapply(x, repair_recursively, parent.tags = c(this.tag, parent.tags))
    y[vapply(y, is.null, logical(1L))] <- NULL
    if (!length(y))
      return(NULL)
    attributes(y) <- attributes(x)
    y
  }
  LL(remove.dups)
  repair_recursively(x, ".toplevel")
}


################################################################################


#' Subset an Rd or pack_desc object
#'
#' Create a subset of an \sQuote{Rd} object based on the attributes of its
#' elements. The \sQuote{pack_desc} method reduces the input object to a list
#' containing certain relevant components.
#'
#' @inheritParams pack_desc
#' @param x Object of class \sQuote{Rd}, \sQuote{pack_desc} or
#'   \sQuote{pack_descs}.
#' @param subset Character vector with names of attributes of elements to
#'   keep.
#' @param values Logical scalar indicating whether list elements or a logical
#'   vector indicating the positions within \code{x} should be returned.
#' @return Logical vector or object of class \sQuote{Rd}. For the
#'   \sQuote{pack_descs} and \sQuote{pack_desc} method, a (nested) list.
#' @name subset
#~ @export
#~ @family rd-functions
#~ @keywords package
#' @keywords internal
#~ @examples
#~ data(xyplot.docu)
#~ (x <- subset(xyplot.docu, c("examples", "keyword")))
#~ stopifnot(is.logical(x), length(x) == length(xyplot.docu),
#~   length(which(x)) == 2)
#~ summary(x <- subset(xyplot.docu, c("examples", "keyword"), values = TRUE))
#~ stopifnot(inherits(x, "Rd"), is.list(x), length(x) == 2)
#'
NULL

#' @rdname subset
#' @method subset Rd
#' @export
#'
subset.Rd <- function(x, subset, values = FALSE, ...) {
  prepend <- !grepl("^\\\\", subset, perl = TRUE)
  subset[prepend] <- sprintf("\\%s", subset[prepend])
  y <- vapply(x, attr, character(1L), which = "Rd_tag") %in% subset
  if (L(values)) {
    x[!y] <- NULL
    x
  } else
    y
}

#' @rdname subset
#' @method subset pack_desc
#' @export
#'
subset.pack_desc <- function(x, ...) {
  x <- strsplit(x[1L, ], "\n", fixed = TRUE)
  x <- lapply(x, function(y) y[nzchar(y)])
  for (name in c("Depends", "Imports", "Suggests"))
    if (!is.null(entry <- x[[name]])) {
      entry <- unlist(strsplit(entry, "\\s*,\\s*", perl = TRUE))
      entry <- sub("\\s*\\([^)]*\\)$", "", entry, perl = TRUE)
      x[[name]] <- entry[entry != "R"]
    }
  for (name in "Collate")
    if (!is.null(entry <- x[[name]])) {
      entry <- unlist(strsplit(entry, "\\s+", perl = TRUE))
      x[[name]] <- gsub('"', "", gsub("'", "", entry, fixed = TRUE),
        fixed = TRUE)
    }
  x
}

#' @rdname subset
#' @method subset pack_descs
#' @export
#'
subset.pack_descs <- function(x, ...) {
  lapply(X = x, FUN = subset, ...)
}


################################################################################


#' Check keywords in Rd files
#'
#' Check keywords in Rd files for their presence, for their occurrence within
#' the list given in the \sQuote{KEYWORDS.db} file that comes with \R{}, and
#' for fitting to the \sQuote{docType} of the \sQuote{Rd} object.
#'
#' @inheritParams pack_desc
#' @param x Object of class \sQuote{Rd}, vector, or \code{NULL}. \sQuote{Rd}
#'   objects are checked as described below. If \code{NULL}, the total list of
#'   known keywords is returned as character vector, otherwise only those
#'   contained in \code{x}. If a logical vector, \code{x} is used like the
#'   \sQuote{full} argument. If a character vector, used to select the
#'   descriptions of certain keywords.
#' @param file Optional character scalar indicating the filename from which the
#'   \sQuote{Rd} object was read. Useful when creating messages describing a
#'   problem. See \code{\link{logfile}}.
#' @param full Logical scalar indicating whether the full content of the
#'   \sQuote{KEYWORDS.db} file should be returned, not just the list of
#'   keywords.
#' @param list.only Logical scalar indicating whether the keywords should just
#'   be collected, not checked against the database.
#' @details The \sQuote{Rd} method examines the content of Rd files.
#'   In addition to checking that the keywords used are listed in the
#'   \sQuote{KEYWORDS.db} file, it is checked that \sQuote{internal} is not
#'   combined with other keywords, and that \sQuote{docType} entries such as
#'   \sQuote{class}, \sQuote{data} or \sQuote{package} are accompanied by the
#'   appropriate keywords.
#' @export
#' @return Character vector. As a side
#'   effect, problem messages are printed to \code{stderr}. See
#'   \code{\link{logfile}} for how to send these messages to a file.
#' @family rd-functions
#' @keywords package
#' @examples
#'
#' # Character data
#' (x <- check_keywords())
#' stopifnot(length(x) > 1, is.character(x), is.null(names(x)))
#' (x <- check_keywords(c("aplot", "loess")))
#' stopifnot(length(x) == 2, is.character(x), names(x) == c("aplot", "loess"))
#'
#' # 'Rd' objects
#' data(xyplot.docu)
#' (x <- check_keywords(xyplot.docu))
#' stopifnot(identical(x, "hplot"))
#'
check_keywords <- function(x, ...) UseMethod("check_keywords")

#' @rdname check_keywords
#' @method check_keywords NULL
#' @export
#'
check_keywords.NULL <- function(x, full = FALSE, ...) {
  x <- readLines(file.path(R.home(), "doc", "KEYWORDS.db"))
  x <- grep("|", x, fixed = TRUE, value = TRUE)
  x <- sub("^.*\\|", "", x, perl = TRUE)
  x <- do.call(rbind, strsplit(x, "\\s*:\\s*", perl = TRUE))
  if (L(full))
    return(structure(.Data = x[, 2L], .Names = x[, 1L]))
  sort.int(unique.default(x[, 1L]))
}

#' @rdname check_keywords
#' @method check_keywords character
#' @export
#'
check_keywords.character <- function(x, ...) {
  check_keywords(x = NULL, full = TRUE, ...)[x]
}

#' @rdname check_keywords
#' @method check_keywords logical
#' @export
#'
check_keywords.logical <- function(x, ...) {
  check_keywords(x = NULL, full = x, ...)
}

#' @rdname check_keywords
#' @method check_keywords Rd
#' @export
#'
check_keywords.Rd <- function(x, file = NULL, list.only = FALSE, ...) {
  if (!length(kw <- which(subset(x, "keyword")))) {
    if (L(list.only)) {
      problem("no keywords", file)
      return(character())
    } else
      return(invisible(character()))
  }
  kw <- unlist(x[kw])
  if (L(list.only))
    return(kw)
  if (exists("rd.keywords", PKGUTILS_OPTIONS))
    known.kw <- PKGUTILS_OPTIONS$rd.keywords
  else
    PKGUTILS_OPTIONS$rd.keywords <- known.kw <- check_keywords()
  if (length(bad <- setdiff(kw, known.kw)))
    problem(paste("unknown keywords:", paste(bad, collapse = ", ")), file)
  else if ("internal" %in% kw && length(kw) > 1L)
    problem("'internal' combined with other keyword(s)", file)
  else if (length(doctype <- unlist(subset(x, "docType", values = TRUE))))
    switch(doctype,
      class = if (!all(c("classes", "methods") %in% kw))
        problem("'class' docType but no 'classes' or no 'methods' keyword",
          file),
      data = if (!"datasets" %in% kw)
        problem("'data' docType but no 'datasets' keyword", file),
      package = if (!"package" %in% kw)
        problem("'package' docType but no 'package' keyword", file)
    )
  invisible(kw)
}


################################################################################


#' Check presence of examples in Rd files
#'
#' Check presence/absence of examples in Rd files. They should not be present
#' if the list of \sQuote{keywords} contains \sQuote{internal}. Otherwise they
#' should be present unless \sQuote{docType} is \sQuote{class} or
#' \sQuote{package}.
#'
#' @inheritParams check_keywords
#' @param x Object of class \sQuote{Rd}.
#' @export
#' @note This does \strong{not} run the examples, if any, and check their
#'   results.
#' @return Logical scalar, \code{TRUE} if examples are either there and should
#'   be there or are missing and should be missing, \code{FALSE} otherwise.
#'   As a side effect, problem messages are printed to \code{stderr}. See
#'   \code{\link{logfile}} for how to send these messages to a file.
#' @family rd-functions
#' @keywords package
#' @examples
#' data(xyplot.docu)
#' (x <- check_examples(xyplot.docu))
#' stopifnot(x)
#'
check_examples <- function(x, ...) UseMethod("check_examples")

#' @rdname check_examples
#' @method check_examples Rd
#' @export
#'
check_examples.Rd <- function(x, file = NULL, ...) {
  kw <- check_keywords(x = x, file = file, list.only = TRUE, ...)
  if (length(ex <- which(subset(x, "examples"))) && "internal" %in% kw) {
    problem("'internal' keyword but examples present", file)
    FALSE
  } else if (!length(ex) && !"internal" %in% kw) {
    doctype <- unlist(subset(x, "docType", values = TRUE))
    if (all(doctype %in% c("class", "package")))
      TRUE
    else {
      problem("no 'internal' keyword but examples missing", file)
      FALSE
    }
  } else
    TRUE
}


################################################################################


