#' @name sprinkle_rotate_degree
#' @title Sprinkle Appearance of NA's
#' 
#' @description The content of cells may be rotated when it is desired to save
#' space (such as long table column names), or to draw attention to the 
#' cells.
#'   
#' @param x An object of class \code{dust}
#' @param rows Either a numeric vector of rows in the tabular object to be 
#'   modified or an object of class \code{call}.  When a \code{call}, 
#'   generated by \code{quote(expression)}, the expression resolves to 
#'   a logical vector the same length as the number of rows in the table.
#'   Sprinkles are applied to where the expression resolves to \code{TRUE}.
#' @param cols Either a numeric vector of columns in the tabular object to
#'   be modified, or a character vector of column names. A mixture of 
#'   character and numeric indices is permissible.
#' @param rotate_degree \code{numeric(1)} Indicates how much to rotate the 
#'   cell text in degrees.
#' @param part A character string denoting which part of the table to modify.
#' @param fixed \code{logical(1)} indicating if the values in \code{rows} 
#'   and \code{cols} should be read as fixed coordinate pairs.  By default, 
#'   sprinkles are applied at the intersection of \code{rows} and \code{cols}, 
#'   meaning that the arguments do not have to share the same length.  
#'   When \code{fixed = TRUE}, they must share the same length.
#' @param recycle A \code{character} one that determines how sprinkles are 
#'   managed when the sprinkle input doesn't match the length of the region
#'   to be sprinkled.  By default, recycling is turned off.  Recycling 
#'   may be performed across rows first (left to right, top to bottom), 
#'   or down columns first (top to bottom, left to right).
#' @param ... Additional arguments to pass to other methods. Currently ignored.
#' 
#' @section Functional Requirements:
#' \enumerate{
#'  \item Correctly reassigns the appropriate elements \code{rotate_degree} column
#'    in the table part.
#'  \item Casts an error if \code{x} is not a \code{dust} object.
#'  \item Casts an error if \code{rotate_degree} is not a \code{numeric(1)}
#'  \item Casts an error if \code{part} is not one of \code{"body"}, 
#'    \code{"head"}, \code{"foot"}, or \code{"interfoot"}
#'  \item Casts an error if \code{fixed} is not a \code{logical(1)}
#'  \item Casts an error if \code{recycle} is not one of \code{"none"},
#'    \code{"rows"}, or \code{"cols"}
#'  \item Cast an error if \code{recycle = "none"} and \code{rotate_degree}
#'    does not have length 1.
#' }
#' 
#' The functional behavior of the \code{fixed} and \code{recycle} arguments 
#' is not tested for this function. It is tested and validated in the
#' tests for \code{\link{index_to_sprinkle}}.
#' 
#' @seealso \code{\link{sprinkle}}, 
#'   \code{\link{index_to_sprinkle}}
#'
#' @export

sprinkle_rotate_degree <- function(x, rows = NULL, cols = NULL,
                               rotate_degree = NULL, 
                               part = c("body", "head", "foot", "interfoot", "table"),
                               fixed = FALSE, 
                               recycle = c("none", "rows", "cols", "columns"),
                               ...)
{
  UseMethod("sprinkle_rotate_degree")
}

#' @rdname sprinkle_rotate_degree
#' @export

sprinkle_rotate_degree.default <- function(x, rows = NULL, cols = NULL,
                                       rotate_degree = NULL, 
                                       part = c("body", "head", "foot", "interfoot", "table"),
                                       fixed = FALSE, 
                                       recycle = c("none", "rows", "cols", "columns"),
                                       ...)
{
  coll <- checkmate::makeAssertCollection()
  
  indices <- index_to_sprinkle(x = x, 
                               rows = rows, 
                               cols = cols, 
                               fixed = fixed,
                               part = part,
                               recycle = recycle,
                               coll = coll)
  
  recycle <- recycle[1]
  
  sprinkle_rotate_degree_index_assert(rotate_degree = rotate_degree, 
                                      recycle = recycle,
                                      coll = coll)
  
  checkmate::reportAssertions(coll)
  
  # At this point, part should have passed the assertions in 
  # index_to_sprinkle. The first element is expected to be valid.
  
  part <- part[1]
  
  sprinkle_rotate_degree_index(x = x, 
                               indices = indices, 
                               rotate_degree = rotate_degree, 
                               part = part)
}

#' @rdname sprinkle_rotate_degree
#' @export

sprinkle_rotate_degree.dust_list <- function(x, rows = NULL, cols = NULL,
                                         rotate_degree = NULL, 
                                         part = c("body", "head", "foot", "interfoot", "table"),
                                         fixed = FALSE, 
                                         recycle = c("none", "rows", "cols", "columns"),
                                         ...)
{
  structure(
    lapply(X = x,
           FUN = sprinkle_rotate_degree.default,
           rows = rows,
           cols = cols,
           rotate_degree = rotate_degree,
           part = part,
           fixed = fixed,
           recycle = recycle,
           ...),
    class = "dust_list"
  )
}

# Unexported Utility ------------------------------------------------

# These functions are to be used inside of the general `sprinkle` call
# When used inside `sprinkle`, the indices are already determined, 
# the only the `rotate_degree` argument needs to be validated. 
# The assert function is kept separate so it may be called earlier
# without attempting to perform the assignment.

sprinkle_rotate_degree_index_assert <- function(rotate_degree = NULL, 
                                                recycle = "none",
                                                coll)
{
  if (!is.null(rotate_degree))
  {
    checkmate::assert_numeric(x = rotate_degree,
                              add = coll,
                              .var.name = "rotate_degree")
    
    if (recycle == "none" && length(rotate_degree) != 1)
      coll$push("When `recycle` = 'none', rotate_degree must have length 1.")
  }
}

sprinkle_rotate_degree_index <- function(x, indices, rotate_degree, part)
{
  x[[part]][["rotate_degree"]][indices] <- rotate_degree
  
  x
}