################################################################################
#' updateInput_phenofit
#'
#' Update shiny components when input changes.
#' @param session shiny session
#' @param rv reactiveValues defined in the server, has variables of
#' 'df', 'st' and 'sites'
updateInput_phenofit <- function(session, rv) {
    ## update `nptperyear`
    var_time   <-  intersect(c("date", "t"), colnames(rv$df))[1]
    deltaT     <-  as.numeric(diff(rv$df[[var_time]][c(1, 2)]))
    nptperyear <<- ceiling(365/deltaT)

    updateNumericInput(session, 'nptperyear', value = nptperyear)

    ## update `var_VI``
    updateSelectInput(session, "txt_varVI",
        choices = select_var_VI(rv$df),
        selected = select_var_VI(rv$df)[1])

    # print("here")
    # browser()
    # update_VI(rv, input$txt_varVI) # update_VI right now.

    ## update `var_QC`
    varQCs <- colnames(rv$df) %>% .[grep("QA|QC|qa|qc", .)]
    if (length(varQCs) == 0){
        sel_qc_title <- paste0("vairable of QC: ",
            "No QC variables!")
        seq_qc <- ""; varQCs <- ""
    } else {
        sel_qc_title <- "vairable of QC:"
        sel_qc <- varQCs[1]
    }

    # Do not update values when varQC changes. Because, it also rely on
    # qcFUN
    updateSelectInput(session, "txt_varQC", sel_qc_title,
        choices = varQCs, varQCs[1])

    ## Update `site` in main panel
    updateSelectInput(session, "site",
                      choices = rv$sites, rv$sites[1])
}

#' getDf.site
#'
#' Select the data of specific site. Only those variables
#' \code{c('t', 'y', 'w')} selected.
getDf.site  <- function(df, sitename, dateRange){
    d <- dplyr::select(df[site == sitename, ], dplyr::matches("t|y|w|QC_flag"))
    # if has no \code{QC_flag}, it will be generated by \code{w}.

    # filter dateRange
    if (!missing(dateRange)){
        bandname <- intersect(c("t", "date"), colnames(d))[1]
        dates    <- d[[bandname]]
        I <- dates >= dateRange[1] & dates <= dateRange[2]
        d <- d[I, ]
    }
    d
    #%T>% plot_input(365)
}

getINPUT.site <- function(df, st, sitename, dateRange){
    sp       <- st[site == sitename]
    south    <- sp$lat < 0
    IGBP     <- sp$IGBP %>% {ifelse(is.null(.), "", .)}

    titlestr <- sprintf("[%s] IGBP=%s, lat = %.2f", sp$site, IGBP, sp$lat)

    d <- getDf.site(df, sitename, dateRange)

    dnew     <- add_HeadTail(d, south = south, nptperyear = nptperyear)
    INPUT    <- check_input(dnew$t, dnew$y, dnew$w, QC_flag = dnew$QC_flag,
        nptperyear = nptperyear, south = south,
        maxgap = nptperyear/4, alpha = 0.02, wmin = 0.2)

    INPUT$titlestr <- titlestr
    INPUT
}

#' Cal growing season dividing information
#'
#' @param input Shiny \code{input} variable
#' @param INPUT An object returned by \code{check_season}
cal_season <- function(input, INPUT){
    param <- list(
        FUN_season     = input$FUN_season,
        rFUN           = input$rFUN,
        iters          = input$iters,
        lambda         = input$lambda,
        nf             = input$nf,
        frame          = input$frame,
        wFUN           = input$wFUN,
        maxExtendMonth = input$maxExtendMonth,
        rtrough_max   = input$rtrough_max,
        r_max  = input$r_max,
        r_min  = input$r_min
    )

    print(str(param, 1))
    print(sprintf('nptperyear = %d', INPUT$nptperyear))
    # param <- lapply(varnames, function(var) input[[var]])

    param <- c(list(INPUT = INPUT), param)
    # print(str(param))

    do.call(check_season, param) # brk return
}

check_season <- function(INPUT,
                         FUN_season = c("season", "season_mov"),
                         rFUN = "wWHIT",
                         wFUN = "wTSM",
                         lambda = 1000,
                         iters = 3,
                         IsPlot = F, ...) {
    # sitename <- "US-ARM" # "FR-LBr", "ZA-Kru", "US-ARM"
    FUN_season <- get(FUN_season[1])
    wFUN       <- get(wFUN)

    res  <- FUN_season(INPUT,
                     rFUN = get(rFUN),
                     wFUN = wFUN,
                     IsPlot = IsPlot,
                     IsPlot.OnlyBad = FALSE,
                     lambda = lambda,
                     iters = iters,
                     MaxPeaksPerYear = 3,
                     MaxTroughsPerYear = 4,
                     ...,
                     # caution about the following parameters
                     minpeakdistance = nptperyear/6,
                     ypeak_min = 0
    )

    if (IsPlot){
        abline(h = 1, col = "red")
        title(INPUT$titlestr)
    }
    return(res)
}

phenofit_all <- function(input, progress = NULL){
    n   <- length(sites)
    res <- list()

    # parameters for Fine Fitting
    params_fineFitting <- list(
        methods      = input$FUN, #c("AG", "zhang", "beck", "elmore", 'Gu'), #,"klos",
        # debug        = FALSE,
        wFUN         = get(input$wFUN2),
        nextent      = 2,
        maxExtendMonth = 3,
        minExtendMonth = 1,
        QC_flag        = NULL,
        minPercValid = 0.2,
        print        = TRUE
    )

    showProgress <- !is.null(progress)
    if (showProgress){
        on.exit(progress$close())
        progress$set(message = sprintf("phenofit (n=%d) | running ", n), value = 0)
    }

    # print('debug 1 ...')
    # browser()
    for (i in 1:n){
        # tryCatch({
        # }, error = function(e){
        # })
        if (showProgress){
            progress$set(i, detail = paste("Doing part", i))
        }
        fprintf("phenofit (n = %d) | running %03d ... \n", i, n)

        sitename <- sites[i]
        INPUT    <- getINPUT.site(df, st, sitename, input$dateRange)

        # Rough Fitting and gs dividing
        brks   <- cal_season(input, INPUT)

        params <- c(list(INPUT = INPUT, brks = brks), params_fineFitting)
        fit    <- do.call(curvefits, params)

        stat  <- get_GOF(fit)                       # Goodness-Of-Fit
        pheno <- get_pheno(fit, IsPlot=FALSE)   # Phenological metrics

        ans   <- list(fit = fit, INPUT = INPUT, seasons = brks, stat = stat, pheno = pheno)
        ############################# CALCULATION FINISHED #####################
        res[[i]] <- ans
    }
    set_names(res, sites)
}

# plot_data <- function(d, title){
#     par(setting)
#     do.call(check_input, d) %>% plot_input()
#     mtext(title, side = 2, line = 2, cex = 1.3, font = 2)
# }
