% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pense.R
\name{pense}
\alias{pense}
\title{Penalized Elastic Net S-estimators for Regression}
\usage{
pense(x, y, alpha = 0.5, nlambda = 50, lambda, lambda_min_ratio,
  standardize = TRUE, initial = c("warm", "cold"), warm_reset = 10,
  cv_k = 5, cv_objective, ncores = getOption("mc.cores", 1L), cl = NULL,
  options = pense_options(), init_options = initest_options(),
  en_options = en_options_aug_lars())
}
\arguments{
\item{x}{design matrix with predictors.}

\item{y}{response vector.}

\item{alpha}{elastic net mixing parameter with \eqn{0 \le \alpha \le 1}.
\code{alpha = 1} is the LASSO penalty, and \code{alpha = 0} the Ridge
penalty.}

\item{nlambda}{if \code{lambda} is not given or \code{NULL} (default),
a grid of \code{nlambda} lambda values is generated based on the data.}

\item{lambda}{a single value or a grid of values for the regularization
parameter lambda.
Assumed to be on the same scale as the data and adjusted for
S-estimation. If missing a grid of lambda
values is automatically generated (see parameter \code{nlambda}).
If given and \code{standardize = TRUE}, the lambda values will be
adjusted accordingly.}

\item{lambda_min_ratio}{If the grid should be chosen automatically, the
ratio of the smallest lambda to the (computed) largest lambda.}

\item{standardize}{should the data be standardized robustly? Estimates
are returned on the original scale. Defaults to \code{TRUE}.}

\item{initial}{how to initialize the estimator at a new lambda in the grid.
The default, \code{"warm"}, computes a cold initial estimator at several
lambda values and uses the PENSE coefficient to warm-start the
estimator at the next larger lambda value. At the largest value in
the lambda grid, PENSE will be initialized with the 0-vector.
\code{"cold"} computes the full initial estimator at
every lambda value.}

\item{warm_reset}{if \code{initial = "warm"} (default), how many cold initial
estimates be computed?}

\item{cv_k}{number of cross-validation segments to use to choose the optimal
lambda from the grid. If only a single value of lambda is given,
cross-validation can still done to estimate the prediction performance
at this particular lambda.}

\item{cv_objective}{a function (name) to compute the CV performance.
By default, the robust tau-scale is used.}

\item{ncores, cl}{the number of processor cores or an actual parallel cluster
to use to estimate the optimal value of lambda. See
\code{\link{makeCluster}} on how to create a cluster manually.}

\item{options}{additional options for the PENSE algorithm.
See \code{\link{pense_options}} for details.}

\item{init_options}{additional options for computing the cold initial
estimates.
Ignored if \code{initial = "warm"} and \code{warm_reset = 0}.
See \code{\link{initest_options}} for details.}

\item{en_options}{additional options for the EN algorithm.
See \code{\link{elnet}} and \code{\link{en_options}} for details.}
}
\value{
An object of class \code{"pense"} with elements
\item{lambda}{grid of regularization parameter values for which an estimate
     is available.}
\item{lambda_opt}{the optimal value of the regularization parameter
     according to CV.}
\item{coefficients}{a sparse matrix of coefficients for each lambda in the
     grid.}
\item{residuals}{a matrix of residuals for each lambda in the grid.}
\item{cv_lambda_grid}{a data frame with CV prediction errors and several
     statistics of the solutions.}
\item{scale}{the estimated scales each lambda in the grid.}
\item{objective}{value of the objective function at each lambda in the grid.}
\item{adjusted}{necessary information to compute the corrected EN estimates.}
\item{call}{the call that produced this object.}
\item{...}{values of the given arguments.}
}
\description{
Computes the highly robust Penalized Elastic Net S-estimators (PENSE) for
linear regression models.
}
\details{
The PENSE estimate minimizes the robust M-scale of the residuals penalized
by the L1 and L2 norm of the regression coefficients (elastic net penalty).
The level of penalization is chosen to minimize the \code{cv_k}-fold
cross-validated prediction error (using a robust measure).
}
\section{Initial Estimate}{

By default (\code{initial == "warm"}), the method does not compute a
full initial estimate at each
lambda value in the grid, but only at \code{warm_reset} of the lambda
values. At the remaining lambda values, the estimate at the previous
lambda value is used to initialize the estimator (the lambda grid is
first traversed in descending and then in ascending direction). If
\code{warm_reset} is 1, only the 0-vector is used to initialize PENSE at the
largest penalty value. No further initial estimates are computed.

If \code{initial == "cold"}, a full initial estimate is computed at each
lambda value. This is equal to setting \code{warm_reset} to
\code{length(lambda)}.
}

\examples{
##
## A very simple example on artificial data
##

# Generate some dummy data
set.seed(12345)
n <- 30
p <- 15
x <- 1 + matrix(rnorm(n * p), ncol = p)
y <- x \%*\% c(2:5, numeric(p - 4)) + rnorm(n)

x_test <- matrix(rnorm(10 * n * p), ncol = p)
y_test <- x_test \%*\% c(2:5, numeric(p - 4)) + rnorm(n)

# Compute the S-estimator with an EN penalty for 30 lambda values
# (Note: In real applications, warm_reset should be at least 5)
set.seed(1234)
est <- pense(
    x, y,
    alpha = 0.6,
    nlambda = 20,
    warm_reset = 1L,
    cv_k = 3
)

# We can plot the CV prediction error curve
plot(est)

# What is the RMSPE on test data
(rmspe <- sqrt(mean((y_test - predict(est, newdata = x_test))^2)))

##
## What happens if we replace 5 observations in the dummy data
## with outliers?
##
y_out <- y
y_out[1:3] <- rnorm(3, -500)

# Compute the S-estimator again
# (Note: In real applications, warm_reset should be at least 5)
set.seed(12345)
est_out <- pense(
    x, y_out,
    alpha = 0.6,
    nlambda = 20,
    warm_reset = 1L,
    cv_k = 3
)

# How does the RMSPE compare?
rmspe_out <- sqrt(mean((y_test - predict(est_out, newdata = x_test))^2))
c(rmspe = rmspe, rmspe_out = rmspe_out)

}
\seealso{
To improve the S-estimate with an M-step, see \code{\link{pensem}}.
}
