% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/likelihood.R, R/likelihood2.R
\name{likelihood}
\alias{likelihood}
\alias{likelihood.ped}
\alias{likelihood.list}
\alias{likelihood2}
\alias{likelihood2.ped}
\alias{likelihood2.list}
\title{Pedigree likelihood}
\usage{
likelihood(x, ...)

\method{likelihood}{ped}(
  x,
  markers = NULL,
  peelOrder = NULL,
  lump = TRUE,
  special = FALSE,
  alleleLimit = Inf,
  logbase = NULL,
  loopBreakers = NULL,
  allX = NULL,
  verbose = FALSE,
  theta = 0,
  ...
)

\method{likelihood}{list}(x, markers = NULL, logbase = NULL, ...)

likelihood2(x, ...)

\method{likelihood2}{ped}(
  x,
  marker1,
  marker2,
  rho = NULL,
  peelOrder = NULL,
  lump = TRUE,
  special = TRUE,
  alleleLimit = Inf,
  logbase = NULL,
  loopBreakers = NULL,
  verbose = FALSE,
  ...
)

\method{likelihood2}{list}(x, marker1, marker2, logbase = NULL, ...)
}
\arguments{
\item{x}{A \code{ped} object, a \code{singleton} object, or a list of such objects.}

\item{\dots}{Further arguments.}

\item{markers}{One or several markers compatible with \code{x}. Several input
forms are possible:
\itemize{
\item A \code{marker} object compatible with \code{x}.
\item A list of \code{marker} objects.
\item A vector of names or indices of markers attached to \code{x}. If \code{x} is a
list, this is the only valid input.
}}

\item{peelOrder}{For internal use.}

\item{lump}{Activate allele lumping, i.e., merging unobserved alleles. This
is an important time saver, and should be applied in nearly all cases. (The
parameter exists mainly for debugging purposes.) If any markers use a
non-lumpable mutation model, the \code{special} argument may be used to apply
more advanced methods.}

\item{special}{A logical indicating if special lumping procedures should be
attempted if the mutation model is not generally lumpable. By default FALSE
in \code{likelihood()} and TRUE in \code{likelihood2()}.}

\item{alleleLimit}{A positive number or \code{Inf} (default). If the mutation
model is not generally lumpable, and the allele count exceeds this limit,
switch to an \code{equal} model with the same rate and reapply lumping.}

\item{logbase}{Either NULL (default) or a positive number indicating the
basis for logarithmic output. Typical values are \code{exp(1)} and 10.}

\item{loopBreakers}{A vector of ID labels indicating loop breakers. If NULL
(default), automatic selection of loop breakers will be performed. See
\code{\link[pedtools:inbreedingLoops]{pedtools::breakLoops()}}.}

\item{allX}{For internal use; set to TRUE if all markers are X-chromosomal.}

\item{verbose}{A logical.}

\item{theta}{Theta correction.}

\item{marker1, marker2}{Single markers compatible with \code{x}.}

\item{rho}{The recombination rate between \code{marker1} and \code{marker2}. To make
biological sense \code{rho} should be between 0 and 0.5.}
}
\value{
A numeric with the same length as the number of markers indicated by
\code{markers}. If \code{logbase} is a positive number, the output is
\code{log(likelihood, logbase)}.
}
\description{
The \code{likelihood()} and \code{likelihood2()} functions constitute the heart of
\strong{pedprobr}. The former computes the pedigree likelihood for each indicated
marker. The latter computes the likelihood for a pair of linked markers
separated by a given recombination rate.
}
\details{
The implementation is based on the peeling algorithm of Elston and Stewart
(1971). A variety of situations are covered; see the Examples section for
some demonstrations.
\itemize{
\item autosomal and X-linked markers
\item complex inbred pedigrees
\item markers with mutation models
\item pedigrees with inbred founders
\item single markers or two linked markers
}

For more than two linked markers, see \code{\link[=likelihoodMerlin]{likelihoodMerlin()}}.

Allele lumping can significantly reduce computation time with highly
polymorphic STR markers and many untyped pedigree members. This is
particularly important in \code{likelihood2()} which is prone to run out of memory
without lumping. If a non-lumpable mutation model is used, specialised
lumping may still be possible in some situations. This is attempted if
\code{special = TRUE}, which is the default in \code{likelihood2()} but not in
\code{likelihood()}.
}
\examples{

### Simple likelihoods ###
p = 0.1
q = 1 - p
afr = c("1" = p, "2" = q)

# Singleton
s = singleton() |> addMarker(geno = "1/2", afreq = afr)
stopifnot(all.equal(likelihood(s), 2*p*q))

# Trio
x = nuclearPed() |> addMarker(geno = c("1/1", "1/2", "1/1"), afreq = afr)
lik = likelihood(x, verbose = TRUE)
stopifnot(all.equal(lik, p^2 * 2*p*q * 0.5))


### Example with inbred founder ###

# Set 100\% inbreeding for the father in the previous example
y = setFounderInbreeding(x, ids = 1, value = 1)

# Plot (notice the inbreeding coefficient)
plot(y, marker = 1)

stopifnot(all.equal(likelihood(y), p * 2*p*q * 0.5))


### Example with two linked markers

# Add a second marker, highly polymorphic
x = addMarker(x, geno = c(NA, NA, "1/1"), alleles = 1:10)

# Likelihood assuming complete linkage
likelihood2(x, 1, 2, rho = 0, verbose = TRUE)

}
\references{
Elston and Stewart (1971). \emph{A General Model for the Genetic
Analysis of Pedigree Data}. \doi{https://doi.org/10.1159/000152448}
}
\seealso{
\code{\link[=likelihoodMerlin]{likelihoodMerlin()}}, for likelihoods involving more than 2 linked markers.
}
\author{
Magnus Dehli Vigeland
}
