% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/likelihood.R, R/likelihood2.R
\name{likelihood}
\alias{likelihood}
\alias{likelihood.ped}
\alias{likelihood.list}
\alias{likelihood2}
\alias{likelihood2.ped}
\alias{likelihood2.list}
\title{Pedigree likelihood}
\usage{
likelihood(x, ...)

\method{likelihood}{ped}(
  x,
  markers = NULL,
  peelOrder = NULL,
  lump = TRUE,
  logbase = NULL,
  loopBreakers = NULL,
  allX = NULL,
  verbose = FALSE,
  theta = 0,
  ...
)

\method{likelihood}{list}(x, markers = NULL, logbase = NULL, ...)

likelihood2(x, ...)

\method{likelihood2}{ped}(
  x,
  marker1,
  marker2,
  rho = NULL,
  peelOrder = NULL,
  logbase = NULL,
  loopBreakers = NULL,
  verbose = FALSE,
  ...
)

\method{likelihood2}{list}(x, marker1, marker2, logbase = NULL, ...)
}
\arguments{
\item{x}{A \code{ped} object, a \code{singleton} object, or a list of such objects.}

\item{\dots}{Further arguments.}

\item{markers}{One or several markers compatible with \code{x}. Several input
forms are possible:
\itemize{
\item A \code{marker} object compatible with \code{x}.
\item A list of \code{marker} objects.
\item A vector of names or indices of markers attached to \code{x}. If \code{x} is a
list, this is the only valid input.
}}

\item{peelOrder}{For internal use.}

\item{lump}{Activate allele lumping, i.e., merging unobserved alleles. This
is an important time saver, and should be applied in nearly all cases. (The
parameter exists mainly for debugging purposes.) The lumping algorithm will
detect (and complain) if any markers use a non-lumpable mutation model.
Default: TRUE.}

\item{logbase}{Either NULL (default) or a positive number indicating the
basis for logarithmic output. Typical values are \code{exp(1)} and 10.}

\item{loopBreakers}{A vector of ID labels indicating loop breakers. If NULL
(default), automatic selection of loop breakers will be performed. See
\code{\link[pedtools:inbreedingLoops]{pedtools::breakLoops()}}.}

\item{allX}{For internal use; set to TRUE if all markers are X-chromosomal.}

\item{verbose}{A logical.}

\item{theta}{Theta correction.}

\item{marker1, marker2}{Single markers compatible with \code{x}.}

\item{rho}{The recombination rate between \code{marker1} and \code{marker2}. To make
biological sense \code{rho} should be between 0 and 0.5.}
}
\value{
A numeric with the same length as the number of markers indicated by
\code{markers}. If \code{logbase} is a positive number, the output is
\code{log(likelihood, logbase)}.
}
\description{
The \code{likelihood()} and \code{likelihood2()} functions constitute the heart of
\strong{pedprobr}. The former computes the pedigree likelihood for each indicated
marker. The latter computes the likelihood for a pair of linked markers
separated by a given recombination rate.
}
\details{
The implementation is based on the peeling algorithm of Elston and Stewart
(1971). A variety of situations are covered; see the Examples section for
some demonstrations.
\itemize{
\item autosomal and X-linked markers
\item 1 marker or 2 linked markers
\item complex inbred pedigrees
\item markers with mutation models
\item pedigrees with inbred founders
}

For more than two linked markers, see \code{\link[=likelihoodMerlin]{likelihoodMerlin()}}.
}
\examples{

### Simple likelihood ###
p = 0.1
q = 1 - p
afr = c("1" = p, "2" = q)

# Singleton
s = singleton() |> addMarker(geno = "1/2", afreq = afr)

stopifnot(all.equal(likelihood(s), 2*p*q))

# Trio
trio = nuclearPed() |>
  addMarker(geno = c("1/1", "1/2", "1/1"), afreq = afr)

stopifnot(all.equal(likelihood(trio), p^2 * 2*p*q * 0.5))


### Example of calculation with inbred founders ###

### Case 1: Trio with inbred father
x = cousinPed(0, child = TRUE)
x = addSon(x, 5)
x = relabel(x, old = 5:7, new = c("father", "mother", "child"))

# Add equifrequent SNP; father homozygous, child heterozygous
x = addMarker(x, father = "1/1", child = "1/2")

# Plot with genotypes
plot(x, marker = 1)

# Compute the likelihood
lik1 = likelihood(x, markers = 1)


### Case 2: Using founder inbreeding
# Remove ancestry of father
y = subset(x, c("father", "mother", "child"))

# Indicate that the father has inbreeding coefficient 1/4
founderInbreeding(y, "father") = 1/4

# Plot (notice the inbreeding coefficient)
plot(y, marker = 1)

# Likelihood should be the same as above
lik2 = likelihood(y, markers = 1)

stopifnot(all.equal(lik1, lik2))



}
\references{
Elston and Stewart (1971). \emph{A General Model for the Genetic
Analysis of Pedigree Data}. \doi{https://doi.org/10.1159/000152448}
}
\seealso{
\code{\link[=likelihoodMerlin]{likelihoodMerlin()}}, for likelihoods involving more than 2 linked markers.
}
\author{
Magnus Dehli Vigeland
}
