% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cluster.R
\name{pdSpecClust}
\alias{pdSpecClust}
\title{Wavelet-based clustering of multivariate spectra.}
\usage{
pdSpecClust(P, D.hat = NULL, K, m = 2, jmax, d.jmax = 0.1,
  eps = c(1e-04, 1e-04), tau = 0.5, return.D = FALSE, ...)
}
\arguments{
\item{P}{a (\eqn{d,d,m,S})-dimensional array, with \eqn{m} a dyadic number.}

\item{D.hat}{a list with \eqn{S} elements, where each element is an object given as output by the function
\code{\link{pdSpecEst}}. If both arguments \code{P} and \code{D.hat} are provided, only the values in \code{D.hat}
are taken into account.}

\item{K}{the number of clusters to be considered.}

\item{m}{the fuzziness parameter for both the fuzzy c-medoids and the weighted fuzzy c-means algorithm. \code{m}
should be larger or equal to \eqn{1}. If \eqn{m = 1} the cluster assignments are no longer fuzzy (i.e. the procedure
performs hard clustering).}

\item{jmax}{an upper bound on the maximum wavelet scale to be considered in the clustering procedure. If
\code{jmax} is not specified, it is set equal to the largest (i.e. finest) possible wavelet scale.}

\item{d.jmax}{a proportion that is used to determine the maximum wavelet scale to be considered in the clustering
procedure. A larger value \code{d.jmax} leads to less wavelet coefficients being taken into account, and therefore
lower computational effort in the procedure. If \code{d.jmax} is not specified, by default \code{d.jmax = 0.1}.}

\item{eps}{an optional vector with two elements, \code{eps[1]} determines the termination criterion in the fuzzy c-medoids
algorithm (i.e. first clustering step), and \code{eps[2]} determines the termination criterion in the weighted fuzzy c-means algorithm
(i.e. second clustering step). If \code{eps} is not specified, by default \code{eps = c(1E-4, 1E-4)}.}

\item{tau}{an optional argument tuning the weight given to the cluster assignments obtained in the first step of
the clustering algorithm. If \code{tau} is not specified, by default \code{tau = 0.5}.}

\item{return.D}{an optional argument specifying whether to return also the list of coarsest midpoints and wavelet coefficients
(i.e. the features) for each individual subject, by default \code{return.D = FALSE}.}

\item{...}{additional arguments passed on to \code{\link{pdSpecEst}}. These arguments are only used if
\code{is.null(D.hat)}, otherwise the function \code{\link{pdSpecEst}} is not called.}
}
\value{
The function returns an (\eqn{S, K})-dimensional matrix, where the value at position (\eqn{s,k}) in the
matrix corresponds to the probabilistic cluster membership assignment of subject \eqn{s} with respect to
cluster \eqn{k}. If \code{isTRUE(return.D)} also returns the list of coarsest midpoints and wavelet coefficients
of each subject used in the clustering procedure.
}
\description{
\code{pdSpecClust} performs clustering of multivariate spectral matrices via a two-step fuzzy
clustering algorithm in the manifold wavelet domain. The clustering procedure is described in
detail in (Chau and von Sachs, 2017).
}
\details{
The input array \code{P} contains initial noisy Hermitian PD spectral estimates of the
(\eqn{d \times d})-dimensional spectral matrices at \eqn{m} different frequencies for \eqn{S}
different subjects. Here \eqn{m = 2^J} for some \eqn{J > 0}. The initial spectral estimates can
be e.g. the averaged periodograms given as output in \code{\link{pdPgram}}. \cr
For each subject \eqn{s}, the thresholded wavelet coefficients in the manifold wavelet domain are
calculated by the function \code{\link{pdSpecEst}}. Instead of providing the noisy input data \code{P},
it is possible to directly provide the thresholded wavelet coefficients via the argument \code{D.hat}.
\code{D.hat} should be a list of \eqn{S} elements, where each element corresponds to an object obtained
from the function \code{\link{pdSpecEst}}.\cr
The maximum wavelet scale taken into account in the clustering procedure is determined by the arguments
\code{jmax} and \code{d.jmax}. The maximum scale is set to the minimum of \code{jmax} and the wavelet
scale \eqn{j} for which the proportion of nonzero components of thresholded wavelet coefficients (averaged
across subjects) is smaller than \code{d.jmax}.\cr
The \eqn{S} subjects are assigned to \eqn{K} different fuzzy clusters according to a two-step procedure.
In the first step, a fuzzy c-medoids algorithm based on the Riemannian distance function as in
\code{\link{pdDist}}, with fuzziness parameter \eqn{m}, is applied to the \eqn{S} coarsest-scale midpoint
vectors, i.e. scale \code{j = 1}. \cr
In the second step, a weighted fuzzy c-means algorithm based on the Euclidean
distance function, also with fuzziness parameter \eqn{m}, is applied to the nonzero thresholded wavelet
coefficients for the \eqn{S} different subjects. The tuning parameter \code{tau} controls the weight given
to the cluster assignments obtained in the first step of the clustering algorithm. For more details on the
procedure we refer to (Chau and von Sachs, 2017).
}
\examples{
## ARMA(1,1) process: Example 11.4.1 in (Brockwell and Davis, 1991)

Phi1 <- array(c(0.7, 0, 0, 0.6, rep(0, 4)), dim = c(2, 2, 2))
Phi2 <- array(c(0.7, 0, 0, 0.5, rep(0, 4)), dim = c(2, 2, 2))
Theta <- array(c(0.5, -0.7, 0.6, 0.8, rep(0, 4)), dim = c(2, 2, 2))
Sigma <- matrix(c(1, 0.71, 0.71, 2), nrow = 2)

## Generate periodogram data for 10 subjects
pgram <- function(Phi) pdPgram(rARMA(2^9, 2, Phi, Theta, Sigma)$X)$P
P <- array(c(replicate(5, pgram(Phi1)), replicate(5, pgram(Phi2))), dim=c(2,2,2^7,10))

cl <- pdSpecClust(P, K=2)

}
\references{
Chau, J. and von Sachs, R. (2017). \emph{Positive definite multivariate spectral
estimation: a geometric wavelet approach}. Available at \url{http://arxiv.org/abs/1701.03314}.

Brockwell, P.J. and Davis, R.A. (1991). \emph{Time series: Theory and Methods}. New York: Springer.
}
\seealso{
\code{\link{pdPgram}}, \code{\link{pdSpecEst}}, \code{\link{pdDist}}
}
