#' @title
#' Class for Package Options
#'
#' @description
#' This class holds public fields that represent the package
#' [`options`][base::options()] used to configure the default behavior of the
#' functionality [`parabar::parabar`] provides.
#'
#' @details
#' An instance of this class is automatically created and stored in the session
#' [`base::.Options`] at load time. This instance can be accessed and changed
#' via [`getOption("parabar")`][base::getOption()]. Specific package
#' [`options`][base::options()] can be retrieved using the helper function
#' [parabar::get_option()].
#'
#' @examples
#' # Set the default package options (i.e., automatically set at load time).
#' set_default_options()
#'
#' # First, get the options instance from the session options.
#' parabar <- getOption("parabar")
#'
#' # Then, disable progress tracking.
#' parabar$progress_track <- FALSE
#'
#' # Check that the change was applied (i.e., `progress_track: FALSE`).
#' getOption("parabar")
#'
#' # To restore defaults, set the default options again.
#' set_default_options()
#'
#' # Check that the change was applied (i.e., `progress_track: TRUE`).
#' getOption("parabar")
#'
#' # We can also use the built-in helpers to get and set options more conveniently.
#'
#' # Get the progress tracking option.
#' get_option("progress_track")
#'
#' # Set the progress tracking option to `FALSE`.
#' set_option("progress_track", FALSE)
#'
#' # Check that the change was applied (i.e., `progress_track: FALSE`).
#' get_option("progress_track")
#'
#' # Get a temporary file for logging the progress.
#' get_option("progress_log_path")
#'
#' # Fix the logging file path.
#' set_option("progress_log_path", "./progress.log")
#'
#' # Check that the logging path change was applied.
#' get_option("progress_log_path")
#'
#' # Restore the logging path to the default behavior.
#' set_option("progress_log_path", NULL)
#'
#' # Check that the logging path change was applied.
#' get_option("progress_log_path")
#'
#' # Restore the defaults.
#' set_default_options()
#'
#' @seealso [parabar::get_option()], [parabar::set_option()], and
#' [parabar::set_default_options()].
#'
#' @export
Options <- R6::R6Class("Options",
    cloneable = FALSE,

    private = list(
        .progress_log_path = NULL
    ),

    public = list(
        #' @field progress_track A logical value indicating whether progress
        #' tracking should be enabled (i.e., `TRUE`) or disabled  (i.e.,
        #' `FALSE`) globally for compatible backends. The default value is
        #' `TRUE`.
        progress_track = TRUE,

        #' @field progress_timeout A numeric value indicating the timeout (i.e.,
        #' in seconds) between subsequent checks of the log file for new progress
        #' records. The default value is `0.001`.
        progress_timeout = 0.001,

        #' @field progress_bar_type A character string indicating the default
        #' bar type to use with compatible backends. Possible values are
        #' `"modern"` (the default) or `"basic"`.
        progress_bar_type = "modern",

        #' @field progress_bar_config A list of lists containing the default bar
        #' configuration for each supported bar engine. Elements of these lists
        #' represent arguments for the corresponding bar engines. Currently, the
        #' supported bar engines are:
        #' - `modern`: The [`progress::progress_bar`] engine, with the following
        #'   default configuration:
        #'   - `show_after = 0`
        #'   - `format = "> completed :current out of :total tasks [:percent] [:elapsed]"`
        #' - `basic`: The [`utils::txtProgressBar`] engine, with no default
        #'   configuration.
        progress_bar_config = list(
            # See `progress::progress_bar`.
            modern = list(
                show_after = 0,
                format = " > completed :current out of :total tasks [:percent] [:elapsed]"
            ),

            # See `utils::txtProgressBar`.
            basic = list()
        )
    ),

    active = list(
        #' @field progress_log_path A character string indicating the path to
        #' the log file where to track the execution progress of a running task.
        #' The default value is a temporary file generated by
        #' [`base::tempfile()`]. Calling this active binding repeatedly will
        #' yield different temporary file paths. Fixing the path to a specific
        #' value is possible by setting this active binding to a character
        #' string representing the desired path. Setting this active binding to
        #' `NULL` will reset it to the default value (i.e., yielding different
        #' temporary file paths).
        progress_log_path = function(value) {
            # If no value is provided, nor already set.
            if (missing(value)) {
                # If the path has not been set yet.
                if (is.null(private$.progress_log_path)) {
                    # Then generate and return random path.
                    return(tempfile(pattern = "parabar"))
                }

                # Otherwise return the path already set.
                return(private$.progress_log_path)
            }

            # Otherwise set the path to what has been requested by the user.
            private$.progress_log_path <- value

            # Return the path.
            return(private$.progress_log_path)
        }
    )
)
