% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get.R
\name{oe_get}
\alias{oe_get}
\title{Find, download, translate and read OSM extracts from several providers}
\usage{
oe_get(
  place,
  layer = "lines",
  ...,
  provider = "geofabrik",
  match_by = "name",
  max_string_dist = 1,
  level = NULL,
  download_directory = oe_download_directory(),
  force_download = FALSE,
  max_file_size = 5e+08,
  vectortranslate_options = NULL,
  osmconf_ini = NULL,
  extra_tags = NULL,
  force_vectortranslate = FALSE,
  boundary = NULL,
  boundary_type = c("spat", "clipsrc"),
  download_only = FALSE,
  skip_vectortranslate = FALSE,
  never_skip_vectortranslate = FALSE,
  quiet = FALSE
)
}
\arguments{
\item{place}{Description of the geographical area that should be matched with
a \code{.osm.pbf} file. Can be either a length-1 character vector, an
\code{sf}/\code{sfc}/\code{bbox} object, or a numeric vector of coordinates with length 2.
In the last case, it is assumed that the EPSG code is 4326 specified as
c(LON, LAT), while you can use any CRS with \code{sf}/\code{sfc}/\code{bbox} objects. See
Details and Examples in \code{\link[=oe_match]{oe_match()}}.}

\item{layer}{Which \code{layer} should be read in? Typically \code{points}, \code{lines}
(the default), \code{multilinestrings}, \code{multipolygons} or \code{other_relations}. If
you specify an ad-hoc query using the argument \code{query} (see introductory
vignette and examples), then \code{oe_get()} and \code{oe_read()} will read the layer
specified in the query and ignore \code{layer}. See also
\href{https://github.com/ropensci/osmextract/issues/122}{#122}.}

\item{...}{Arguments that will be passed to \code{\link[sf:st_read]{sf::st_read()}}, like \code{query},
\code{wkt_filter} or \code{stringsAsFactors}.  Check the introductory vignette to
understand how to create your own (SQL-like) queries.}

\item{provider}{Which provider should be used to download the data? Available
providers can be found with the following command: \code{\link[=oe_providers]{oe_providers()}}. For
\code{\link[=oe_get]{oe_get()}} and \code{\link[=oe_match]{oe_match()}}, if \code{place} is equal to \verb{ITS Leeds}, then
\code{provider} is set equal to \code{test}. This is just for simple examples and
internal tests.}

\item{match_by}{Which column of the provider's database should be used for
matching the input \code{place} with a \code{.osm.pbf} file? The default is \code{"name"}.
Check Details and Examples in \code{\link[=oe_match]{oe_match()}} to understand how this parameter
works. Ignored if \code{place} is not a character vector since the matching is
performed through a spatial operation.}

\item{max_string_dist}{Numerical value greater or equal than 0. What is the
maximum distance in fuzzy matching (i.e. Approximate String Distance, see
\code{\link[=adist]{adist()}}) between input \code{place} and \code{match_by} column to tolerate before
testing alternative providers or looking for geographical matching with
Nominatim API? This parameter is set equal to 0 if \code{match_by} is equal to
\code{iso3166_1_alpha2} or \code{iso3166_2}. Check Details and Examples in
\code{\link[=oe_match]{oe_match()}} to understand why this parameter is important. Ignored if
\code{place} is not a character vector since the matching is performed through a
spatial operation.}

\item{level}{An integer representing the desired hierarchical level in case
of spatial matching. For the \code{geofabrik} provider, for example, \code{1}
corresponds with continent-level datasets, \code{2} for countries, \code{3}
corresponds to regions and \code{4} to subregions. Hence, we could approximately
say that smaller administrative units correspond to bigger levels. If
\code{NULL}, the default, the \verb{oe_*} functions will select the highest available
level. See Details and Examples in \code{oe_match()}.}

\item{download_directory}{Where to download the file containing the OSM data?
By default this is equal to \code{\link[=oe_download_directory]{oe_download_directory()}}, which is equal to
\code{\link[=tempdir]{tempdir()}} and it changes each time you restart R. You can set a
persistent \code{download_directory} by adding the following to your \code{.Renviron}
file (e.g. with \code{edit_r_environ} function in \code{usethis} package):
\verb{OSMEXT_DOWNLOAD_DIRECTORY=/path/to/osm/data}.}

\item{force_download}{Should the \code{.osm.pbf} file be updated if it has already
been downloaded? \code{FALSE} by default. This parameter is used to update old
\code{.osm.pbf} files.}

\item{max_file_size}{The maximum file size to download without asking in
interactive mode. Default: \code{5e+8}, half a gigabyte.}

\item{vectortranslate_options}{Options passed to the \code{\link[sf:gdal_utils]{sf::gdal_utils()}}
argument \code{options}. Set by default. Check details in the introductory
vignette and the help page of \code{\link[=oe_vectortranslate]{oe_vectortranslate()}}.}

\item{osmconf_ini}{The configuration file. See documentation at
\href{https://gdal.org/drivers/vector/osm.html}{gdal.org}. Check details in the
introductory vignette and the help page of \code{\link[=oe_vectortranslate]{oe_vectortranslate()}}. Set by
default.}

\item{extra_tags}{Which additional columns, corresponding to OSM tags, should
be in the resulting dataset? \code{NULL} by default. Check the introductory
vignette and the help pages of \code{\link[=oe_vectortranslate]{oe_vectortranslate()}} and \code{\link[=oe_get_keys]{oe_get_keys()}}.
Ignored when \code{osmconf_ini} is not \code{NULL}.}

\item{force_vectortranslate}{Boolean. Force the original \code{.pbf} file to be
translated into a \code{.gpkg} file, even if a \code{.gpkg} with the same name
already exists? \code{FALSE} by default. If tags in \code{extra_tags} match data
in previously translated \code{.gpkg} files no translation occurs
(see \href{https://github.com/ropensci/osmextract/issues/173}{#173} for details).
Check the introductory vignette and the help page of
\code{\link[=oe_vectortranslate]{oe_vectortranslate()}}.}

\item{boundary}{An \code{sf} or \code{sfc} object that will be used to create a spatial
filter during the vectortranslate operations. The type of filter can be
chosen using the argument \code{boundary_type}.}

\item{boundary_type}{A character vector of length 1 specifying the type of
spatial filter. The \code{spat} filter selects only those features that
intersect a given area, while \code{clipsrc} also clips the geometries. See the
examples and check \href{https://gdal.org/programs/ogr2ogr.html}{here} for more
details.}

\item{download_only}{Boolean. If \code{TRUE}, then the function only returns the
path where the matched file is stored, instead of reading it. \code{FALSE} by
default.}

\item{skip_vectortranslate}{Boolean. If \code{TRUE}, then the function skips all
vectortranslate operations and it reads (or simply returns the path) of the
\code{.osm.pbf} file. \code{FALSE} by default.}

\item{never_skip_vectortranslate}{Boolean. This is used in case the user
passed its own \code{.ini} file or vectortranslate options (since, in those case,
it's too difficult to determine if an existing \code{.gpkg} file was generated
following the same options.)}

\item{quiet}{Boolean. If \code{FALSE}, the function prints informative messages.
Starting from \code{sf} version
\href{https://r-spatial.github.io/sf/news/index.html#version-0-9-6-2020-09-13}{0.9.6},
if \code{quiet} is equal to \code{FALSE}, then vectortranslate operations will
display a progress bar.}
}
\value{
An \code{sf} object.
}
\description{
This function is used to find, download, translate and read OSM extracts
obtained from several providers. It is a wrapper around \code{\link[=oe_match]{oe_match()}} and
\code{\link[=oe_read]{oe_read()}}. Check the introductory vignette, the examples and the help pages
of the wrapped functions to understand the details behind all parameters.
}
\details{
The algorithm that we use for importing an OSM extract data into R
is divided into 4 steps: 1) match the input \code{place} with the url of a
\code{.pbf} file; 2) download the \code{.pbf} file; 3) convert it into \code{.gpkg} format
and 4) read-in the \code{.gpkg} file. The function \code{oe_match()} is used to
perform the first operation and the function \code{oe_read()} (which is a
wrapper around \code{oe_download()}, \code{oe_vectortranslate()} and \code{sf::st_read()})
performs the other three operations.
}
\examples{
# Match, download and convert OSM extracts associated to a simple test.
its = oe_get("ITS Leeds", quiet = FALSE, download_directory = tempdir())
class(its)
unique(sf::st_geometry_type(its))

# Get another layer from ITS Leeds extract
its_points = oe_get("ITS Leeds", layer = "points")
unique(sf::st_geometry_type(its_points))

# Get the .osm.pbf and .gpkg files paths
oe_get("ITS Leeds", download_only = TRUE, quiet = TRUE)
oe_get("ITS Leeds", download_only = TRUE, skip_vectortranslate = TRUE, quiet = TRUE)
# See also ?oe_find()

# Add additional tags
its_with_oneway = oe_get("ITS Leeds", extra_tags = "oneway")
names(its_with_oneway)
table(its_with_oneway$oneway, useNA = "ifany")

# Use the query argument to get only oneway streets:
q = "SELECT * FROM 'lines' WHERE oneway == 'yes'"
its_oneway = oe_get("ITS Leeds", query = q)
its_oneway[, c(1, 3, 9)]

# Apply a spatial filter during the vectortranslate operations
its_poly = sf::st_sfc(
  sf::st_polygon(
    list(rbind(
      c(-1.55577, 53.80850),
      c(-1.55787, 53.80926),
      c(-1.56096, 53.80891),
      c(-1.56096, 53.80736),
      c(-1.55675, 53.80658),
      c(-1.55495, 53.80749),
      c(-1.55577, 53.80850)
    ))
  ),
  crs = 4326
)
its_spat = oe_get("ITS Leeds", boundary = its_poly)
its_clipped = oe_get("ITS Leeds", boundary = its_poly, boundary_type = "clipsrc", quiet = TRUE)

plot(sf::st_geometry(its), reset = FALSE, col = "lightgrey")
plot(sf::st_boundary(its_poly), col = "black", add = TRUE)
plot(sf::st_boundary(sf::st_as_sfc(sf::st_bbox(its_poly))), col = "black", add = TRUE)
plot(sf::st_geometry(its_spat), add = TRUE, col = "darkred")
plot(sf::st_geometry(its_clipped), add = TRUE, col = "orange")

# More complex examples
\dontrun{
west_yorkshire = oe_get("West Yorkshire")
# If you run it again, the function will not download the file
# or convert it again
west_yorkshire = oe_get("West Yorkshire")
# Match with place name
oe_get("Milan") # Warning: the .pbf file is 400MB
oe_get("Vatican City") # Check all providers
oe_get("Zurich") # Use Nominatim API for geolocating places

# Match with coordinates (any EPSG)
milan_duomo = sf::st_sfc(sf::st_point(c(1514924, 5034552)), crs = 3003)
oe_get(milan_duomo, quiet = FALSE) # Warning: the .pbf file is 400MB
# Match with numeric coordinates (EPSG = 4326)
oe_match(c(9.1916, 45.4650), quiet = FALSE)

# Check also alternative providers
baku = oe_get(place = "Baku")

# Other examples:
oe_get("RU", match_by = "iso3166_1_alpha2", quiet = FALSE)
# The following example mimics read_sf
oe_get("Andora", stringsAsFactors = FALSE, quiet = TRUE, as_tibble = TRUE)}

# Remove .pbf and .gpkg files in tempdir
# (since they may interact with other examples)
file.remove(list.files(path = tempdir(), pattern = "(pbf|gpkg)", full.names = TRUE))
}
\seealso{
\code{\link[=oe_match]{oe_match()}}, \code{\link[=oe_download]{oe_download()}}, \code{\link[=oe_vectortranslate]{oe_vectortranslate()}},
and \code{\link[=oe_read]{oe_read()}}.
}
