\name{record}
\alias{record}
\title{Recombination Counting and Ordering}
\description{
  Implements the marker ordering algorithm \emph{Recombination Counting
    and Ordering} (\cite{Van Os et al., 2005}).
}
\usage{
record(input.seq, times=10, LOD=0, max.rf=0.5, tol=10E-5)
}
\arguments{
  \item{input.seq}{an object of class \code{sequence}.}
  \item{times}{integer. Number of replicates of the RECORD procedure.}
  \item{LOD}{minimum LOD-Score threshold used when constructing the
    pairwise recombination fraction matrix.}
  \item{max.rf}{maximum recombination fraction threshold used as the LOD
    value above.}
  \item{tol}{tolerance for the C routine, i.e., the value used to evaluate
    convergence.}
}
\details{
  \emph{Recombination Counting and Ordering} (\emph{RECORD}) is an
  algorithm for marker ordering in linkage groups. It is not an
  exhaustive search method and, therefore, is not computationally
  intensive. However, it does not guarantee that the best order is
  always found. The only requirement is a matrix with recombination
  fractions between markers. Next is an adapted excerpt from
  \cite{Mollinari et al (2009)} describing the \emph{RECORD}
  algorithm:
  
  \emph{Based on the expected number of recombination events, an
    \eqn{S} matrix is constructed, \eqn{S = [S_{M_{i}M_{j}}]_{m \times
    m}}{S = [S_MiMj]_(m x m)} (for \eqn{M_{i} = M_{j}}{M_i = M_j},
    \eqn{S_{M_{i}M_{j}} = 0}{S_MiMj = 0}), where \eqn{m} is the number
    of markers. The procedure to obtain \eqn{S} is based on the
    expected number of crossovers between marker pairs, conditioned by
    the observation of the markers' phenotype. The optimization
    criterion \emph{COUNT} for a sequence of \eqn{m} markers may be
    calculated by \eqn{COUNT = \sum_{i=1}^{m-1} S_{M_{i}M_{i+1}}}{COUNT
    = Sum_{i=1}^{m-1} S_MiMi+1}, where smaller \emph{COUNT} values
    correspond to better orders. Map building is carried out by randomly
    taking two markers and positioning a third one at the beginning, at
    the end and between them. The marker is fixed at the position that
    gives a smaller value of \emph{COUNT}. Similarly, the remaining
    markers are positioned at pre-established orders until completion of
    the map. Subsequently, a search for smaller values of \emph{COUNT}
    is performed, inverting the position on the map of subsequences of
    size \eqn{m' = 2, \ldots, 20}{m' = 2, ..., 20}. If the map resulting
    from the inverted positions presents a \emph{COUNT} value smaller
    than the previous one, it is kept. The procedure is repeated
    \code{times} times and the sequence presenting the smallest
    \emph{COUNT} value is chosen.}
  
  After determining the order with \emph{RECORD}, the final map is constructed
  using the multipoint approach (function \code{\link[onemap]{map}}).
}
\value{
  An object of class \code{sequence}, which is a list containing the
  following components:
  \item{seq.num}{a \code{vector} containing the (ordered) indices of
    markers in the sequence, according to the input file.}
  \item{seq.phases}{a \code{vector} with the linkage phases between markers
    in the sequence, in corresponding positions. \code{-1} means that there are
    no defined linkage phases.}
  \item{seq.rf}{a \code{vector} with the recombination frequencies between
    markers in the sequence. \code{-1} means that there are no estimated
    recombination frequencies.}
  \item{seq.like}{log-likelihood of the corresponding linkage map.}
  \item{data.name}{name of the object of class \code{outcross} with the
    raw data.}
  \item{twopt}{name of the object of class \code{rf.2pts} with the 2-point
    analyses.}
}
\references{
  Mollinari, M., Margarido, G. R. A., Vencovsky, R. and Garcia,
  A. A. F. (2009) Evaluation of algorithms used to order markers on
  genetics maps. \emph{Heredity} 103: 494-502.

  Van Os, H., Stam, P., Visser, R.G.F. and Van Eck, H.J. (2005) RECORD:
  a novel method for ordering loci on a genetic linkage
  map. \emph{Theoretical and Applied Genetics} 112: 30-40.
}
\author{Marcelo Mollinari, \email{mmollina@esalq.usp.br}}
\seealso{
  \code{\link[onemap]{make.seq}} and  \code{\link[onemap]{map}}
}
\examples{
  ##outcross example
  data(example.out)
  twopt <- rf.2pts(example.out)

  all.mark <- make.seq(twopt,"all")
  groups <- group(all.mark)
  LG1 <- make.seq(groups,1)
  LG1.rec <- record(LG1)

  ##F2 example
  data(fake.f2.onemap)
  twopt <- rf.2pts(fake.f2.onemap)
  all.mark <- make.seq(twopt,"all")
  groups <- group(all.mark)
  LG1 <- make.seq(groups,1)
  LG1.rec <- record(LG1)
  LG1.rec
}
\keyword{utilities}
