% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/spectral.R
\name{pwelch}
\alias{pwelch}
\title{Welch Periodogram}
\usage{
pwelch(
  x,
  window,
  noverlap,
  nfft,
  fs,
  spec,
  demean = FALSE,
  detrend = TRUE,
  plot = TRUE,
  debug = getOption("oceDebug"),
  ...
)
}
\arguments{
\item{x}{a vector or timeseries to be analyzed.  If \code{x} is a timeseries, then
it there is no need to \code{fs}, and doing so will result in an error if it does
not match the value inferred from \code{x}.}

\item{window}{optional numeric vector specifying a window to be applied
to the timeseries subsamples.  This is ignored if \code{spec} is provided.
Otherwise, if \code{window} is provided, then it must either
be of the same length as \code{nfft} or be of length 1.  In the first case,
the vector is multiplied into the timeseries subsample, and the length
of \code{window} must equal \code{nfft} is that is supplied.
In the second then \code{window} is taken to be the number
of sub-intervals into which the time series is to be broken up, with a
hamming window being used for each sub-interval.  If \code{window} is not
specified and \code{nfft} is given, then the window is constructed as
a hamming window with length \code{nfft}.  And, if neither \code{window}
nor \code{nfft} are specified, then \code{x} will be broken up
into 8 portions.}

\item{noverlap}{number of points to overlap between windows.  If not
specified, this will be set to half the window length.}

\item{nfft}{length of FFT. See \code{window} for how \code{nfft} interacts with
that argument.}

\item{fs}{frequency of time-series.  If \code{x} is a time-series, and if
\code{fs} is supplied, then time-series is altered to have frequency
\code{fs}.}

\item{spec}{optional function to be used for the computation of the spectrum,
to allow finer-grained control of the processing.
If provided, \code{spec} must accept a time-series as its first argument, and
must return a list containing the spectrum in \code{spec} and the
frequency in \code{freq}.
Note that no window will be applied to the data after subsampling,
and an error will be reported if \code{window} and \code{spec} are both given.
An error will be reported if \code{spec} is given but \code{nfft} is not given.
Note that the values of \code{demean}, \code{detrend} and \code{plot} are ignored if \code{spec}
is given. However, the \dots argument \emph{is} passed to \code{spec}.}

\item{demean, detrend}{logical values that can control the spectrum calculation,
in the default case of \code{spec}.  These are passed to \code{\link[=spectrum]{spectrum()}} and thence
\code{\link[=spec.pgram]{spec.pgram()}}; see the help pages for the latter for an explanation.}

\item{plot}{logical, set to \code{TRUE} to plot the spectrum.}

\item{debug}{a flag that turns on debugging.  Set to 1 to get a moderate
amount of debugging information, or to 2 to get more.}

\item{\dots}{optional extra arguments to be passed to
\code{\link[=spectrum]{spectrum()}}, or to \code{spec}, if the latter is given.}
}
\value{
\code{pwelch} returns a list mimicking the return value from \code{\link[=spectrum]{spectrum()}},
containing frequency \code{freq}, spectral power \code{spec}, degrees of
freedom \code{df}, bandwidth \code{bandwidth}, etc.
}
\description{
Compute periodogram using the Welch (1967) method. This is
somewhat analogous to the Matlab function of the same name,
but it is \emph{not} intended as a drop-in replacement.
}
\details{
First, \code{x} is broken up into chunks,
overlapping as specified by \code{noverlap}.  These chunks are then
multiplied by the window, and then
passed to \code{\link[=spectrum]{spectrum()}}.  The resulting spectra are then averaged,
with the results being stored in \code{spec} of the return value.  Other
entries of the return value mimic those returned by \code{\link[=spectrum]{spectrum()}}.

It should be noted that the actions of several parameters are interlocked,
so this can be a complex function to use.  For example, if \code{window} is
given and has length exceeding 1, then its length must equal \code{nfft}, if the
latter is also provided.
}
\section{Bugs}{

Both bandwidth and degrees of freedom are just copied from
the values for one of the chunk spectra, and are thus incorrect.  That means
the cross indicated on the graph is also incorrect.
}

\section{Historical notes}{

\itemize{
\item \strong{2021-06-26:} Until this date, \code{\link[=pwelch]{pwelch()}} passed the
subsampled timeseries portions through \code{\link[=detrend]{detrend()}}
before applying the window. This practice was dropped
because it could lead to over-estimates of low frequency
energy (as noticed by Holger Foysi of the University of Siegen),
perhaps because \code{\link[=detrend]{detrend()}} considers only endpoints and
therefore can yield inaccurate trend estimates.
In a related change, \code{demean} and \code{detrend} were added
as formal arguments, to avoid users having to trace the documentation
for \code{\link[=spectrum]{spectrum()}} and then \code{\link[=spec.pgram]{spec.pgram()}}, to learn how to
remove means and trends from data.
For more control, the \code{spec} argument was
added to let users sidestep \code{\link[=spectrum]{spectrum()}} entirely, by providing
their own spectral computation functions.
}
}

\examples{
library(oce)
Fs <- 1000
t <- seq(0, 0.296, 1 / Fs)
x <- cos(2 * pi * t * 200) + rnorm(n = length(t))
X <- ts(x, frequency = Fs)
s <- spectrum(X, spans = c(3, 2), main = "random + 200 Hz", log = "no")
w <- pwelch(X, plot = FALSE)
lines(w$freq, w$spec, col = "red")
w2 <- pwelch(X, nfft = 75, plot = FALSE)
lines(w2$freq, w2$spec, col = "green")
abline(v = 200, col = "blue", lty = "dotted")
cat("Checking spectral levels with Parseval's theorem:\n")
cat("var(x)                              = ", var(x), "\n")
cat("2 * sum(s$spec) * diff(s$freq[1:2]) = ", 2 * sum(s$spec) * diff(s$freq[1:2]), "\n")
cat("sum(w$spec) * diff(s$freq[1:2])     = ", sum(w$spec) * diff(w$freq[1:2]), "\n")
cat("sum(w2$spec) * diff(s$freq[1:2])    = ", sum(w2$spec) * diff(w2$freq[1:2]), "\n")
# co2
par(mar = c(3, 3, 2, 1), mgp = c(2, 0.7, 0))
s <- spectrum(co2, plot = FALSE)
plot(log10(s$freq), s$spec * s$freq,
    xlab = expression(log[10] * Frequency), ylab = "Power*Frequency", type = "l"
)
title("Variance-preserving spectrum")
pw <- pwelch(co2, nfft = 256, plot = FALSE)
lines(log10(pw$freq), pw$spec * pw$freq, col = "red")

}
\references{
Welch, P. D., 1967. The Use of Fast Fourier Transform for the
Estimation of Power Spectra: A Method Based on Time Averaging Over Short,
Modified Periodograms. \emph{IEEE Transactions on Audio Electroacoustics},
AU-15, 70--73.
}
\author{
Dan Kelley
}
