% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/section.R
\name{plot,section-method}
\alias{plot,section-method}
\alias{plot.section}
\title{Plot a section Object}
\usage{
\S4method{plot}{section}(
  x,
  which = c(1, 2, 3, 99),
  eos,
  at = NULL,
  labels = TRUE,
  grid = FALSE,
  contourLevels = NULL,
  contourLabels = NULL,
  stationIndices,
  coastline = "best",
  xlim = NULL,
  ylim = NULL,
  zlim = NULL,
  zbreaks = NULL,
  zcol = NULL,
  map.xlim = NULL,
  map.ylim = NULL,
  clongitude,
  clatitude,
  span,
  projection = NULL,
  xtype = "distance",
  ytype = "depth",
  ztype = "contour",
  longitude0,
  latitude0,
  legend.loc = "bottomright",
  legend.text = NULL,
  showStations = FALSE,
  showStart = TRUE,
  stationTicks = TRUE,
  showBottom = TRUE,
  showSpine = TRUE,
  drawPalette = TRUE,
  axes = TRUE,
  mgp,
  mar,
  col,
  cex,
  pch,
  labcex = 1,
  debug,
  ...
)
}
\arguments{
\item{x}{a \linkS4class{section} object.}

\item{which}{a list of desired plot types, as explained in \dQuote{Details}.
There may be up to four panels in total, and the desired plots are placed in
these panels, in reading order.  If only one panel is plotted, \code{par} is
not adjusted, which makes it easy to add to the plot with subsequent plotting
commands.}

\item{eos}{Character indication of the seawater equation of state
to use. The permitted choices are \code{"gsw"} and \code{"unesco"}. If \code{eos} is
not supplied, it defaults to \code{\link{getOption}}\verb{("oceEOS",default="gsw")}.}

\item{at}{If \code{NULL} (the default), the x axis will indicate the distance
of the stations from the first in the section.  (This may give errors in the
contouring routine, if the stations are not present in a geographical order.)
If a list, then it indicates the values at which stations will be plotted.}

\item{labels}{Either a logical, indicating whether to put labels on the x axis,
or a vector that is a list of labels to be placed at the x positions indicated
by \code{at}.}

\item{grid}{If \code{TRUE}, points are drawn at data locations.}

\item{contourLevels}{Optional contour levels.}

\item{contourLabels}{Optional contour labels.}

\item{stationIndices}{Optional list of the indices of stations to use.  Note
that an index is \emph{not} a station number, e.g. to show the first 4
stations, use \code{station.indices=1:4}.}

\item{coastline}{Either a \linkS4class{coastline} object to be used,
or a string.  In the second case, the permitted
choices are \code{"best"} (the default) to pick
a variant that suits the scale, \code{"coastlineWorld"} for the coarse
version that is provided by \CRANpkg{oce},
\code{"coastlineWorldMedium"} or \code{"coastlineWorldFine"} for two
coastlines provided by the \CRANpkg{ocedata} package, or \code{"none"}, to avoid
drawing a coastline.}

\item{xlim}{Optional limit for x axis (only in sections, not map).}

\item{ylim}{Optional limit for y axis (only in sections, not map)}

\item{zlim, zbreaks, zcol}{Elements that control colours for \code{image} and \code{points}
plot types, i.e. if \code{ztype} is either \code{"points"} or \code{"image"}.
\code{zlim} is a two-element numerical vector specifying the
limit on the plotted field.  If not provided, it defaults to the data
range.
\code{zbreaks} controls the colour breaks, in a manner that is similar to
the \code{\link[=image]{image()}} parameter named \code{breaks}.  If not provided, \code{zbreaks} is
inferred from \code{zlim}.
\code{zcol}, which controls the colour scheme, may be a vector of colours
(of length 1 less than \code{zbreaks}), or a function that takes an
integer as its sole argument and returns that number of colours.
If not provided, \code{zcol} defaults to \code{\link[=oceColorsViridis]{oceColorsViridis()}}.
These three parameters are used in Example 6, an illustration of
Atlantic salinity along 36N.}

\item{map.xlim, map.ylim}{Optional limits for station map; \code{map.ylim} is
ignored if \code{map.xlim} is provided.}

\item{clongitude, clatitude, span}{Optional map centre position and span (km).}

\item{projection}{Parameter specifying map
projection; see \code{\link[=mapPlot]{mapPlot()}}.  If \code{projection="automatic"},
however, a projection is devised from the data, with \code{stereographic} if
the mean latitude exceeds 70N and \code{mollweide} otherwise.}

\item{xtype}{Type of x axis, for contour plots, either \code{"distance"} for
distance (in km) to the first point in the section, \code{"track"} for distance
along the cruise track, \code{"longitude"}, \code{"latitude"},
\code{"time"} or \code{"spine"} (distance along a spine that was added
with \code{\link[=addSpine]{addSpine()}}).  Note that if the x values are not in order, they will be put in
order, and since that might not make physical sense, a warning will be issued.}

\item{ytype}{Type of y axis for contour plots, either \code{"pressure"} for
pressure (in dbar, with zero at the surface) or \code{"depth"} for depth (in m
below the surface, calculated from pressure with \code{\link[=swDepth]{swDepth()}}).}

\item{ztype}{String indicating whether to how to indicate the "z"
data (in the R sense, i.e. this could be salinity, temperature, etc; it does
not mean the vertical coordinate) The choices are: \code{"contour"} for
contours, \code{"image"} for an image (drawn with \code{\link[=imagep]{imagep()}} with
\code{filledContours=TRUE}), or \code{"points"} to draw points.
In the first two cases, the data must be gridded, with identical pressures at
each station.}

\item{longitude0, latitude0}{Location of the point from which distance is measured.
These values are ignored unless \code{xtype} is \code{"distance"}.}

\item{legend.loc}{Location of legend, as supplied to \code{\link[=legend]{legend()}}, or
set to the empty string to avoid plotting a legend.}

\item{legend.text}{character value indicating the text for the legend.
If this is NULL (the default) then the legend is automatically
constructed by \code{\link[=labelWithUnit]{labelWithUnit()}}, based on the value of \code{which}.}

\item{showStations}{Logical indicating whether to draw station numbers on maps.}

\item{showStart}{Logical indicating whether to indicate the first station with}

\item{stationTicks}{A logical value indicating whether to indicate station
locations with ticks at the top margin of cross-section plots. Setting this
parameter to \code{FALSE} frees the user up to do their own labelling
at this spot.}

\item{showBottom}{a value indicating whether (and how) to indicate the
ocean bottom on cross-section views.  There are three possibilities.
(a) If \code{showBottom} is \code{FALSE}, then the bottom is not rendered.  If it
is \code{TRUE}, then the  bottom is rendered with a gray polygon.
(b) If \code{showBottom} is the character value \code{"polygon"}, then a polygon is drawn,
and similarly lines are drawn for \code{"lines"}, and points for \code{"points"}.
(c) If \code{showBottom} is a \linkS4class{topo} object, then the station locations are
interpolated to that topography and the results are shown with a polygon.
See \dQuote{Examples}.}

\item{showSpine}{logical value used if \code{which="map"}.  If \code{showSpine} is
\code{TRUE} and \code{section} has had a spine added with \code{\link[=addSpine]{addSpine()}}, then
the spine is drawn in blue.}

\item{drawPalette}{Logical value indicating whether to draw a palette when \code{ztype="image"}
ignored otherwise.}

\item{axes}{Logical value indicating whether to draw axes.}

\item{mgp}{A 3-element numerical vector to use for \code{par(mgp)}, and also for
\code{par(mar)}, computed from this. If not provided, this defaults to
\code{\link{getOption}}\code{("oceMgp")}.}

\item{mar}{Value to be used with \code{\link{par}}\code{("mar")}. If not provided,
a default is set up.}

\item{col}{Color for line types.  If not provided, this defaults to
\code{\link{par}}\code{("col")}.  See \code{zcol}, for \code{ztype="image"} and \code{ztype="points"}.}

\item{cex}{Numerical character-expansion factor, which defaults to \code{\link{par}}\code{("cex")}.}

\item{pch}{Indication of symbol type; defaults to \code{\link{par}}\code{("pch")}.}

\item{labcex}{Size of characters in contour labels (passed to
\code{\link[=contour]{contour()}}).}

\item{debug}{an integer specifying whether debugging information is
to be printed during the processing. This is a general parameter that
is used by many \code{oce} functions. Generally, setting \code{debug=0}
turns off the printing, while higher values suggest that more information
be printed. If \code{debug} is not supplied, it defaults to
\code{\link{getOption}}\code{("oceDebug")}.}

\item{...}{Optional arguments passed to the contouring function.}
}
\value{
If the original section was gridded, the return value is that section.
Otherwise, the gridded section that was constructed for the plot is returned.
In both cases, the value is returned silently. The
purpose of returning the section is to enable subsequent processing
of the grid, including adding elements to the plot (see example 5).
}
\description{
Creates a summary plot for a CTD section, with one panel for each value of
\code{which}.
}
\details{
The type of plot is governed by \code{which}, as follows.
\itemize{
\item \code{which=0} or \code{"potential temperature"} for potential temperature contours
\item \code{which=1} or \code{"temperature"} for in-situ temperature contours (the default)
\item \code{which=2} or \code{"salinity"} for salinity contours
\item \code{which=3} or \code{"sigmaTheta"} for sigma-theta contours
\item \code{which=4} or \code{"nitrate"} for nitrate concentration contours
\item \code{which=5} or \code{"nitrite"} for nitrite concentration contours
\item \code{which=6} or \code{"oxygen"} for oxygen concentration  contours
\item \code{which=7} or \code{"phosphate"} for phosphate concentration contours
\item \code{which=8} or \code{"silicate"} for silicate concentration contours
\item \code{which=9} or \code{"u"} for eastward velocity
\item \code{which=10} or \code{"uz"} for vertical derivative of eastward velocity
\item \code{which=11} or \code{"v"} for northward velocity
\item \code{which=12} or \code{"vz"} for vertical derivative of northward velocity
\item \code{which=20} or \code{"data"} for a dot for each data location
\item \code{which=99} or \code{"map"} for a location map
}

The y-axis for the contours is pressure, plotted in the conventional reversed
form, so that the water surface appears at the top of the plot.  The x-axis is
more complicated. If \code{at} is not supplied, then the routine calculates x
as the distance between the first station in the section and each of the other
stations. (This will produce an error if the stations are not ordered
geographically, because the \code{\link[=contour]{contour()}} routine cannot handle
non-increasing axis coordinates.) If \code{at} is specified, then it is taken
to be the location, in arbitrary units, along the x-axis of labels specified by
\code{labels}; the way this works is designed to be the same as for
\code{\link[=axis]{axis()}}.
}
\section{Ancillary Examples}{


The following examples were once part of the \dQuote{Examples}
section, but were moved here in May 2022, to reduce the build-check
time for CRAN submission.\preformatted{library(oce)
data(section)
GS <- subset(section, 113<=stationId&stationId<=129)
GSg <- sectionGrid(GS, p=seq(0, 2000, 100))

# Gulf Stream, salinity data and contoured
par(mfrow=c(2, 1))
plot(GS, which=1, ylim=c(2000, 0), ztype="points",
     zbreaks=seq(0,30,2), pch=20, cex=3)
plot(GSg, which=1, ztype="image", zbreaks=seq(0,30,2))

# Gulf Stream, temperature grid (image) and data (dots)
par(mfrow=c(1, 1))
plot(GSg, which=1, ztype="image")
T <- GS[["temperature"]]
col <- oceColorsViridis(100)[rescale(T, rlow=1, rhigh=100)]
points(GS[["distance"]],GS[["depth"]],pch=20,cex=3,col="white")
points(GS[["distance"]],GS[["depth"]],pch=20,cex=2.5,col=col)

# 4. Image of temperature, with a high-salinity contour on top;
#    note the Mediterranean water.
sec <- plot(section, which="temperature", ztype="image")
S <- sec[["salinity", "grid:distance-pressure"]]
contour(S$distance, S$pressure, S$field, level=35.8, lwd=3, add=TRUE)

# 5. Contours of salinity, with dots for high pressure and spice
plot(section, which="salinity")
distance <- section[["distance"]]
depth <- section[["depth"]]
spice <- section[["spice"]]
look <- spice > 1.8 & depth > 500
points(distance[look], depth[look], col="red")

# Image of Absolute Salinity, with 4-minute bathymetry
# It's easy to calculate the desired area for the bathymetry,
# but for brevity we'll hard-code it. Note that download.topo()
# requires the "raster" and "ncdf4" packages to be installed.
if (requireNamespace("raster") && requireNamespace("ncdf4")) \{
    f <- download.topo(west=-80, east=0, south=35, north=40, resolution=4)
    t <- read.topo(f)
    plot(section, which="SA", xtype="longitude", ztype="image", showBottom=t)
\}

# Temperature with salinity added in red
plot(GSg, which="temperature")
distance <- GSg[["distance", "byStation"]]
depth <- GSg[["station", 1]][["depth"]]
S <- matrix(GSg[["salinity"]], byrow=TRUE, nrow=length(GSg[["station"]]))
contour(distance, depth, S, col=2, add=TRUE)

# Image with controlled colours
plot(GSg, which="salinity", ztype="image",
    zlim=c(35, 37.5),
    zbreaks=seq(35, 37.5, 0.25),
    zcol=oceColorsTurbo)
}
}

\examples{
library(oce)
data(section)
GS <- subset(section, 113<=stationId&stationId<=129)
GSg <- sectionGrid(GS, p=seq(0, 2000, 100))

# Gulf Stream, salinity and temperature contours
plot(GSg, which=c("salinity", "temperature"))

# Gulf Stream, Temperature image
plot(GSg, which="temperature", ztype="image",
    zbreaks=seq(0,25,2), zcol=oceColorsTemperature)

}
\seealso{
The documentation for \linkS4class{section} explains the
structure of section objects, and also outlines the other functions dealing
with them.

Other functions that plot oce data: 
\code{\link{download.amsr}()},
\code{\link{plot,adp-method}},
\code{\link{plot,adv-method}},
\code{\link{plot,amsr-method}},
\code{\link{plot,argo-method}},
\code{\link{plot,bremen-method}},
\code{\link{plot,cm-method}},
\code{\link{plot,coastline-method}},
\code{\link{plot,ctd-method}},
\code{\link{plot,gps-method}},
\code{\link{plot,ladp-method}},
\code{\link{plot,landsat-method}},
\code{\link{plot,lisst-method}},
\code{\link{plot,lobo-method}},
\code{\link{plot,met-method}},
\code{\link{plot,odf-method}},
\code{\link{plot,rsk-method}},
\code{\link{plot,satellite-method}},
\code{\link{plot,sealevel-method}},
\code{\link{plot,tidem-method}},
\code{\link{plot,topo-method}},
\code{\link{plot,windrose-method}},
\code{\link{plot,xbt-method}},
\code{\link{plotProfile}()},
\code{\link{plotScan}()},
\code{\link{plotTS}()},
\code{\link{tidem-class}}

Other things related to section data: 
\code{\link{[[,section-method}},
\code{\link{[[<-,section-method}},
\code{\link{as.section}()},
\code{\link{handleFlags,section-method}},
\code{\link{initializeFlagScheme,section-method}},
\code{\link{read.section}()},
\code{\link{section-class}},
\code{\link{sectionAddStation}()},
\code{\link{sectionGrid}()},
\code{\link{sectionSmooth}()},
\code{\link{sectionSort}()},
\code{\link{section}},
\code{\link{subset,section-method}},
\code{\link{summary,section-method}}
}
\author{
Dan Kelley
}
\concept{functions that plot oce data}
\concept{things related to section data}
