% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/nscancor.R
\name{nscancor}
\alias{nscancor}
\title{Non-Negative and Sparse CCA}
\usage{
nscancor(x, y, xcenter = TRUE, ycenter = TRUE, xscale = FALSE,
  yscale = FALSE, nvar = min(dim(x), dim(y)), xpredict, ypredict,
  cor_tol = NULL, nrestart = 10, iter_tol = 0, iter_max = 50,
  partial_model = NULL, verbosity = 0)
}
\arguments{
\item{x}{a numeric matrix which provides the data from the first domain}

\item{y}{a numeric matrix which provides the data from the second domain}

\item{xcenter}{a logical value indicating whether the empirical mean of (each
column of) \code{x} should be subtracted. Alternatively, a vector of length
equal to the number of columns of \code{x} can be supplied. The value is
passed to \code{\link{scale}}.}

\item{ycenter}{analogous to \code{xcenter}}

\item{xscale}{a logical value indicating whether the columns of \code{x}
should be scaled to have unit variance before the analysis takes place. The
default is \code{FALSE} for consistency with \code{cancor}. Alternatively,
a vector of length equal to the number of columns of \code{x} can be
supplied. The value is passed to \code{\link{scale}}.}

\item{yscale}{analogous to \code{xscale}}

\item{nvar}{the number of canonical variables to be computed for each domain.
With the default setting, canonical variables are computed until either
\code{x} or \code{y} is fully deflated.}

\item{xpredict}{the regression function to predict the canonical variable for
\code{x}, given \code{y}. The formal arguments are the design matrix
\code{y}, the regression target \code{xc} as the current canonical variable
for \code{x}, and \code{cc} as a counter of the current pair of canonical
variables (e.g. for enforcing different constraints for different canonical
vectors). See the examples for an illustration.}

\item{ypredict}{analogous to \code{xpredict}}

\item{cor_tol}{a threshold indicating the magnitude below which canonical
variables should be omitted. Variables are omitted if their explained
correlations are less than or equal to \code{cor_tol} times the correlation
of the first pair of canonical variables. With the default \code{NULL}
setting, no variables are omitted.}

\item{nrestart}{the number of random restarts for computing the canonical
variables via iterated regression steps. The solution achieving maximum
explained correlation over all random restarts is kept. A value greater
than one can help to avoid poor local maxima.}

\item{iter_tol}{If the relative change of the objective is less than
\code{iter_tol} between iterations, the procedure is asssumed to have
converged to a local optimum.}

\item{iter_max}{the maximum number of iterations to be performed. The
procedure is terminated if either the \code{iter_tol} or the
\code{iter_max} criterion is satisfied.}

\item{partial_model}{\code{NULL} or an object of class \code{nscancor}. The
computation can be continued from a partial model by providing an
\code{nscancor} object (either from a previous run of this function or from
\code{\link{acor}}) and setting \code{nvar} to a value greater than the
number of canonical variables contained in the partial model. See the
examples for an illustration.}

\item{verbosity}{an integer specifying the verbosity level. Greater values
result in more output, the default is to be quiet.}
}
\value{
\code{nscancor} returns a list of class \code{nscancor} containing
  the following elements: \item{cor}{the additional correlation explained by
  each pair of canonical variables, see \code{\link{acor}}.} \item{xcoef}{the
  matrix containing the canonical vectors related to \code{x} as its columns}
  \item{ycoef}{analogous to \code{xcoef}} \item{xcenter}{if \code{xcenter} is
  \code{TRUE} the centering vector, else the zero vector (in accordance with
  \code{cancor})} \item{ycenter}{analogous to \code{xcenter}}
  \item{xscale}{if \code{xscale} is \code{TRUE} the scaling vector, else
  FALSE } \item{yscale}{analogous to \code{xscale}} \item{xp}{the deflated
  data matrix corresponding to \code{x}} \item{yp}{anologous to \code{xp}}
}
\description{
Performs a canonical correlation analysis (CCA) where constraints such as
non-negativity or  sparsity are enforced on the canonical vectors. The result
of the analysis is returned as a list of class \code{nscancor}, which
contains a superset of the elements returned by \code{\link{cancor}}.
}
\details{
\code{nscancor} computes the canonical vectors (called \code{xcoef} and
\code{ycoef}) using iterated regression steps, where the constraints suitable
for each domain are enforced by choosing the avvropriate regression method.
See Sigg et al. (2007) for an early application of the principle (not yet
including generalized deflation).

Because constrained canonical vectors no longer correspond to true
eigenvectors of the cross-covariance matrix and are usually not pairwise
conjugate (i.e. the canonical variables are not uncorrelated), special
attention needs to be paid when computing more than a single pair of
canonical vectors. \code{nscancor} implements a generalized deflation (GD)
scheme which builds on GD for PCA as proposed by Mackey (2009). For each
domain, a basis of the space spanned by the previous canonical variables is
computed. Then, the correlation of the current pair of canonical variables is
maximized after projecting each current canonical vector to the
ortho-complement space of its respective basis. This procedure maximizes the
additional correlation not explained by previous canonical variables, and is
identical to standard CCA if the canonical vectors are the eigenvectors of
the cross-covariance matrix.

See the references for further details.
}
\examples{
\donttest{
if (requireNamespace("glmnet", quietly = TRUE) &&
    requireNamespace("MASS", quietly = TRUE) &&
    requireNamespace("CCA", quietly = TRUE)) {

  data(nutrimouse, package = "CCA")

  set.seed(1)

  ###
  # Unconstrained CCA, produces results close to calling
  # cancor(nutrimouse$gene[ , 1:5], nutrimouse$lipid)

  ypredict <- function(x, yc, cc) {
    return(MASS::ginv(x)\%*\%yc)
  }
  xpredict <- function(y, xc, cc) {
    return(MASS::ginv(y)\%*\%xc)
  }
  nscancor(nutrimouse$gene[ , 1:5], nutrimouse$lipid, xpredict = xpredict,
           ypredict = ypredict)


  ###
  # Non-negative sparse CCA using glmnet() as the regression function, where
  # different regularisers are enforced on the different data domains and pairs
  # of canonical variables.

  dfmax_w <- c(40, 15, 10)
  ypredict <- function(x, yc, cc) {
    en <- glmnet::glmnet(x, yc, alpha = 0.5, intercept = FALSE,
                         dfmax = dfmax_w[cc], lower.limits = 0)
    W <- coef(en)
    return(W[2:nrow(W), ncol(W)])
  }
  dfmax_v <- c(7, 5, 5)
  xpredict <- function(y, xc, cc) {
    en <- glmnet::glmnet(y, xc, alpha = 0.5, intercept = FALSE,
                         dfmax = dfmax_v[cc])
    V <- coef(en)
    return(V[2:nrow(V), ncol(V)])
  }
  nscc <- nscancor(nutrimouse$gene, nutrimouse$lipid, nvar = 2,
                   xpredict = xpredict, ypredict = ypredict)

  # continue the computation of canonical variables from a partial model
  nscc <- nscancor(nutrimouse$gene, nutrimouse$lipid, nvar = 3,
                   xpredict = xpredict, ypredict = ypredict,
                   partial_model = nscc)
}
}
}
\references{
Sigg, C. and Fischer, B. and Ommer, B. and Roth, V. and Buhmann,
  J. (2007) Nonnegative CCA for Audiovisual Source Separation. In
  \emph{Proceedings of the 2007 IEEE Workshop on Machine Learning for Signal
  Processing} (vv. 253--258).

Mackey, L. (2009) Deflation Methods for Sparse PCA. In
  \emph{Advances in Neural Information Processing Systems} (vv. 1017--1024).
}
\seealso{
\code{\link{acor}}, \code{\link{cancor}}, \code{\link{scale}}
}
