#' @title Print netrankr_full object to terminal
#' @description Prints the result of an object obtained from [exact_rank_prob] to terminal
#' @param x A netrank object 
#' @param ... additional arguments to print
#' @author David Schoch
#' @export
print.netrankr_full <- function(x,...){
  
  if(!"netrankr_full"%in% class(x)){
    stop("x is not a netrankr_full object")
  }
  # maxpr  <- getOption("width")
  n <- length(x$mse)
  cat("Number of possible centrality rankings: ",x$lin.ext,"\n")
  cat("Equivalence Classes (max. possible): ",length(unique(x$mse))," (",n,")\n",sep = "")
  cat(rep("-",10),"\n")
  cat("Rank Probabilities (rows:nodes/cols:ranks)\n")
  print(x$rank.prob)
  cat(rep("-",10),"\n")
  cat("Relative Rank Probabilities (row ranked lower than col)\n")
  print(x$relative.rank)
  cat(rep("-",10),"\n")
  cat("Expected Ranks (higher values are better)\n")
  print(x$expected.rank)
  cat(rep("-",10),"\n")
  cat("SD of Rank Probabilities\n")
  print(x$rank.spread)
  cat(rep("-",10),"\n")
  if(!is.null(x$topo.order)){
    cat("(no printing method available for intermediate data structures)")
  }
  invisible(x)
}

#' @title Plot netrankr_full object
#' @importFrom grDevices hcl.colors
#' @importFrom graphics arrows barplot image legend par points
#' @importFrom utils stack
#' @description Plots the result of an object obtained from [exact_rank_prob]
#' @param x A netrankr_full object
#' @param icols a list of colors such as that generated by [hcl.colors], [gray.colors] or similar functions for the heatmaps
#' @param bcol color used for the barcharts
#' @param ecol color used for errorbars
#' @param ... additional plot parameters
#' @author David Schoch
#' @export
plot.netrankr_full <- function(x,icols = hcl.colors(12, "viridis"),bcol="grey66",ecol="black",...){
  op <- par(mfrow = c(2, 2))
  
  rk_pr <- x$rank.prob
  rk_pr[rk_pr==0] <- NA
  rk_rl <- x$relative.rank
  rk_rl[rk_rl==0] <- NA
  image(x=1:ncol(rk_pr),y=1:nrow(rk_pr),t(rk_pr),col=icols,xlab="ranks",
        ylab="nodes",tck=0,breaks = seq(0,1,length.out=length(icols)+1),main="rank probabilities")
  image(x=1:ncol(rk_rl),y=1:nrow(rk_rl),t(rk_rl),col=icols,xlab="",
        ylab="",tck=0,breaks = seq(0,1,length.out=length(icols)+1),main="relative rank probabilities")
  mid <- barplot(x$expected.rank,main="expected ranks",col=bcol,ylim = c(0,ncol(rk_pr)))
  sd_min <- pmax(x$expected.rank-x$rank.spread,0)
  sd_max <- pmin(x$expected.rank+x$rank.spread,ncol(rk_pr))
  idx <- !(sd_min==sd_max)
  barplot(x$expected.rank,main="expected ranks and spread",col=bcol,ylim = c(0,ncol(rk_pr)))
  arrows(x0=mid[idx], y0=sd_min[idx], x1=mid[idx], y1=sd_max[idx], code=3, angle=90, length=0.1,col=ecol)
  par(op)
  invisible(NULL)
}

#' @title Print netrankr_interval object to terminal
#' @description Prints the result of an object obtained from [rank_intervals] to terminal
#' @param x A netrankr_interval object 
#' @param ... additional arguments to print
#' @author David Schoch
#' @export
print.netrankr_interval <- function(x,...){
  if(!"netrankr_interval"%in% class(x)){
    stop("x is not a netrankr_interval object")
  }
  prows <- min(c(nrow(x),getOption("max.print")))
  int_string <- paste0("node:",x$node," rank interval: [",x$min_rank,", ",x$max_rank,"] mid point: ",x$mid_point,"\n",sep="")
  int_string <- int_string[1:prows]
  cat(int_string)
}

#' @title plot netrankr_interval objects
#' @importFrom graphics arrows legend par points
#' @importFrom utils stack
#' @description Plots results from [rank_intervals] 
#' @param x A netrank object 
#' @param cent_scores A data frame containing centrality scores of indices (optional)
#' @param ties.method how to treat ties in the rankings. see [rank] for details
#' @param ... additional arguments to plot
#' @author David Schoch
#' @export
plot.netrankr_interval <- function(x,cent_scores = NULL,ties.method="min",...){
  if(!"netrankr_interval"%in% class(x)){
    stop("x is not a netrankr_interval object")
  }
  ord <- order(x$mid_point)
  if(!is.null(cent_scores)){
    op <- par(mar = c(4, 4, 6, 4),xpd = TRUE)
    cent_rk_wide <- as.data.frame(apply(cent_scores[ord,],2,rank,ties.method=ties.method))
    cent_rk_long <- cbind(id=(1:nrow(x)),stack(cent_rk_wide))
    
  }
  plot(x=factor(x$node,levels = x$node[ord]),y=x$mid_point,pch=0,ylim=c(0,nrow(x)),ylab="ranks",xlab="",...)
  idx <- !(x$min_rank[ord]==x$max_rank[ord])
  arrows(x0=(1:nrow(x))[idx], 
         y0=x$min_rank[ord[idx]], 
         x1=(1:nrow(x))[idx], 
         y1=x$max_rank[ord[idx]], code=3, angle=90, length=0.1,col="black")
  if(!is.null(cent_scores)){
    points(x=jitter(cent_rk_long$id,0.5),y=cent_rk_long$values,col=cent_rk_long$ind,pch=20,cex=1.3)
    legend("top", inset = c(0, -0.15),bty = "n",horiz = TRUE,
           legend=names(cent_rk_wide),
           col=1:ncol(cent_rk_wide),pch=20 )
    par(op)
  }
  invisible(NULL)
}


#' @title Print netrankr_mcmc object to terminal
#' @description Prints the result of an object obtained from [mcmc_rank_prob] to terminal
#' @param x A netrank object 
#' @param ... additional arguments to print
#' @author David Schoch
#' @export
print.netrankr_mcmc <- function(x,...){
  
  if(!"netrankr_mcmc"%in% class(x)){
    stop("x is not a netrankr_mcmc object")
  }
  # maxpr  <- getOption("width")
  cat("MCMC approximated Relative Rank Probabilities (row ranked lower than col)\n")
  print(x$relative.rank)
  cat(rep("-",10),"\n")
  cat("MCMC approximated Expected Ranks (higher values are better)\n")
  print(x$expected.rank)
  cat(rep("-",10),"\n")
  invisible(x)
}

#' @title Plot netrankr_mcmc object
#' @importFrom grDevices hcl.colors
#' @importFrom graphics barplot image par
#' @description Plots the result of an object obtained from [mcmc_rank_prob]
#' @param x A netrankr_mcmc object
#' @param icols a list of colors such as that generated by [hcl.colors], [gray.colors] or similar functions for the heatmaps
#' @param bcol color used for the barcharts
#' @param ... additional plot parameters
#' @author David Schoch
#' @export
plot.netrankr_mcmc <- function(x,icols = hcl.colors(12, "viridis"),bcol="grey66",...){
  op <- par(mfrow = c(1, 2))
  
  rk_rl <- x$relative.rank
  rk_rl[rk_rl==0] <- NA
  image(x=1:ncol(rk_rl),y=1:nrow(rk_rl),t(rk_rl),col=icols,xlab="",
        ylab="",tck=0,breaks = seq(0,1,length.out=length(icols)+1),
        main="MCMC approximated relative rank probabilities")
  barplot(x$expected.rank,main="MCMC approximated expected ranks",
          col=bcol,ylim = c(0,ncol(rk_rl)))
  par(op)
  invisible(NULL)
}