% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/makeAinv.R, R/makeGGAinv.R
\name{makeAinv}
\alias{makeAinv}
\alias{makeAinv.default}
\alias{makeAinv.fuzzy}
\alias{makeGGAinv}
\title{Creates the inverse additive genetic relationship matrix}
\usage{
makeAinv(
  pedigree,
  f = NULL,
  ggroups = NULL,
  fuzz = NULL,
  gOnTop = FALSE,
  det = TRUE,
  ...
)

\method{makeAinv}{default}(
  pedigree,
  f = NULL,
  ggroups = NULL,
  fuzz = NULL,
  gOnTop = FALSE,
  det = TRUE,
  ...
)

\method{makeAinv}{fuzzy}(
  pedigree,
  f = NULL,
  ggroups = NULL,
  fuzz,
  gOnTop = FALSE,
  det = TRUE,
  ...
)

makeGGAinv(pedigree, f = NULL, ggroups = NULL, det = TRUE, ...)
}
\arguments{
\item{pedigree}{A pedigree where the columns are ordered ID, Dam, Sire}

\item{f}{A numeric indicating the level of inbreeding. See Details}

\item{ggroups}{Either a vector with the unique name of each genetic group,
or a numeric indicating the number of unique genetic groups. See Details 
for different ways to specify. Note, if NULL then the regular A-inverse
will be constructed. Also, must be NULL if fuzz is non-NULL.}

\item{fuzz}{A matrix containing the fuzzy classification of phantom parents
into genetic groups. See Details.}

\item{gOnTop}{A logical indicating if (when including genetic groups) the
A-inverse should be constructed with the \sQuote{g} genetic groups located 
in the first \sQuote{g} rows and columns if \code{TRUE}, else the
\sQuote{g} genetic groups are located in the last \sQuote{g} rows and 
columns of A-inverse}

\item{det}{Logical, indicating if the (log) determinant of the A matrix
should be returned}

\item{\dots}{Arguments to be passed to methods}
}
\value{
a \code{list}:
  \describe{
    \item{Ainv }{the inverse of the additive genetic relationship matrix
      in sparse matrix form}
    \item{listAinv }{the three column list of the non-zero elements for the 
      inverse of the additive genetic relationship matrix with attributes
      \code{rowNames} and \code{geneticGroups}. \code{attr(*, "rowNames")}
      links the integer for rows/columns to the ID column from the pedigree. 
      \code{attr(*, "geneticGroups")} is a two element vector with the first 
      integer indicating how many genetic groups are included in the 
      pedigree. This last attribute is necessary for some software programs 
      to correctly specify the residual degrees of freedom when calculating 
      the log-likelihood in a model that implicitly fits fixed genetic group 
      effects.}
    \item{f }{the individual coefficients of inbreeding for each individual 
      in the pedigree (matches the order of the first/ID column of the
      pedigree). If the pedigree contains \sQuote{g} genetic groups in the 
      first \sQuote{g} rows, then the first \sQuote{g} elements of \code{f} 
      are assigned 0. If the pedigree contains \sQuote{p} phantom parents in 
      the first \sQuote{p} rows, then the first \sQuote{p} elements of
      \code{f} are assigned 0.}
    \item{logDet }{the log determinant of the A matrix}
    \item{dii }{the (non-zero) elements of the diagonal D matrix of the A=TDT'
      decomposition. Contains the variance of Mendelian sampling. Matches
      the order of the first/ID column of the pedigree. If the pedigree
      contains \sQuote{g} genetic groups in the first \sQuote{g} rows, then
      the first \sQuote{g} elements of \code{f} are assigned 0. If the
      pedigree contains \sQuote{p} phantom parents in the first \sQuote{p}
      rows, then the first \sQuote{p} elements of \code{f} are assigned 0.} 
  }
}
\description{
This returns the inverse of the numerator relationship matrix (inverse
additive genetic relatedness matrix). It can also be used to obtain
coefficients of inbreeding for the pedigreed population.
}
\details{
Missing parents (e.g., base population) should be denoted by either 'NA',
'0', or '*'.

The functions implement an adaptation of the Meuwissen and Luo (1992)
algorithm (particularly, following the description of the algorithm in
Mrode 2005) with some code borrowed from the \code{inverseA} function by
Jarrod Hadfield in the \code{MCMCglmm} package. Further, providing a
non-NULL argument to \code{ggroups} incorporates the Quaas (1988) algorithm
for directly obtaining the augmented A-inverse matrix for genetic groups
into Meuwissen and Luo's (1992) algorithm, thereby, considering inbreeding
during the construction of the A-inverse. Further calculations needed for
the algorithm to incorporate inbreeding and genetic groups follow the theory
presented in VanRaden (1992). Alternatively, group-specific inverse
relatedness matrices can be formed with \code{makeGGAinv}, see below.

At the moment, providing the inbreeding level of individuals or the base
population has not been implemented. However, this argument is a placeholder
for now.

Genetic groups can be incorporated into a single A-inverse by providing a value
to the \code{ggroups} argument in \code{makeAinv}. The value supplied to
\code{ggroups} can either be (1) a single integer indicating the number of
unique genetic groups or (2) a character vector containing the name for each
genetic group. These are referred to as pedigree types "A" and "D",
respectively, and further details follow below.

(Type="A") the pedigree contains unique IDs for the 'g' genetic groups in the
first 'g' lines of the pedigree. The dam and sire of the genetic group rows
should contain missing values (e.g., NA, "0", or "*"). All individuals in the
pedigree should then have one of the `g' genetic groups instead of an unknown
parent.
(Type="D") the pedigree contains only individuals in the ID column (no
genetic groups have an ID) and there should be no missing values for any dams
or sires. Instead, individuals for whom the dam and/or sire is unknown should
have one of the genetic groups identified in the vector supplied to
\code{ggroups} as the dam or sire.

\sQuote{Fuzzy classification} of genetic groups (Fikse 2009) can be
implemented if a \sQuote{matrix} (of class \code{matrix} or \code{Matrix})
is supplied to the \code{fuzzy} argument. The fuzzy classification matrix
must have row names matching all of the phantom parents in the pedigree and
the column names must be present and specify the genetic groups. The fuzzy
classification matrix essentially contains probability of group membership
for each phantom parent. Therefore, each row should sum to 1. The pedigree
must have an identity in a unique row for every phantom parent and cannot
have genetic groups as either identities (in the first column) or as dam or
sire (second and third columns). Further, if fuzzy classification is
desired, the function must specify \code{ggroups = NULL}.

When genetic groups (including the case of fuzzy classification of genetic
groups) are included in the A-inverse matrix, the argument to \code{gOnTop}
specifies if the genetic group elements in the A-inverse should occupy the
top-left (\code{gOnTop = TRUE}) or bottom-right (\code{gOnTop = FALSE}) of
the matrix. Depending on how the software implementing an animal model
solves the mixed model equations, the equations for the genetic groups (and
thus the elements in the augmented A-inverse) should be the first or last
set of equations.
}
\examples{

 ##  Without genetic groups  ##
 makeAinv(Mrode2)
 
 ##  With genetic groups  ##
  ## Type A
   typeAped <- Q1988[-c(3:7), c("id", "damGG", "sireGG")]
   AstarA <- makeAinv(typeAped, ggroups = 2, gOnTop = FALSE)$Ainv
  ## Type D
   typeDped <- Q1988[-c(1:7), c("id", "damGG", "sireGG")]
   AstarD <- makeAinv(typeDped, ggroups = c("g1", "g2"), gOnTop = FALSE)$Ainv
  stopifnot(identical(AstarA, AstarD))
  
  # Show that the augmented A-inverse with genetic groups
  # contains the normal A-inverse (i.e., without genetic groups)
   ## Augmented A-inverse with genetic groups
    ggAinv <- makeAinv(Mrode3[-c(1,2), c("calf", "damGG", "sireGG")],
	ggroups = c("g1", "g2"), gOnTop = FALSE)$Ainv
    noggAinv <- makeAinv(Mrode3[-c(1,2), c("calf", "dam", "sire")],
	ggroups = NULL)$Ainv
    # First 8 rows & columns of ggAinv are same as A-inverse without 
    ## genetic groups
    ggAinv[1:8, 1:8]
    noggAinv
   stopifnot(all.equal(ggAinv[1:8, 1:8], noggAinv))
   
 ##  With fuzzy classification of genetic groups  ##
  ## example in Fikse (2009)
  Fped <- F2009[-c(1:3), c("id", "phantomDam", "phantomSire")]
    Fped$id <- factor(Fped$id, levels = as.character(unique(Fped$id)))
  Ffuzz <- as.matrix(F2009[4:10, c("g1", "g2", "g3")])
    dimnames(Ffuzz)[[1]] <- as.character(F2009[4:10, 1])
  AstarF <- makeAinv(Fped, fuzz = Ffuzz, gOnTop = FALSE)$Ainv

  ## Show that A-inverse with fuzzy classification of genetic groups
  ### can be the same as genetic group A-inverse without fuzzy classification
  ### Create a 'null' fuzzy classification matrix for Q1988 pedigree
  QfuzzNull <- matrix(c(1,0,0,1,0, 0,1,1,0,1), nrow = 5, ncol = 2,
	dimnames = list(letters[1:5], c("g1", "g2")))
  typeFped <- Q1988[-c(1:2), c("id", "phantomDam", "phantomSire")]
  AstarNullFuzzy <- makeAinv(typeFped, fuzz = QfuzzNull, gOnTop = FALSE)$Ainv
  # Same as above using either pedigree type 'A' or 'D'
  stopifnot(identical(AstarNullFuzzy, AstarA),
	identical(AstarNullFuzzy, AstarD))

 ##  With genetic groups  ##
  ## Type A
   typeAped <- Q1988[-c(3:7), c("id", "damGG", "sireGG")]
   (AinvOutA <- makeGGAinv(typeAped, ggroups = 2)$Ainv)
  ## Type D
   typeDped <- Q1988[-c(1:7), c("id", "damGG", "sireGG")]
   (AinvOutD <- makeGGAinv(typeDped, ggroups = c("g1", "g2"))$Ainv)
  stopifnot(identical(AinvOutA, AinvOutD))

}
\references{
Fikse, F. 2009. Fuzzy classification of phantom parent groups in
an animal model. Genetics Selection Evolution 41:42.

Meuwissen, T.H.E & Luo, Z. 1992. Computing inbreeding coefficients in large
populations. Genetics, Selection, Evolution. 24:305-313.

Mrode, R.A. 2005. Linear Models for the Prediction of Animal Breeding
Values, 2nd ed.  Cambridge, MA: CABI Publishing.

Quaas, R.L. 1988. Additive genetic model with groups and relationships.
Journal of Dairy Science. 71:1338-1345.

VanRaden, P.M. 1992. Accounting for inbreeding and crossbreeding in genetic
evaluation of large populations. Journal of Dairy Science. 75:3136-3144.
}
\seealso{
\code{\link{makeAstarMult}}, \code{\link{makeA}}
}
\author{
\email{matthewwolak@gmail.com}
}
