% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/multiview.R
\name{multiview}
\alias{multiview}
\title{Perform cooperative learning using the direct algorithm for
two or more views.}
\usage{
multiview(
  x_list,
  y,
  rho = 0,
  family = gaussian(),
  exclude = NULL,
  mvlambda = NULL,
  ...
)
}
\arguments{
\item{x_list}{a list of \code{x} matrices with same number of rows
\code{nobs}}

\item{y}{the quantitative response with length equal to \code{nobs}, the
(same) number of rows in each \code{x} matrix}

\item{rho}{the weight on the agreement penalty, default 0. \code{rho=0}
is a form of early fusion, and \code{rho=1} is a form of late fusion.
We recommend trying a few values of \code{rho} including 0, 0.1, 0.25, 0.5, and 1 first;
sometimes \code{rho} larger than 1 can also be helpful.}

\item{family}{A description of the error distribution and link
function to be used in the model. This is the result of a call to
a family function. Default is \link[stats:family]{stats::gaussian}. (See
\link[stats:family]{stats::family} for details on family functions.)}

\item{exclude}{Indices of variables to be excluded from the
model. Default is none. Equivalent to an infinite penalty factor
for the variables excluded (next item).  Users can supply instead
an \code{exclude} function that generates the list of indices.  This
function is most generally defined as \verb{function(x_list, y, ...)},
and is called inside \code{multiview} to generate the indices for
excluded variables.  The \code{...} argument is required, the others
are optional.  This is useful for filtering wide data, and works
correctly with \code{cv.glmnet}.}

\item{mvlambda}{A user supplied \code{lambda} sequence, default
\code{NULL}. Typical usage is to have the program compute its own
\code{mvlambda} sequence. This sequence, in general, is different from
that used in the \code{\link[glmnet:glmnet]{glmnet::glmnet()}} call (named \code{lambda})
Supplying a value of \code{mvlambda} overrides this. WARNING: use with
care. Avoid supplying a single value for \code{mvlambda} (for
predictions after CV use \code{\link[stats:predict]{stats::predict()}} instead.  Supply
instead a decreasing sequence of \code{mvlambda} values as \code{multiview}
relies on its warms starts for speed, and its often faster to fit
a whole path than compute a single fit.}

\item{...}{further arguments to glmnet, some of which may be
overridden as noted above}
}
\value{
An object with S3 class \verb{"multiview","*" }, where \code{"*"} is
\code{"elnet"}, \code{"lognet"}, \code{"multnet"}, \code{"fishnet"} (poisson),
\code{"coxnet"} or \code{"mrelnet"} for the various types of models.
\item{call}{the call that produced this object}
\item{a0}{Intercept sequence of length \code{length(lambda)}}
\item{beta}{For \code{"elnet"}, \code{"lognet"}, \code{"fishnet"} and \code{"coxnet"}
models, a \verb{nvars x length(lambda)} matrix of coefficients, stored
in sparse column format (\code{"CsparseMatrix"}). For \code{"multnet"} and
\code{"mgaussian"}, a list of \code{nc} such matrices, one for each class.}
\item{lambda}{The actual sequence of \code{\link[glmnet:glmnet]{glmnet::glmnet()}} \code{lambda} values used. When
\code{alpha=0}, the largest lambda reported does not quite give the
zero coefficients reported (\code{lambda=inf} would in principle).
Instead, the largest \code{lambda} for \code{alpha=0.001} is used, and the
sequence of \code{lambda} values is derived from this.}
\item{mvlambda}{The corresponding sequence of multiview lambda values}
\item{dev.ratio}{The fraction of (null) deviance explained (for
\code{"elnet"}, this is the R-square). The deviance calculations
incorporate weights if present in the model. The deviance is
defined to be 2*(loglike_sat - loglike), where loglike_sat is the
log-likelihood for the saturated model (a model with a free
parameter per observation). Hence dev.ratio=1-dev/nulldev.}
\item{nulldev}{Null deviance (per observation). This is defined
to be 2*(loglike_sat -loglike(Null)); The NULL model refers to
the intercept model, except for the Cox, where it is the 0
model.} \item{df}{The number of nonzero coefficients for each
value of \code{lambda}. For \code{"multnet"}, this is the number of
variables with a nonzero coefficient for \emph{any} class.}
\item{dfmat}{For \code{"multnet"} and \code{"mrelnet"} only. A matrix
consisting of the number of nonzero coefficients per class}
\item{dim}{dimension of coefficient matrix (ices)}
\item{nobs}{number of observations} \item{npasses}{total passes
over the data summed over all lambda values} \item{offset}{a
logical variable indicating whether an offset was included in the
model} \item{jerr}{error flag, for warnings and errors (largely
for internal debugging).}
}
\description{
\code{multiview} uses \code{\link[glmnet:glmnet]{glmnet::glmnet()}} to do most of its work and
therefore takes many of the same parameters, but an intercept is
always included, standardization is always done and several other
parameters do not apply. Therefore they are always overridden and
warnings issued.
}
\details{
The current code can be slow for "large" data sets, e.g. when the
number of features is larger than 1000.  It can be helpful to see
the progress of multiview as it runs; to do this, set trace.it = 1
in the call to multiview or cv.multiview.  With this, multiview
prints out its progress along the way.  One can also pre-filter the
features to a smaller set, using the exclude option, with a filter
function.

If there are missing values in the feature matrices:
we recommend that you center the columns of each feature matrix, and then fill in the missing values with 0.

For example, \cr
\code{x <- scale(x,TRUE,FALSE)} \cr
\code{x[is.na(x)] <- 0} \cr
\code{z <- scale(z,TRUE,FALSE)} \cr
\code{z[is.na(z)] <- 0} \cr

Then run multiview in the usual way. It will exploit the assumed shared latent factors
to make efficient use of the available data.
}
\examples{
# Gaussian
x = matrix(rnorm(100 * 20), 100, 20)
z = matrix(rnorm(100 * 10), 100, 10)
y = rnorm(100)
fit1 = multiview(list(x=x,z=z), y, rho = 0)
print(fit1)

# extract coefficients at a single value of lambda
coef(fit1, s = 0.01) 

# extract ordered (standardized) coefficients at a single value of lambda
coef_ordered(fit1, s = 0.01) 

# make predictions
predict(fit1, newx = list(x[1:10, ],z[1:10, ]), s = c(0.01, 0.005))

# make a path plot of features for the fit
plot(fit1, label=TRUE)

# Binomial
by = sample(c(0,1), 100, replace = TRUE)
fit2 = multiview(list(x=x,z=z), by, family = binomial(), rho=0.5)
predict(fit2, newx = list(x[1:10, ],z[1:10, ]), s = c(0.01, 0.005), type="response")
coef_ordered(fit2, s = 0.01) 
plot(fit2, label=TRUE)

# Poisson
py = matrix(rpois(100, exp(y))) 
fit3 = multiview(list(x=x,z=z), py, family = poisson(), rho=0.5)
predict(fit3, newx = list(x[1:10, ],z[1:10, ]), s = c(0.01, 0.005), type="response")
coef_ordered(fit3, s = 0.01) 
plot(fit3, label=TRUE)

}
\seealso{
\code{print}, \code{coef}, \code{coef_ordered}, \code{predict}, and \code{plot}
methods for \code{"multiview"}, and the \code{"cv.multiview"} function.
}
