% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ranks.R
\name{posterior_ranks}
\alias{posterior_ranks}
\alias{posterior_rank_probs}
\title{Treatment rankings and rank probabilities}
\usage{
posterior_ranks(
  x,
  newdata = NULL,
  study = NULL,
  lower_better = TRUE,
  probs = c(0.025, 0.25, 0.5, 0.75, 0.975),
  sucra = FALSE,
  summary = TRUE
)

posterior_rank_probs(
  x,
  newdata = NULL,
  study = NULL,
  lower_better = TRUE,
  cumulative = FALSE,
  sucra = FALSE
)
}
\arguments{
\item{x}{A \code{stan_nma} object created by \code{\link[=nma]{nma()}}}

\item{newdata}{Only used if a regression model is fitted. A data frame of
study details, one row per study, giving the covariate values at which to
produce relative effects. Column names must match variables in the
regression model. If \code{NULL}, relative effects are produced for all studies
in the network.}

\item{study}{Column of \code{newdata} which specifies study names, otherwise
studies will be labelled by row number.}

\item{lower_better}{Logical, are lower treatment effects better (\code{TRUE};
default) or higher better (\code{FALSE})? See details.}

\item{probs}{Numeric vector of quantiles of interest to present in computed
summary, default \code{c(0.025, 0.25, 0.5, 0.75, 0.975)}}

\item{sucra}{Logical, calculate the surface under the cumulative ranking
curve (SUCRA) for each treatment? Default \code{FALSE}.}

\item{summary}{Logical, calculate posterior summaries? Default \code{TRUE}.}

\item{cumulative}{Logical, return cumulative rank probabilities? Default is
\code{FALSE}, return posterior probabilities of each treatment having a given
rank. If \code{TRUE}, cumulative posterior rank probabilities are returned for
each treatment having a given rank or better.}
}
\value{
A \link{nma_summary} object if \code{summary = TRUE}, otherwise a list
containing a 3D MCMC array of samples and (for regression models) a data
frame of study information.
}
\description{
Produce posterior treatment rankings and rank probabilities from a fitted NMA
model. When a meta-regression is fitted with effect modifier interactions
with treatment, these will differ by study population.
}
\details{
The function \code{posterior_ranks()} produces posterior rankings, which
have a distribution (e.g. mean/median rank and 95\% Credible Interval). The
function \code{posterior_rank_probs()} produces rank probabilities, which give
the posterior probabilities of being ranked first, second, etc. out of all
treatments.

The argument \code{lower_better} specifies whether lower treatment
effects or higher treatment effects are preferred. For example, with a
negative binary outcome lower (more negative) log odds ratios are
preferred, so \code{lower_better = TRUE}. Conversely, for example, if treatments
aim to increase the rate of a positive outcome then \code{lower_better = FALSE}.
}
\examples{
## Smoking cessation
\donttest{
# Run smoking RE NMA example if not already available
if (!exists("smk_fit_RE")) example("example_smk_re", run.donttest = TRUE)
}
\donttest{
# Produce posterior ranks
smk_rank_RE <- posterior_ranks(smk_fit_RE, lower_better = FALSE)
smk_rank_RE
plot(smk_rank_RE)

# Produce rank probabilities
smk_rankprob_RE <- posterior_rank_probs(smk_fit_RE, lower_better = FALSE)
smk_rankprob_RE
plot(smk_rankprob_RE)

# Produce cumulative rank probabilities
smk_cumrankprob_RE <- posterior_rank_probs(smk_fit_RE, lower_better = FALSE,
                                           cumulative = TRUE)
smk_cumrankprob_RE
plot(smk_cumrankprob_RE)

# Further customisation is possible with ggplot commands
plot(smk_cumrankprob_RE) +
  ggplot2::facet_null() +
  ggplot2::aes(colour = Treatment)
}

## Plaque psoriasis ML-NMR
\donttest{
# Run plaque psoriasis ML-NMR example if not already available
if (!exists("pso_fit")) example("example_pso_mlnmr", run.donttest = TRUE)
}
\donttest{
# Produce population-adjusted rankings for all study populations in
# the network

# Ranks
pso_rank <- posterior_ranks(pso_fit)
pso_rank
plot(pso_rank)

# Rank probabilities
pso_rankprobs <- posterior_rank_probs(pso_fit)
pso_rankprobs
plot(pso_rankprobs)

# Cumulative rank probabilities
pso_cumrankprobs <- posterior_rank_probs(pso_fit, cumulative = TRUE)
pso_cumrankprobs
plot(pso_cumrankprobs)

# Produce population-adjusted rankings for a different target
# population
new_agd_means <- data.frame(
  bsa = 0.6,
  prevsys = 0.1,
  psa = 0.2,
  weight = 10,
  durnpso = 3)

# Ranks
posterior_ranks(pso_fit, newdata = new_agd_means)

# Rank probabilities
posterior_rank_probs(pso_fit, newdata = new_agd_means)

# Cumulative rank probabilities
posterior_rank_probs(pso_fit, newdata = new_agd_means,
                     cumulative = TRUE)
}
}
\seealso{
\code{\link[=plot.nma_summary]{plot.nma_summary()}} for plotting the ranks and rank probabilities.
}
