% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/predict.R
\name{predict.stan_nma}
\alias{predict.stan_nma}
\title{Predictions of absolute effects from NMA models}
\usage{
\method{predict}{stan_nma}(
  object,
  ...,
  baseline = NULL,
  newdata = NULL,
  study = NULL,
  trt_ref = NULL,
  type = c("link", "response"),
  level = c("aggregate", "individual"),
  baseline_type = c("link", "response"),
  baseline_level = c("individual", "aggregate"),
  probs = c(0.025, 0.25, 0.5, 0.75, 0.975),
  summary = TRUE
)
}
\arguments{
\item{object}{A \code{stan_nma} object created by \code{\link[=nma]{nma()}}.}

\item{...}{Additional arguments, passed to \code{\link[=uniroot]{uniroot()}} for regression models
if \code{baseline_level = "aggregate"}.}

\item{baseline}{An optional \code{\link[=distr]{distr()}} distribution for the baseline response
(i.e. intercept), about which to produce absolute effects. If \code{NULL},
predictions are produced using the baseline response for each study in the
network with IPD or arm-based AgD.

For regression models, this may be a list of \code{\link[=distr]{distr()}} distributions of the
same length as the number of studies in \code{newdata} (possibly named by the
study names, or otherwise in order of appearance in \code{newdata}).

Use the \code{baseline_type} and \code{baseline_level} arguments to specify whether
this distribution is on the response or linear predictor scale, and (for
ML-NMR or models including IPD) whether this applies to an individual at
the reference level of the covariates or over the entire \code{newdata}
population, respectively. For example, in a model with a logit link with
\code{baseline_type = "link"}, this would be a distribution for the baseline log
odds of an event.

Use the \code{trt_ref} argument to specify which treatment this distribution
applies to.}

\item{newdata}{Only required if a regression model is fitted and \code{baseline}
is specified. A data frame of covariate details, for which to produce
predictions. Column names must match variables in the regression model.

If \code{type = "aggregate"} this should either be a data frame with integration
points as produced by \code{\link[=add_integration]{add_integration()}} (one row per study), or a data
frame with individual covariate values (one row per individual) which are
summarised over.

If \code{type = "individual"} this should be a data frame of individual
covariate values, one row per individual.

If \code{NULL}, predictions are produced for all studies with IPD and/or
arm-based AgD in the network, depending on the value of \code{type}.}

\item{study}{Column of \code{newdata} which specifies study names or IDs. When not
specified: if \code{newdata} contains integration points produced by
\code{\link[=add_integration]{add_integration()}}, studies will be labelled sequentially by row;
otherwise data will be assumed to come from a single study.}

\item{trt_ref}{Treatment to which the \code{baseline} response distribution
refers, if \code{baseline} is specified. By default, the baseline response
distribution will refer to the network reference treatment. Coerced to
character string.}

\item{type}{Whether to produce predictions on the \code{"link"} scale (the
default, e.g. log odds) or \code{"response"} scale (e.g. probabilities).}

\item{level}{The level at which predictions are produced, either
\code{"aggregate"} (the default), or \code{"individual"}. If \code{baseline} is not
specified, predictions are produced for all IPD studies in the network if
\code{type} is \code{"individual"} or \code{"aggregate"}, and for all arm-based AgD
studies in the network if \code{type} is \code{"aggregate"}.}

\item{baseline_type}{When a \code{baseline} distribution is given, specifies
whether this corresponds to the \code{"link"} scale (the default, e.g. log odds)
or \code{"response"} scale (e.g. probabilities).}

\item{baseline_level}{When a \code{baseline} distribution is given, specifies
whether this corresponds to an individual at the reference level of the
covariates (\code{"individual"}, the default), or from an (unadjusted) average
outcome on the reference treatment in the \code{newdata} population
(\code{"aggregate"}). Ignored for AgD NMA, since the only option is
\code{"aggregate"} in this instance.}

\item{probs}{Numeric vector of quantiles of interest to present in computed
summary, default \code{c(0.025, 0.25, 0.5, 0.75, 0.975)}}

\item{summary}{Logical, calculate posterior summaries? Default \code{TRUE}.}
}
\value{
A \link{nma_summary} object if \code{summary = TRUE}, otherwise a list
containing a 3D MCMC array of samples and (for regression models) a data
frame of study information.
}
\description{
Obtain predictions of absolute effects from NMA models fitted with \code{\link[=nma]{nma()}}.
For example, if a model is fitted to binary data with a logit link, predicted
outcome probabilities or log odds can be produced.
}
\examples{
## Smoking cessation
\donttest{
# Run smoking RE NMA example if not already available
if (!exists("smk_fit_RE")) example("example_smk_re", run.donttest = TRUE)
}
\donttest{
# Predicted log odds of success in each study in the network
predict(smk_fit_RE)

# Predicted probabilities of success in each study in the network
predict(smk_fit_RE, type = "response")

# Predicted probabilities in a population with 67 observed events out of 566
# individuals on No Intervention, corresponding to a Beta(67, 566 - 67)
# distribution on the baseline probability of response, using
# `baseline_type = "response"`
(smk_pred_RE <- predict(smk_fit_RE,
                        baseline = distr(qbeta, 67, 566 - 67),
                        baseline_type = "response",
                        type = "response"))
plot(smk_pred_RE, ref_line = c(0, 1))

# Predicted probabilities in a population with a baseline log odds of
# response on No Intervention given a Normal distribution with mean -2
# and SD 0.13, using `baseline_type = "link"` (the default)
# Note: this is approximately equivalent to the above Beta distribution on
# the baseline probability
(smk_pred_RE2 <- predict(smk_fit_RE,
                         baseline = distr(qnorm, mean = -2, sd = 0.13),
                         type = "response"))
plot(smk_pred_RE2, ref_line = c(0, 1))
}

## Plaque psoriasis ML-NMR
\donttest{
# Run plaque psoriasis ML-NMR example if not already available
if (!exists("pso_fit")) example("example_pso_mlnmr", run.donttest = TRUE)
}
\donttest{
# Predicted probabilities of response in each study in the network
(pso_pred <- predict(pso_fit, type = "response"))
plot(pso_pred, ref_line = c(0, 1))

# Predicted probabilites of response in a new target population, with means
# and SDs or proportions given by
new_agd_int <- data.frame(
  bsa_mean = 0.6,
  bsa_sd = 0.3,
  prevsys = 0.1,
  psa = 0.2,
  weight_mean = 10,
  weight_sd = 1,
  durnpso_mean = 3,
  durnpso_sd = 1
)

# We need to add integration points to this data frame of new data
# We use the weighted mean correlation matrix computed from the IPD studies
new_agd_int <- add_integration(new_agd_int,
                               durnpso = distr(qgamma, mean = durnpso_mean, sd = durnpso_sd),
                               prevsys = distr(qbern, prob = prevsys),
                               bsa = distr(qlogitnorm, mean = bsa_mean, sd = bsa_sd),
                               weight = distr(qgamma, mean = weight_mean, sd = weight_sd),
                               psa = distr(qbern, prob = psa),
                               cor = pso_net$int_cor,
                               n_int = 1000)

# Predicted probabilities of achieving PASI 75 in this target population, given
# a Normal(-1.75, 0.08^2) distribution on the baseline probit-probability of
# response on Placebo (at the reference levels of the covariates), are given by
(pso_pred_new <- predict(pso_fit,
                         type = "response",
                         newdata = new_agd_int,
                         baseline = distr(qnorm, -1.75, 0.08)))
plot(pso_pred_new, ref_line = c(0, 1))
}
}
\seealso{
\code{\link[=plot.nma_summary]{plot.nma_summary()}} for plotting the predictions.
}
