% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tune.R
\name{tune}
\alias{tune}
\title{Function for Tuning a Learner}
\usage{
tune(
  method,
  task,
  learner,
  resampling,
  measures = NULL,
  term_evals = NULL,
  term_time = NULL,
  terminator = NULL,
  search_space = NULL,
  store_benchmark_result = TRUE,
  store_models = FALSE,
  check_values = FALSE,
  allow_hotstart = FALSE,
  keep_hotstart_stack = FALSE,
  evaluate_default = FALSE,
  callbacks = list(),
  ...
)
}
\arguments{
\item{method}{(\code{character(1)} | \link{Tuner})\cr
Key to retrieve tuner from \link{mlr_tuners} dictionary or \link{Tuner} object.}

\item{task}{(\link[mlr3:Task]{mlr3::Task})\cr
Task to operate on.}

\item{learner}{(\link[mlr3:Learner]{mlr3::Learner})\cr
Learner to tune.}

\item{resampling}{(\link[mlr3:Resampling]{mlr3::Resampling})\cr
Resampling that is used to evaluated the performance of the hyperparameter
configurations. Uninstantiated resamplings are instantiated during
construction so that all configurations are evaluated on the same data
splits. Already instantiated resamplings are kept unchanged. Specialized
\link{Tuner} change the resampling e.g. to evaluate a hyperparameter
configuration on different data splits. This field, however, always returns
the resampling passed in construction.}

\item{measures}{(\link[mlr3:Measure]{mlr3::Measure} or list of \link[mlr3:Measure]{mlr3::Measure})\cr
A single measure creates a \link{TuningInstanceSingleCrit} and multiple measures a \link{TuningInstanceMultiCrit}.
If \code{NULL}, default measure is used.}

\item{term_evals}{(\code{integer(1)})\cr
Number of allowed evaluations.}

\item{term_time}{(\code{integer(1)})\cr
Maximum allowed time in seconds.}

\item{terminator}{(\link{Terminator})\cr
Stop criterion of the tuning process.}

\item{search_space}{(\link[paradox:ParamSet]{paradox::ParamSet})\cr
Hyperparameter search space. If \code{NULL} (default), the search space is
constructed from the \link{TuneToken} of the learner's parameter set
(learner$param_set).}

\item{store_benchmark_result}{(\code{logical(1)})\cr
If \code{TRUE} (default), store resample result of evaluated hyperparameter
configurations in archive as \link[mlr3:BenchmarkResult]{mlr3::BenchmarkResult}.}

\item{store_models}{(\code{logical(1)})\cr
If \code{TRUE}, fitted models are stored in the benchmark result
(\code{archive$benchmark_result}). If \code{store_benchmark_result = FALSE}, models
are only stored temporarily and not accessible after the tuning. This
combination is needed for measures that require a model.}

\item{check_values}{(\code{logical(1)})\cr
If \code{TRUE}, hyperparameter values are checked before evaluation and
performance scores after. If \code{FALSE} (default), values are unchecked but
computational overhead is reduced.}

\item{allow_hotstart}{(\code{logical(1)})\cr
Allow to hotstart learners with previously fitted models. See also
\link[mlr3:HotstartStack]{mlr3::HotstartStack}. The learner must support hotstarting. Sets
\code{store_models = TRUE}.}

\item{keep_hotstart_stack}{(\code{logical(1)})\cr
If \code{TRUE}, \link[mlr3:HotstartStack]{mlr3::HotstartStack} is kept in \verb{$objective$hotstart_stack}
after tuning.}

\item{evaluate_default}{(\code{logical(1)})\cr
If \code{TRUE}, learner is evaluated with hyperparameters set to their default
values at the start of the optimization.}

\item{callbacks}{(list of \link{CallbackTuning})\cr
List of callbacks.}

\item{...}{(named \code{list()})\cr
Named arguments to be set as parameters of the tuner.}
}
\value{
\link{TuningInstanceSingleCrit} | \link{TuningInstanceMultiCrit}
}
\description{
Function to tune a \link[mlr3:Learner]{mlr3::Learner}.
The function internally creates a \link{TuningInstanceSingleCrit} or \link{TuningInstanceMultiCrit} which describe the tuning problem.
It executes the tuning with the \link{Tuner} (\code{method}) and returns the result with the tuning instance (\verb{$result}).
The \link{ArchiveTuning} (\verb{$archive}) stores all evaluated hyperparameter configurations and performance scores.
}
\details{
The \link[mlr3:Task]{mlr3::Task}, \link[mlr3:Learner]{mlr3::Learner}, \link[mlr3:Resampling]{mlr3::Resampling}, \link[mlr3:Measure]{mlr3::Measure} and \link{Terminator} are used to construct a \link{TuningInstanceSingleCrit}.
If multiple performance \link[=Measure]{Measures} are supplied, a \link{TuningInstanceMultiCrit} is created.
The parameter \code{term_evals} and \code{term_time} are shortcuts to create a \link{Terminator}.
If both parameters are passed, a \link{TerminatorCombo} is constructed.
For other \link[=Terminator]{Terminators}, pass one with \code{terminator}.
If no termination criterion is needed, set \code{term_evals}, \code{term_time} and \code{terminator} to \code{NULL}.
The search space is created from \link[paradox:to_tune]{paradox::TuneToken} or is supplied by \code{search_space}.
}
\section{Resources}{

\itemize{
\item \href{https://mlr3book.mlr-org.com/optimization.html}{book chapter} on hyperparameter optimization.
\item \href{https://mlr3book.mlr-org.com/technical.html#sec-tuning-space}{book chapter} on tuning spaces.
\item \href{https://mlr-org.com/gallery/2021-03-09-practical-tuning-series-tune-a-support-vector-machine/}{gallery post} on tuning an svm.
\item \href{https://mlr3tuningspaces.mlr-org.com/}{mlr3tuningspaces} extension package.
}
}

\section{Analysis}{

For analyzing the tuning results, it is recommended to pass the \link{ArchiveTuning} to \code{as.data.table()}.
The returned data table is joined with the benchmark result which adds the \link[mlr3:ResampleResult]{mlr3::ResampleResult} for each hyperparameter evaluation.

The archive provides various getters (e.g. \verb{$learners()}) to ease the access.
All getters extract by position (\code{i}) or unique hash (\code{uhash}).
For a complete list of all getters see the methods section.

The benchmark result (\verb{$benchmark_result}) allows to score the hyperparameter configurations again on a different measure.
Alternatively, measures can be supplied to \code{as.data.table()}.

The \CRANpkg{mlr3viz} package provides visualizations for tuning results.
}

\examples{
# Hyperparameter optimization on the Palmer Penguins data set
task = tsk("pima")

# Load learner and set search space
learner = lrn("classif.rpart",
  cp = to_tune(1e-04, 1e-1, logscale = TRUE)
)

# Run tuning
instance = tune(
  method = tnr("random_search", batch_size = 2),
  task = tsk("pima"),
  learner = learner,
  resampling = rsmp ("holdout"),
  measures = msr("classif.ce"),
  terminator = trm("evals", n_evals = 4)
)

# Set optimal hyperparameter configuration to learner
learner$param_set$values = instance$result_learner_param_vals

# Train the learner on the full data set
learner$train(task)

# Inspect all evaluated configurations
as.data.table(instance$archive)
}
