% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/OptimizerAsyncMbo.R
\name{mlr_optimizers_async_mbo}
\alias{mlr_optimizers_async_mbo}
\alias{OptimizerAsyncMbo}
\title{Asynchronous Model Based Optimization}
\description{
\code{OptimizerAsyncMbo} class that implements Asynchronous Model Based Optimization (AMBO).
AMBO starts multiple sequential MBO runs on different workers.
The worker communicate asynchronously through a shared archive relying on the \CRANpkg{rush} package.
The optimizer follows a modular layout in which the surrogate model, acquisition function, and acquisition optimizer can be changed.
The \link{SurrogateLearner} will impute missing values due to pending evaluations.
A stochastic \link{AcqFunction}, e.g., \link{AcqFunctionStochasticEI} or \link{AcqFunctionStochasticCB} is used to create varying versions of the acquisition
function on each worker, promoting different exploration-exploitation trade-offs.
The \link{AcqOptimizer} class remains consistent with the one used in synchronous MBO.

In contrast to \link{OptimizerMbo}, no \link{loop_function} can be specified that determines the AMBO flavor as \code{OptimizerAsyncMbo} simply relies on
a surrogate update, acquisition function update and acquisition function optimization step as an internal loop.

Currently, only single-objective optimization is supported and \code{OptimizerAsyncMbo} is considered an experimental feature and API might be subject to changes.

Note that in general the \link{SurrogateLearner} is updated one final time on all available data after the optimization process has terminated.
However, in certain scenarios this is not always possible or meaningful.
It is therefore recommended to manually inspect the \link{SurrogateLearner} after optimization if it is to be used, e.g., for visualization purposes to make
sure that it has been properly updated on all available data.
If this final update of the \link{SurrogateLearner} could not be performed successfully, a warning will be logged.

By specifying a \link{ResultAssigner}, one can alter how the final result is determined after optimization, e.g.,
simply based on the evaluations logged in the archive \link{ResultAssignerArchive} or based on the \link{Surrogate} via \link{ResultAssignerSurrogate}.
}
\section{Archive}{

The \link[bbotk:ArchiveAsync]{bbotk::ArchiveAsync} holds the following additional columns that are specific to AMBO algorithms:
\itemize{
\item \code{acq_function$id} (\code{numeric(1)})\cr
The value of the acquisition function.
\item \code{".already_evaluated"} (\verb{logical(1))}\cr
Whether this point was already evaluated. Depends on the \code{skip_already_evaluated} parameter of the \link{AcqOptimizer}.
}

If the \link[bbotk:ArchiveAsync]{bbotk::ArchiveAsync} does not contain any evaluations prior to optimization, an initial design is needed.
If the \code{initial_design} parameter is specified to be a \code{data.table}, this data will be used.
Otherwise, if it is \code{NULL}, an initial design of size \code{design_size} will be generated based on the \code{generate_design} sampling function.
See also the parameters below.
}

\section{Parameters}{

\describe{
\item{\code{initial_design}}{\code{data.table::data.table()}\cr
Initial design of the optimization.
If \code{NULL}, a design of size \code{design_size} is generated with the specified \code{design_function}.
Default is \code{NULL}.}
\item{\code{design_size}}{\code{integer(1)}\cr
Size of the initial design if it is to be generated.
Default is \code{100}.}
\item{\code{design_function}}{\code{character(1)}\cr
Sampling function to generate the initial design.
Can be \code{random} \link[paradox:generate_design_random]{paradox::generate_design_random}, \code{lhs} \link[paradox:generate_design_lhs]{paradox::generate_design_lhs}, or \code{sobol} \link[paradox:generate_design_sobol]{paradox::generate_design_sobol}.
Default is \code{sobol}.}
\item{\code{n_workers}}{\code{integer(1)}\cr
Number of parallel workers.
If \code{NULL}, all rush workers specified via \code{\link[rush:rush_plan]{rush::rush_plan()}} are used.
Default is \code{NULL}.}
}
}

\examples{
\donttest{
if (requireNamespace("rush") &
    requireNamespace("mlr3learners") &
    requireNamespace("DiceKriging") &
    requireNamespace("rgenoud")) {

  if (redis_available()) {

    library(bbotk)
    library(paradox)
    library(mlr3learners)

    fun = function(xs) {
      list(y = xs$x ^ 2)
    }
    domain = ps(x = p_dbl(lower = -10, upper = 10))
    codomain = ps(y = p_dbl(tags = "minimize"))
    objective = ObjectiveRFun$new(fun = fun, domain = domain, codomain = codomain)

    instance = OptimInstanceAsyncSingleCrit$new(
      objective = objective,
      terminator = trm("evals", n_evals = 10))

    rush::rush_plan(n_workers=2)

    optimizer = opt("async_mbo", design_size = 4, n_workers = 2)

    optimizer$optimize(instance)
  } else {
    message("Redis server is not available.\nPlease set up Redis prior to running the example.")
  }
}
}
}
\section{Super classes}{
\code{\link[bbotk:Optimizer]{bbotk::Optimizer}} -> \code{\link[bbotk:OptimizerAsync]{bbotk::OptimizerAsync}} -> \code{OptimizerAsyncMbo}
}
\section{Active bindings}{
\if{html}{\out{<div class="r6-active-bindings">}}
\describe{
\item{\code{surrogate}}{(\link{Surrogate} | \code{NULL})\cr
The surrogate.}

\item{\code{acq_function}}{(\link{AcqFunction} | \code{NULL})\cr
The acquisition function.}

\item{\code{acq_optimizer}}{(\link{AcqOptimizer} | \code{NULL})\cr
The acquisition function optimizer.}

\item{\code{result_assigner}}{(\link{ResultAssigner} | \code{NULL})\cr
The result assigner.}

\item{\code{param_classes}}{(\code{character()})\cr
Supported parameter classes that the optimizer can optimize.
Determined based on the \code{surrogate} and the \code{acq_optimizer}.
This corresponds to the values given by a \link[paradox:ParamSet]{paradox::ParamSet}'s
\verb{$class} field.}

\item{\code{properties}}{(\code{character()})\cr
Set of properties of the optimizer.
Must be a subset of \code{\link[bbotk:bbotk_reflections]{bbotk_reflections$optimizer_properties}}.
MBO in principle is very flexible and by default we assume that the optimizer has all properties.
When fully initialized, properties are determined based on the loop, e.g., the \code{loop_function}, and \code{surrogate}.}

\item{\code{packages}}{(\code{character()})\cr
Set of required packages.
A warning is signaled prior to optimization if at least one of the packages is not installed, but loaded (not attached) later on-demand via \code{\link[=requireNamespace]{requireNamespace()}}.
Required packages are determined based on the \code{acq_function}, \code{surrogate} and the \code{acq_optimizer}.}
}
\if{html}{\out{</div>}}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-OptimizerAsyncMbo-new}{\code{OptimizerAsyncMbo$new()}}
\item \href{#method-OptimizerAsyncMbo-print}{\code{OptimizerAsyncMbo$print()}}
\item \href{#method-OptimizerAsyncMbo-reset}{\code{OptimizerAsyncMbo$reset()}}
\item \href{#method-OptimizerAsyncMbo-optimize}{\code{OptimizerAsyncMbo$optimize()}}
\item \href{#method-OptimizerAsyncMbo-clone}{\code{OptimizerAsyncMbo$clone()}}
}
}
\if{html}{\out{
<details open><summary>Inherited methods</summary>
<ul>
<li><span class="pkg-link" data-pkg="bbotk" data-topic="Optimizer" data-id="format"><a href='../../bbotk/html/Optimizer.html#method-Optimizer-format'><code>bbotk::Optimizer$format()</code></a></span></li>
<li><span class="pkg-link" data-pkg="bbotk" data-topic="Optimizer" data-id="help"><a href='../../bbotk/html/Optimizer.html#method-Optimizer-help'><code>bbotk::Optimizer$help()</code></a></span></li>
</ul>
</details>
}}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-OptimizerAsyncMbo-new"></a>}}
\if{latex}{\out{\hypertarget{method-OptimizerAsyncMbo-new}{}}}
\subsection{Method \code{new()}}{
Creates a new instance of this \link[R6:R6Class]{R6} class.

If \code{surrogate} is \code{NULL} and the \code{acq_function$surrogate} field is populated, this \link{SurrogateLearner} is used.
Otherwise, \code{default_surrogate(instance)} is used.
If \code{acq_function} is \code{NULL} and the \code{acq_optimizer$acq_function} field is populated, this \link{AcqFunction} is used (and therefore its \verb{$surrogate} if populated; see above).
Otherwise \code{default_acqfunction(instance)} is used.
If \code{acq_optimizer} is \code{NULL}, \code{default_acqoptimizer(instance)} is used.

Even if already initialized, the \code{surrogate$archive} field will always be overwritten by the \link[bbotk:ArchiveAsync]{bbotk::ArchiveAsync} of the current \link[bbotk:OptimInstanceAsyncSingleCrit]{bbotk::OptimInstanceAsyncSingleCrit} to be optimized.

For more information on default values for \code{surrogate}, \code{acq_function}, \code{acq_optimizer} and \code{result_assigner}, see \code{?mbo_defaults}.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{OptimizerAsyncMbo$new(
  id = "async_mbo",
  surrogate = NULL,
  acq_function = NULL,
  acq_optimizer = NULL,
  result_assigner = NULL,
  param_set = NULL,
  label = "Asynchronous Model Based Optimization",
  man = "mlr3mbo::OptimizerAsyncMbo"
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{id}}{(\code{character(1)})\cr
Identifier for the new instance.}

\item{\code{surrogate}}{(\link{Surrogate} | \code{NULL})\cr
The surrogate.}

\item{\code{acq_function}}{(\link{AcqFunction} | \code{NULL})\cr
The acquisition function.}

\item{\code{acq_optimizer}}{(\link{AcqOptimizer} | \code{NULL})\cr
The acquisition function optimizer.}

\item{\code{result_assigner}}{(\link{ResultAssigner} | \code{NULL})\cr
The result assigner.}

\item{\code{param_set}}{(\link[paradox:ParamSet]{paradox::ParamSet})\cr
Set of control parameters.}

\item{\code{label}}{(\code{character(1)})\cr
Label for this object.
Can be used in tables, plot and text output instead of the ID.}

\item{\code{man}}{(\code{character(1)})\cr
String in the format \verb{[pkg]::[topic]} pointing to a manual page for this object.
The referenced help package can be opened via method \verb{$help()}.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-OptimizerAsyncMbo-print"></a>}}
\if{latex}{\out{\hypertarget{method-OptimizerAsyncMbo-print}{}}}
\subsection{Method \code{print()}}{
Print method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{OptimizerAsyncMbo$print()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
(\code{character()}).
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-OptimizerAsyncMbo-reset"></a>}}
\if{latex}{\out{\hypertarget{method-OptimizerAsyncMbo-reset}{}}}
\subsection{Method \code{reset()}}{
Reset the optimizer.
Sets the following fields to \code{NULL}:
\code{surrogate}, \code{acq_function}, \code{acq_optimizer},\code{result_assigner}
Resets parameter values \code{design_size} and \code{design_function} to their defaults.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{OptimizerAsyncMbo$reset()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-OptimizerAsyncMbo-optimize"></a>}}
\if{latex}{\out{\hypertarget{method-OptimizerAsyncMbo-optimize}{}}}
\subsection{Method \code{optimize()}}{
Performs the optimization on an \link[bbotk:OptimInstanceAsyncSingleCrit]{bbotk::OptimInstanceAsyncSingleCrit} until termination.
The single evaluations will be written into the \link[bbotk:ArchiveAsync]{bbotk::ArchiveAsync}.
The result will be written into the instance object.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{OptimizerAsyncMbo$optimize(inst)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{inst}}{(\link[bbotk:OptimInstanceAsyncSingleCrit]{bbotk::OptimInstanceAsyncSingleCrit}).}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
\code{\link[data.table:data.table]{data.table::data.table()}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-OptimizerAsyncMbo-clone"></a>}}
\if{latex}{\out{\hypertarget{method-OptimizerAsyncMbo-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{OptimizerAsyncMbo$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
