% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/OptimizerHyperband.R
\name{mlr_optimizers_hyperband}
\alias{mlr_optimizers_hyperband}
\alias{OptimizerHyperband}
\title{Optimizer using the Hyperband algorithm}
\source{
Li L, Jamieson K, DeSalvo G, Rostamizadeh A, Talwalkar A (2018).
\dQuote{Hyperband: A Novel Bandit-Based Approach to Hyperparameter Optimization.}
\emph{Journal of Machine Learning Research}, \bold{18}(185), 1-52.
\url{https://jmlr.org/papers/v18/16-558.html}.
}
\description{
\code{OptimizerHyperband} class that implements hyperband optimization. Hyperband
is a budget oriented-procedure, weeding out suboptimal performing
points early in a sequential training process, increasing
optimization efficiency as a consequence.

For this, several brackets are constructed with an associated set of
points for each bracket. Each bracket as several stages. Different
brackets are initialized with different amounts of points and
different budget sizes. To get an idea of how the bracket layout looks like
for a given argument set, use \code{\link[=hyperband_schedule]{hyperband_schedule()}}.

To identify the budget for evaluating hyperband, the user has to specify
explicitly which parameter of the objective function influences the budget by
tagging a single parameter in the \link[paradox:ParamSet]{paradox::ParamSet} with \code{"budget"}.

Naturally, hyperband terminates once all of its brackets are evaluated, so a
\link[bbotk:Terminator]{bbotk::Terminator} in the \link[bbotk:OptimInstanceSingleCrit]{bbotk::OptimInstanceSingleCrit} |
\link[bbotk:OptimInstanceMultiCrit]{bbotk::OptimInstanceMultiCrit} acts as an upper bound and should be only set to a
low value if one is unsure of how long hyperband will take to finish under
the given settings.
}
\section{Dictionary}{

This \link{Optimizer} can be instantiated via the \link[mlr3misc:Dictionary]{dictionary}
\link{mlr_optimizers} or with the associated sugar function \code{\link[=opt]{opt()}}:\preformatted{mlr_optimizers$get("hyperband")
opt("hyperband")
}
}

\section{Parameters}{

\describe{
\item{\code{eta}}{\code{numeric(1)}\cr
Fraction parameter of the successive halving algorithm: With every stage the
configuration budget is increased by a factor of \code{eta} and only the best
\code{1/eta} points are used for the next stage. Non-integer values are supported,
but \code{eta} is not allowed to be less or equal 1.
}
\item{\code{sampler}}{\link[paradox:Sampler]{paradox::Sampler}\cr
Object defining how the samples of the parameter space should be drawn during
the initialization of each bracket. The default is uniform sampling.
}
\item{\code{repeats}}{\code{logical(1)}\cr
If \code{FALSE} (default), hyperband terminates once all brackets are evaluated.
Otherwise, hyperband starts over again once the last bracket is evaluated.
}}
}

\section{Archive}{

The \link[bbotk:Archive]{bbotk::Archive} holds the following additional columns that are specific
to the hyperband algorithm:
\itemize{
\item \code{bracket} (\code{integer(1)})\cr
The bracket index. Counts down to 0.
\item \code{stage} (\verb{integer(1))}\cr
The stages of each bracket. Starts counting at 0.
}
}

\section{Custom sampler}{

Hyperband supports custom \link[paradox:Sampler]{paradox::Sampler} object for initial
configurations in each bracket.
A custom sampler may look like this (the full example is given in the
\emph{examples} section):\preformatted{# - beta distribution with alpha = 2 and beta = 5
# - categorical distribution with custom probabilities
sampler = SamplerJointIndep$new(list(
  Sampler1DRfun$new(params[[2]], function(n) rbeta(n, 2, 5)),
  Sampler1DCateg$new(params[[3]], prob = c(0.2, 0.3, 0.5))
))
}
}

\section{Runtime}{

The calculation of each bracket currently assumes a linear runtime in the
chosen budget parameter is always given. Hyperband is designed so each
bracket requires approximately the same runtime as the sum of the budget
over all configurations in each bracket is roughly the same. This will not
hold true once the scaling in the budget parameter is not linear
anymore, even though the sum of the budgets in each bracket remains the
same. A possible adaption would be to introduce a trafo, like it is shown in
the \emph{examples} section.
}

\section{Progress Bars}{

\verb{$optimize()} supports progress bars via the package \CRANpkg{progressr}
combined with a \link{Terminator}. Simply wrap the function in
\code{progressr::with_progress()} to enable them. We recommend to use package
\CRANpkg{progress} as backend; enable with \code{progressr::handlers("progress")}.
}

\section{Parallelization}{

In order to support general termination criteria and parallelization, we
evaluate points in a batch-fashion of size \code{batch_size}. The points of one
stage in a bracket are evaluated in one batch. Parallelization is supported
via package \CRANpkg{future} (see \code{\link[mlr3:benchmark]{mlr3::benchmark()}}'s section on
parallelization for more details).
}

\section{Logging}{

Hyperband uses a logger (as implemented in \CRANpkg{lgr}) from package
\CRANpkg{bbotk}.
Use \code{lgr::get_logger("bbotk")} to access and control the logger.
}

\examples{
library(bbotk)
library(data.table)

search_space = domain = ps(
  x1 = p_dbl(-5, 10),
  x2 = p_dbl(0, 15),
  fidelity = p_dbl(1e-2, 1, tags = "budget")
)

# modified branin function
objective = ObjectiveRFun$new(
  fun = branin,
  domain = domain,
  codomain = ps(y = p_dbl(tags = "minimize"))
)

instance = OptimInstanceSingleCrit$new(
  objective = objective,
  search_space = search_space,
  terminator = trm("none")
)

optimizer = opt("hyperband")

# modifies the instance by reference
optimizer$optimize(instance)

# best scoring evaluation
instance$result

# all evaluations
as.data.table(instance$archive)
}
\section{Super class}{
\code{\link[bbotk:Optimizer]{bbotk::Optimizer}} -> \code{OptimizerHyperband}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-new}{\code{OptimizerHyperband$new()}}
\item \href{#method-clone}{\code{OptimizerHyperband$clone()}}
}
}
\if{html}{
\out{<details open ><summary>Inherited methods</summary>}
\itemize{
\item \out{<span class="pkg-link" data-pkg="bbotk" data-topic="Optimizer" data-id="format">}\href{../../bbotk/html/Optimizer.html#method-format}{\code{bbotk::Optimizer$format()}}\out{</span>}
\item \out{<span class="pkg-link" data-pkg="bbotk" data-topic="Optimizer" data-id="optimize">}\href{../../bbotk/html/Optimizer.html#method-optimize}{\code{bbotk::Optimizer$optimize()}}\out{</span>}
\item \out{<span class="pkg-link" data-pkg="bbotk" data-topic="Optimizer" data-id="print">}\href{../../bbotk/html/Optimizer.html#method-print}{\code{bbotk::Optimizer$print()}}\out{</span>}
}
\out{</details>}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-new"></a>}}
\if{latex}{\out{\hypertarget{method-new}{}}}
\subsection{Method \code{new()}}{
Creates a new instance of this \link[R6:R6Class]{R6} class.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{OptimizerHyperband$new()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-clone"></a>}}
\if{latex}{\out{\hypertarget{method-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{OptimizerHyperband$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
