% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/TunerHyperband.R
\name{mlr_tuners_hyperband}
\alias{mlr_tuners_hyperband}
\alias{TunerHyperband}
\title{Tuner using the Hyperband algorithm}
\source{
Li L, Jamieson K, DeSalvo G, Rostamizadeh A, Talwalkar A (2018).
\dQuote{Hyperband: A Novel Bandit-Based Approach to Hyperparameter Optimization.}
\emph{Journal of Machine Learning Research}, \bold{18}(185), 1-52.
\url{https://jmlr.org/papers/v18/16-558.html}.
}
\description{
\code{TunerHyperband} class that implements hyperband tuning. Hyperband is a
budget oriented-procedure, weeding out suboptimal performing configurations
early in a sequential training process, increasing tuning efficiency as a
consequence.

For this, several brackets are constructed with an associated set of
configurations for each bracket. Each bracket as several stages. Different
brackets are initialized with different amounts of configurations and
different budget sizes. To get an idea of how the bracket layout looks like
for a given argument set, please have a look in the \code{details}.

To identify the budget for evaluating hyperband, the user has to specify
explicitly which hyperparameter of the learner influences the budget by
tagging a single hyperparameter in the \link[paradox:ParamSet]{paradox::ParamSet} with \code{"budget"}.
An alternative approach using subsampling and pipelines is described below.

Naturally, hyperband terminates once all of its brackets are evaluated, so a
\link[bbotk:Terminator]{bbotk::Terminator} in the tuning instance acts as an upper bound and should
be only set to a low value if one is unsure of how long hyperband will take
to finish under the given settings.
}
\details{
This sections explains the calculation of the constants for each bracket.
A small overview will be given here, but for more details please check
out the original paper (see \code{references}).
To keep things uniform with the notation in the paper (and to safe space in
the formulas), \code{R} is used for the upper budget that last remaining
configuration should reach. The formula to calculate the amount of brackets
is \code{floor(log(R, eta)) + 1}. To calculate the starting budget in each
bracket, use \code{R * eta^(-s)}, where \code{s} is the maximum bracket minus the
current bracket index.
For the starting configurations in each bracket it is
\code{ceiling((B/R) * ((eta^s)/(s+1)))}, with \verb{B = (bracket amount) * R}.
To receive a table with the full brackets layout, load the following function
and execute it for the desired \code{R} and \code{eta}.\preformatted{hyperband_brackets = function(R, eta) \{

  result = data.frame()
  smax = floor(log(R, eta))
  B = (smax + 1) * R

  # outer loop - iterate over brackets
  for (s in smax:0) \{

    n = ceiling((B/R) * ((eta^s)/(s+1)))
    r = R * eta^(-s)

    # inner loop - iterate over bracket stages
    for (i in 0:s) \{

      ni = floor(n * eta^(-i))
      ri = r * eta^i
      result = rbind(result, c(smax - s + 1, i + 1, ri, ni))
    \}
  \}

  names(result) = c("bracket", "bracket_stage", "budget", "n_configs")
  return(result)
\}

hyperband_brackets(R = 81L, eta = 3L)
}
}
\section{Parameters}{

\describe{
\item{\code{eta}}{\code{numeric(1)}\cr
Fraction parameter of the successive halving algorithm: With every step the
configuration budget is increased by a factor of \code{eta} and only the best
\code{1/eta} configurations are used for the next stage. Non-integer values are
supported, but \code{eta} is not allowed to be less or equal 1.}
\item{\code{sampler}}{\link[paradox:Sampler]{paradox::Sampler}\cr
Object defining how the samples of the parameter space should be drawn during
the initialization of each bracket. The default is uniform sampling.}
}
}

\section{Archive}{

The \link[mlr3tuning:ArchiveTuning]{mlr3tuning::ArchiveTuning} holds the following additional columns that
are specific to the hyperband tuner:
\itemize{
\item \code{bracket} (\code{integer(1)})\cr
The console logs about the bracket index are actually not matching
with the original hyperband algorithm, which counts down the brackets
and stops after evaluating bracket 0. The true bracket indices are
given in this column.
\item \code{bracket_stage} (\verb{integer(1))}\cr
The bracket stage of each bracket. Hyperband starts counting at 0.
\item \code{budget_scaled} (\code{numeric(1)})\cr
The intermediate budget in each bracket stage calculated by hyperband.
Because hyperband is originally only considered for budgets starting at 1, some
rescaling is done to allow budgets starting at different values.
For this, budgets are internally divided by the lower budget bound to
get a lower budget of 1. Before the learner
receives its budgets for evaluation, the budget is transformed back to
match the original scale again.
\item \code{budget_real} (\code{numeric(1)})\cr
The real budget values the learner uses for evaluation after hyperband
calculated its scaled budget.
\item \code{n_configs} (\code{integer(1)})\cr
The amount of evaluated configurations in each stage. These correspond
to the \code{r_i} in the original paper.
}
}

\section{Hyperband without learner budget}{

Thanks to \CRANpkg{mlr3pipelines}, it is possible to use hyperband in
combination with learners lacking a natural budget parameter. For example,
any \link[mlr3:Learner]{mlr3::Learner} can be augmented with a \link[mlr3pipelines:PipeOp]{mlr3pipelines::PipeOp}
operator such as \link[mlr3pipelines:mlr_pipeops_subsample]{mlr3pipelines::PipeOpSubsample}. With the
subsampling rate as budget parameter, the resulting
\link[mlr3pipelines:mlr_learners_graph]{mlr3pipelines::GraphLearner} is fitted on small proportions of
the \link[mlr3:Task]{mlr3::Task} in the first brackets, and on the complete Task in
last brackets. See examples for some code.
}

\section{Custom sampler}{

Hyperband supports custom \link[paradox:Sampler]{paradox::Sampler} object for initial
configurations in each bracket.
A custom sampler may look like this (the full example is given in the
\code{examples} section):\preformatted{# - beta distribution with alpha = 2 and beta = 5
# - categorical distribution with custom probabilities
sampler = SamplerJointIndep$new(list(
  Sampler1DRfun$new(params[[2]], function(n) rbeta(n, 2, 5)),
  Sampler1DCateg$new(params[[3]], prob = c(0.2, 0.3, 0.5))
))
}
}

\section{Runtime scaling w.r.t. the chosen budget}{

The calculation of each bracket currently assumes a linear runtime in the
chosen budget parameter is always given. Hyperband is designed so each
bracket requires approximately the same runtime as the sum of the budget
over all configurations in each bracket is roughly the same. This will not
hold true once the scaling in the budget parameter is not linear
anymore, even though the sum of the budgets in each bracket remains the
same. A basic example can be viewed by calling the function
\code{hyperband_brackets} below with the arguments \code{R = 2} and \code{eta = 2}. If we
run a learner with O(budget^2) time complexity, the runtime of the last
bracket will be 33\% longer than the first bracket
(time of bracket 1 = 2 * 1^2 + 2^2 = 6; time of bracket 2 = 2 * 2^2 = 8).
Of course, this won't break anything, but it should be kept in mind when
applying hyperband. A possible adaption would be to introduce a trafo,
like it is shown in the \code{examples}.
}

\section{Logging}{

When loading the \link{mlr3hyperband} package, two loggers based on the \link{lgr}
package are made available. One is called \code{mlr3}, the other \code{bbotk}. All
\code{mlr3} methods log into the \code{mlr3} logger. All optimization methods form the
packags \link{bbotk}, \link{mlr3tuning} and \link{mlr3hyperband} log into the \code{bbotk}
logger. To hide the \link{mlr3} logging messages run:\preformatted{lgr::get_logger("mlr3")$set_threshold("warn")
}
}

\examples{
if(requireNamespace("xgboost")) {
library(mlr3)
library(mlr3learners)
library(paradox)
library(mlr3tuning)
library(mlr3hyperband)

# Define hyperparameter and budget parameter for tuning with hyperband
ps = ParamSet$new(list(
  ParamInt$new("nrounds", lower = 1, upper = 4, tag = "budget"),
  ParamDbl$new("eta", lower = 0, upper = 1),
  ParamFct$new("booster", levels = c("gbtree", "gblinear", "dart"))
))

# Define termination criterion
# Hyperband terminates itself
terminator = trm("none")

# Create tuning instance
inst = TuningInstanceSingleCrit$new(
  task = tsk("iris"),
  learner = lrn("classif.xgboost"),
  resampling = rsmp("holdout"),
  measure = msr("classif.ce"),
  search_space = ps,
  terminator = terminator,
)

# Load tuner
tuner = tnr("hyperband", eta = 2L)

\donttest{
# Trigger optimization
tuner$optimize(inst)

# Print all evaluations
as.data.table(inst$archive)}
}
}
\section{Super class}{
\code{\link[mlr3tuning:Tuner]{mlr3tuning::Tuner}} -> \code{TunerHyperband}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-new}{\code{TunerHyperband$new()}}
\item \href{#method-clone}{\code{TunerHyperband$clone()}}
}
}
\if{html}{
\out{<details open ><summary>Inherited methods</summary>}
\itemize{
\item \out{<span class="pkg-link" data-pkg="mlr3tuning" data-topic="Tuner" data-id="format">}\href{../../mlr3tuning/html/Tuner.html#method-format}{\code{mlr3tuning::Tuner$format()}}\out{</span>}
\item \out{<span class="pkg-link" data-pkg="mlr3tuning" data-topic="Tuner" data-id="optimize">}\href{../../mlr3tuning/html/Tuner.html#method-optimize}{\code{mlr3tuning::Tuner$optimize()}}\out{</span>}
\item \out{<span class="pkg-link" data-pkg="mlr3tuning" data-topic="Tuner" data-id="print">}\href{../../mlr3tuning/html/Tuner.html#method-print}{\code{mlr3tuning::Tuner$print()}}\out{</span>}
}
\out{</details>}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-new"></a>}}
\if{latex}{\out{\hypertarget{method-new}{}}}
\subsection{Method \code{new()}}{
Creates a new instance of this \link[R6:R6Class]{R6} class.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{TunerHyperband$new()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-clone"></a>}}
\if{latex}{\out{\hypertarget{method-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{TunerHyperband$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
