% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/es_from_stand_COR.R
\name{es_from_pearson_r}
\alias{es_from_pearson_r}
\title{Convert a Pearson's correlation coefficient to several effect size measures}
\usage{
es_from_pearson_r(
  pearson_r,
  sd_iv,
  n_sample,
  n_exp,
  n_nexp,
  cor_to_smd = "viechtbauer",
  unit_increase_iv,
  unit_type = "raw_scale",
  reverse_pearson_r
)
}
\arguments{
\item{pearson_r}{a Pearson's correlation coefficient value}

\item{sd_iv}{the standard deviation of the independent variable}

\item{n_sample}{the total number of participants}

\item{n_exp}{number of the experimental/exposed group}

\item{n_nexp}{number of the non-experimental/non-exposed group}

\item{cor_to_smd}{formula used to convert a \code{pearson_r} or \code{fisher_z} value into a SMD.}

\item{unit_increase_iv}{a value of the independent variable that will be used to estimate the Cohen's d (see details).}

\item{unit_type}{the type of unit for the \code{unit_increase_iv} argument. Must be either "sd" or "value"}

\item{reverse_pearson_r}{a logical value indicating whether the direction of the generated effect sizes should be flipped.}
}
\value{
This function estimates and converts between several effect size measures.

\tabular{ll}{
\code{natural effect size measure} \tab R + Z\cr
\tab \cr
\code{converted effect size measure} \tab D + G + OR\cr
\tab \cr
\code{required input data} \tab See 'Section 4. Pearson's r or Fisher's z'\cr
\tab https://metaconvert.org/html/input.html\cr
\tab \cr
}
}
\description{
Convert a Pearson's correlation coefficient to several effect size measures
}
\details{
This function estimates the variance of a Pearson's correlation coefficient, and computes the Fisher's r-to-z
transformation.
Cohen's d (D), Hedges' g (G) are converted from the Pearson's r, and odds ratio (OR)
are converted from the Cohen's d.
\enumerate{
\item \strong{The formula used to estimate the standard error of the Pearson's correlation coefficient} and 95\% CI
are (Formula 12.27 in Cooper):
\deqn{R\_se = \sqrt{\frac{(1 - pearson\_r^2)^2}{n\_sample - 1}}}
\deqn{R\_lo = pearson\_r - qt(.975, n\_sample - 2) * R\_se}
\deqn{R\_up = pearson\_r + qt(.975, n\_sample - 2) * R\_se}
\item \strong{The formula used to estimate the Fisher's z} are (Formula 12.28 & 12.29 in Cooper):
\deqn{Z = atanh(r)}
\deqn{Z\_se = \frac{1}{n\_sample - 3}}
\deqn{Z\_ci\_lo = Z - qnorm(.975) * Z\_se}
\deqn{Z\_ci\_up = Z + qnorm(.975) * Z\_se}
\item Several approaches can be used to convert a correlation coefficient to a SMD.
}

\strong{A.} Mathur proposes to use this formula (Formula 1.2 in Mathur, \code{cor_to_smd = "mathur"}):
\deqn{increase = ifelse(unit_type == "sd", unit\_increase\_iv * sd\_dv, unit\_increase\_iv)}
\deqn{d = \frac{r * increase}{sd_iv * \sqrt{1 - r^2}}}
\deqn{d\_se = abs(d) * \sqrt{\frac{1}{r^2 * (n\_sample - 3)} + \frac{1}{2*(n\_sample - 1))}}}
The resulting Cohen's d is the average increase in the dependent variable associated with an increase of x units in the independent variable (with x = \code{unit_increase_iv}).

\strong{B.} Viechtbauer proposes to use the delta method to derive a Cohen's d from a correlation coefficient (Viechtbauer, 2023, \code{cor_to_smd = "viechtbauer"})

\strong{C.} Cooper proposes to use this formula (Formula 12.38 & 12.39 in Cooper, \code{cor_to_smd = cooper}):
\deqn{increase = ifelse(unit_type == "sd", unit\_increase\_iv * sd\_dv, unit\_increase\_iv)}
\deqn{d = \frac{r * increase}{sd\_iv * \sqrt{1 - r^2}}}
\deqn{d\_se = abs(d) * \sqrt{\frac{1}{r^2 * (n\_sample - 3)} + \frac{1}{2*(n\_sample - 1))}}}
Note that this formula was initially proposed for converting a point-biserial correlation to
Cohen's d. It will thus produce similar results to the \code{cor_to_smd = "mathur"} option
only when \code{unit_type = "sd"} and \code{unit_increase_iv = 2}.

To know how the Cohen's d value is converted to other effect measures (G/OR), see details of the \code{\link{es_from_cohen_d}} function.
}
\examples{
es_from_pearson_r(
  pearson_r = .51, sd_iv = 0.24, n_sample = 214,
  unit_increase_iv = 1, unit_type = "sd"
)
}
\references{
Cooper, H., Hedges, L.V., & Valentine, J.C. (Eds.). (2019). The handbook of research synthesis and meta-analysis. Russell Sage Foundation.

Mathur, M. B., & VanderWeele, T. J. (2020). A Simple, Interpretable Conversion from Pearson's Correlation to Cohen's for d Continuous Exposures. Epidemiology (Cambridge, Mass.), 31(2), e16–e18. https://doi.org/10.1097/EDE.0000000000001105

Viechtbauer W (2010). “Conducting meta-analyses in R with the metafor package.” Journal of Statistical Software, 36(3), 1–48. doi:10.18637/jss.v036.i03.
}
