#' @title Thresholds for influenza intensity
#'
#' @description
#' Function \code{memintensity} is used to calculate the thresholds for influenza activity
#' using historical records (surveillance rates).\cr
#'
#' @name memintensity
#'
#' @param i.flu An object of class \code{mem}.
#'
#' @return
#' \code{memintensity} returns a list with three objects, two of them are the parameters
#' used (\code{param.levels} and \code{param.seasons}) and the third one
#' (\code{intensity.thresholds}) is a matrix 1x4 with the epidemic and intensity thresholds.
#' \enumerate{
#' \item Epidemic threshold.
#' \item Low intensity threshold.
#' \item Medium intensity threshold.
#' \item High intensity threshold.
#' }
#'
#' @details
#' This method is based on the Moving Epidemics Method (MEM) used to monitor influenza
#' activity in a weekly surveillance system.
#'
#' Input data is a data frame containing rates that represent historical influenza surveillance
#' data. It can start and end at any given week (tipically at week 40th), and rates can be
#' expressed as per 100,000 inhabitants (or per consultations, if population is not
#' available) or any other scale.
#'
#' MEM is used to locate the epidemic for each season. Then confidence intervals are
#' calculated at different levels.
#'
#' The parameter \code{i.levels}, define the three levels of the confidence intervals
#' used to calculate thresholds.
#'
#' The \code{i.n.max} parameter indicates how many epidemic values to use to calculate
#' the thresholds. A value of -1 indicates the program to use an appropiate number of
#' points depending on the number of seasons provided as input.
#'
#' The \code{i.seasons} parameter indicates how many seasons are used for calculating
#' thresholds. A value of -1 indicates the program to use as many as possible. If there
#' are less than this parameter, the program used all seasons avalaible.
#'
#' Intensity thresholds and Epidemic threshold defines 5 levels of intensity:
#'
#' \enumerate{
#' \item Baseline level - Below epidemic threshold.
#' \item Low level - Above epidemic threshold and below low intensity threshold.
#' \item Medium level - Above low intensity threshold and below medium intensity threshold.
#' \item High level - Above medium intensity threshold and below high intensity threshold.
#' \item Very high level - Above high intensity threshold.
#' }
#'
#' @examples
#' # Castilla y Leon Influenza Rates data
#' data(flucyl)
#' # mem model
#' flucyl.mem<-memmodel(flucyl)
#' # Calculates intensity thresholds
#' intensity<-memintensity(flucyl.mem)
#' intensity
#'
#' @author Jose E. Lozano \email{lozalojo@@gmail.com}
#'
#' @references
#' Vega T, Lozano JE, Ortiz de Lejarazu R, Gutierrez Perez M. Modelling influenza epidemic - can we
#' detect the beginning and predict the intensity and duration? Int Congr Ser. 2004 Jun;1263:281-3.
#'
#' Vega T, Lozano JE, Meerhoff T, Snacken R, Mott J, Ortiz de Lejarazu R, et al. Influenza surveillance
#' in Europe: establishing epidemic thresholds by the moving epidemic method. Influenza Other Respir
#' Viruses. 2013 Jul;7(4):546-58. DOI:10.1111/j.1750-2659.2012.00422.x.
#'
#' Vega T, Lozano JE, Meerhoff T, Snacken R, Beaute J, Jorgensen P, et al. Influenza surveillance in
#' Europe: comparing intensity levels calculated using the moving epidemic method. Influenza Other
#' Respir Viruses. 2015 Sep;9(5):234-46. DOI:10.1111/irv.12330.
#'
#' Lozano JE. lozalojo/mem: Second release of the MEM R library. Zenodo [Internet]. [cited 2017 Feb 1];
#' Available from: \url{https://zenodo.org/record/165983}. DOI:10.5281/zenodo.165983
#'
#' @keywords influenza
#'
#' @export
memintensity <- function(i.flu) {
  if (is.null(i.flu)) {
    memintensity.output <- NULL
  } else {
    intensity.thresholds <- matrix(c(i.flu$pre.post.intervals[1, 3], i.flu$epi.intervals[, 4]), ncol = 4)
    colnames(intensity.thresholds) <- c("Epidemic", paste(c("Medium (", "High (", "Very high ("), as.character(round(i.flu$epi.intervals[, 1] * 100, 1)), "%)", sep = ""))
    rownames(intensity.thresholds) <- "Intensity Thresholds"
    memintensity.output <- list(
      intensity.thresholds = intensity.thresholds,
      param.i.flu = i.flu
    )
    memintensity.output$call <- match.call()
  }
  return(memintensity.output)
}
