
#' Add gene tracks to a ggplot2 plot
#'
#' Adds gene tracks to an existing ggplot2 plot.
#'
#' @param p ggplot2 plot object. This can be generated by [gg_scatter()] and
#'   then modified.
#' @param loc Object of class 'locus' to use for plot. See [locus()].
#' @param heights Vector specifying ratio of heights of upper plot and lower
#'   gene track.
#' @param ... Additional arguments passed to [gg_genetracks()] to control
#'   colours of gene tracks etc.
#' @return A ggplot2 plotting object.
#' @seealso [gg_scatter()] [gg_genetracks()]
#' @examples
#' if(require(EnsDb.Hsapiens.v75)) {
#' data(SLE_gwas_sub)
#' loc <- locus(SLE_gwas_sub, gene = 'IRF5', flank = c(7e4, 2e5), LD = "r2",
#'              ens_db = "EnsDb.Hsapiens.v75")
#' p <- gg_scatter(loc)
#' gg_addgenes(p, loc)
#' }
#' @importFrom ggplot2 layer_scales
#' @importFrom cowplot plot_grid
#' @export

gg_addgenes <- function(p, loc,
                        heights = c(3, 2),
                        ...) {
  # check x axis range
  xl <- layer_scales(p)$x$get_limits() * 1e6
  if (!identical(as.numeric(loc$xrange), xl)) {
    message("Plot x axis limits and locus range differ: check correct locus?")
    message("x axis limits: ", xl[1], " to ", xl[2])
    message("locus range: ", loc$xrange[1], " to ", loc$xrange[2])
  }
  g <- gg_genetracks(loc, ...)
  
  plot_grid(p, g, nrow = 2, rel_heights = heights, align = "v")
}
