% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/estimators.R
\name{lmtp_ipw}
\alias{lmtp_ipw}
\title{LMTP IPW Estimator}
\usage{
lmtp_ipw(
  data,
  trt,
  outcome,
  baseline = NULL,
  time_vary = NULL,
  cens = NULL,
  k = Inf,
  id = NULL,
  shift,
  outcome_type = c("binomial", "continuous", "survival"),
  learners = "SL.glm",
  folds = 10,
  weights = NULL,
  return_all_ratios = FALSE,
  .bound = 1e-05,
  .trim = 0.999,
  .SL_folds = 10
)
}
\arguments{
\item{data}{A data frame in wide format containing all necessary variables
for the estimation problem.}

\item{trt}{A vector containing the column names of treatment variables ordered by time.}

\item{outcome}{The column name of the outcome variable. In the case of time-to-event
analysis, a vector containing the columns names of intermediate outcome variables and the final
outcome variable ordered by time. Only numeric values are allowed. If the outcome type
is binary, data should be coded as 0 and 1.}

\item{baseline}{An optional vector containing the column names of baseline covariates to be
included for adjustment at every time point.}

\item{time_vary}{A list the same length as the number of time points of observation with
the column names for new time-varying covariates introduced at each time point. The list
should be ordered following the time ordering of the model.}

\item{cens}{An optional vector of column names of censoring indicators the same
length as the number of time points of observation. If missingness in the outcome is
present or if time-to-event outcome, must be provided.}

\item{k}{An integer specifying how previous time points should be
used for estimation at the given time point. Default is \code{Inf},
all time points.}

\item{id}{An optional column name containing cluster level identifiers.}

\item{shift}{A two argument function that specifies how treatment variables should be shifted.
See examples for how to specify shift functions for continuous, binary, and categorical exposures.}

\item{outcome_type}{Outcome variable type (i.e., continuous, binomial, survival).}

\item{learners}{A vector of \code{SuperLearner} algorithms for estimation
of the exposure mechanism. Default is \code{"SL.glm"}, a main effects GLM.}

\item{folds}{The number of folds to be used for cross-fitting. Minimum allowable number
is two folds.}

\item{weights}{An optional vector of length n containing sampling weights.}

\item{return_all_ratios}{Logical. If \code{TRUE}, the non-cumulative product density
ratios will be returned. The default is \code{FALSE}.}

\item{.bound}{Determines that maximum and minimum values (scaled) predictions
will be bounded by. The default is 1e-5, bounding predictions by 1e-5 and 0.9999.}

\item{.trim}{Determines the amount the density ratios should be trimmed.
The default is 0.999, trimming the density ratios greater than the 0.999 percentile
to the 0.999 percentile. A value of 1 indicates no trimming.}

\item{.SL_folds}{Integer. Controls the number of splits to be used for fitting
the Super Learner. The default is 10.}
}
\value{
A list of class \code{lmtp} containing the following components:

\item{estimator}{The estimator used, in this case "IPW".}
\item{theta}{The estimated population LMTP effect.}
\item{standard_error}{NA}
\item{low}{NA}
\item{high}{NA}
\item{shift}{The shift function specifying the treatment policy of interest.}
\item{density_ratios}{An n x Tau matrix of the estimated density ratios.}
\item{raw_ratios}{An n x Tau matrix of the estimated non-cumulative product density ratios.
\code{NULL} if \code{return_all_ratios = FALSE}.}
\item{weights_r}{A list the same length as \code{folds}, containing the Super Learner
ensemble weights at each time-point for each fold for the propensity.}
}
\description{
Inverse probability of treatment weighting estimator for the effects of traditional causal
effects and modified treatment policies for both point treatment and longitudinal data
with binary, continuous, or time-to-event outcomes. Supports binary, categorical, and continuous exposures.
}
\examples{
\donttest{

# Example 1.1
# Point treatment, continuous exposure, continuous outcome, no loss-to-follow-up
# Interested in the effect of a population wide decrease in A of 5 units
# The true value under this intervention is about 519.
set.seed(56)
n <- 1000
W <- rnorm(n, 10, 5)
A <- 23 + 5*W + rnorm(n)
Y <- 7.2*A + 3*W + rnorm(n)
ex1_dat <- data.frame(W, A, Y)
d <- function(data, x) data[[x]] - 5
psi1.1 <- lmtp_ipw(ex1_dat, "A", "Y", baseline = "W", shift = d, folds = 2,
                   outcome_type = "continuous", .trim = 0.9975)
psi1.1

# Example 1.2
# Point treatment, continuous exposure, continuous outcome, no loss-to-follow-up
# Interested in the effect of a modified treatment policy where A is decreased by 15
# units only among observations whose observed A was above 80.
# The true value under this intervention is about 513.
d <- function(data, x) (data[[x]] > 80)*(data[[x]] - 15) + (data[[x]] <= 80)*data[[x]]
psi1.2 <- lmtp_ipw(ex1_dat, "A", "Y", baseline = "W", shift = d, folds = 2,
                   outcome_type = "continuous")
psi1.2

# Example 2.1
# Longitudinal setting, time-varying continuous exposure bounded by 0,
# time-varying covariates, and a binary outcome with no loss-to-follow-up.
# Interested in the effect of a treatment policy where exposure decreases by
# one unit at every time point if an observations observed exposure is greater
# than or equal to 2. The true value under this intervention is about 0.305.
head(sim_t4)
# specifying treament variables
a <- c("A_1", "A_2", "A_3", "A_4")
# specifying time varying covariates
tv <- list(c("L_1"), c("L_2"), c("L_3"), c("L_4"))
# treatment policy function to be applied at all time points
d <- function(data, trt) {
  a <- data[[trt]]
  (a - 1) * (a - 1 >= 1) + a * (a - 1 < 1)
}
progressr::with_progress({
  psi2.1 <- lmtp_ipw(sim_t4, a, "Y", time_vary = tv, shift = d, folds = 2)
})
psi2.1

# Example 2.2
# Example 2.1 assumed that the outcome (as well as the treatment variables)
# were directly affected by all other nodes in the past. In certain situtations,
# domain specific knowledge may suggest otherwise leading to a Markov processes.
# This can be controlled using the k argument.
progressr::with_progress({
  psi2.2 <- lmtp_ipw(sim_t4, a, "Y", time_vary = tv, shift = d,
                     k = 0, folds = 2)
})
psi2.2

# Example 2.3
# Using the same data as examples 2.1 and 2.2.
# Now estimating the effect of a dynamic modified treatment policy.
a <- c("A_1", "A_2", "A_3", "A_4")
time_varying <- list(c("L_1"), c("L_2"), c("L_3"), c("L_4"))

# creating a dynamic mtp that applies the shift function
# but also depends on history and the current time
dynamic_mtp <- function(data, trt) {
  mtp <- function(data, trt) {
    (data[[trt]] - 1) * (data[[trt]] - 1 >= 1) + data[[trt]] * (data[[trt]] - 1 < 1)
  }

  # if its the first time point, follow the same mtp as before
  if (trt == "A_1") return(mtp(data, trt))

  # otherwise check if the time varying covariate equals 1
  ifelse(
    data[[sub("A", "L", trt)]] == 1,
    mtp(data, trt), # if yes continue with the policy
    data[[trt]]     # otherwise do nothing
  )
}
psi2.3 <- lmtp_ipw(sim_t4, a, "Y", time_vary = time_varying,
                   k = 0, shift = dynamic_mtp, folds = 2)
psi2.3

# Example 2.4
# Using the same data as examples 2.1, 2.2, and 2.3, but now treating the exposure
# as an ordered categorical variable. To account for the exposure being a
# factor we just need to modify the shift function (and the original data)
# so as to respect this.
for (i in a) {
  sim_t4[[i]] <- factor(sim_t4[[i]], levels = 0:5, ordered = TRUE)
}
d <- function(data, trt) {
  out <- list()
  a <- data[[trt]]
  for (i in 1:length(a)) {
    if (as.character(a[i]) \%in\% c("0", "1")) {
      out[[i]] <- as.character(a[i])
    } else {
      out[[i]] <- as.numeric(as.character(a[i])) - 1
    }
  }
  factor(unlist(out), levels = 0:5, ordered = TRUE)
}
progressr::with_progress({
  psi2.4 <- lmtp_ipw(sim_t4, a, "Y", time_vary = tv, shift = d, k = 0, folds = 2)
})
psi2.4

# Example 3.1
# Longitudinal setting, time-varying binary treatment, time-varying covariates
# and baseline covariates with no loss-to-follow-up. Interested in a traditional
# causal effect where treatment is set to 1 at all time points for all observations.
if (require("twang")) {
  data("iptwExWide", package = "twang")
  a <- paste0("tx", 1:3)
  baseline <- c("gender", "age")
  tv <- list(c("use0"), c("use1"), c("use2"))
  progressr::with_progress({
    psi3.1 <-
      lmtp_ipw(iptwExWide, a, "outcome", baseline = baseline, time_vary = tv,
               shift = static_binary_on, folds = 2, outcome_type = "continuous")
  })
  psi3.1
}

# Example 4.1
# Longitudinal setting, time-varying continuous treatment, time-varying covariates,
# binary outcome with right censoring. Interested in the mean population outcome under
# the observed exposures in a hypothetical population with no loss-to-follow-up.
head(sim_cens)
a <- c("A1", "A2")
tv <- list(c("L1"), c("L2"))
cens <- c("C1", "C2")
y <- "Y"
psi4.1 <- lmtp_ipw(sim_cens, a, y, time_vary = tv, cens = cens,
                   shift = NULL, folds = 2)
psi4.1

# Example 4.2
# Using the same data as example 4.1, but now interested in the causal effect of a
# treatment policy where exposure increased by 0.5 units at all time points. The
# true value under this intervention is about 0.88.
d <- function(data, x) data[[x]] + 0.5
psi4.2 <- lmtp_ipw(sim_cens, a, y, time_vary = tv,
                   cens = cens, shift = d, folds = 2)
psi4.2

# Example 5.1
# Time-to-event analysis with a binary time-invariant exposure. Interested in
# the effect of treatment being given to all observations on the probability of being event
# free at the end of follow-up.
a <- "trt"
# for a survival problem, the outcome argument now takes a vector of outcomes
# if an observation experiences the event prior to the end of follow-up, all future
# outcome nodes should be set to 1 (i.e., last observation carried forward).
y <- paste0("Y.", 1:6)
cens <- paste0("C.", 0:5)
baseline <- c("W1", "W2")
progressr::with_progress({
  psi5.1 <- lmtp_ipw(sim_point_surv, a, y, baseline, cens = cens,
                     shift = static_binary_on, folds = 2,
                     outcome_type = "survival")
})
psi5.1
}
}
