% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/grid_terrain.R
\name{grid_terrain}
\alias{grid_terrain}
\title{Digital Terrain Model}
\usage{
grid_terrain(
  las,
  res = 1,
  algorithm,
  ...,
  keep_lowest = FALSE,
  full_raster = FALSE,
  use_class = c(2L, 9L),
  Wdegenerated = TRUE,
  is_concave = FALSE
)
}
\arguments{
\item{las}{An object of class \link[lidR:LAS-class]{LAS} or \link[lidR:LAScatalog-class]{LAScatalog}.}

\item{res}{numeric. The resolution of the output \code{Raster}. Can optionally be a \code{RasterLayer}.
In that case the \code{RasterLayer} is used as the layout.}

\item{algorithm}{function. A function that implements an algorithm to compute spatial interpolation.
\code{lidR} implements \link{knnidw}, \link{tin}, and \link{kriging} (see respective documentation and examples).}

\item{...}{Unused}

\item{keep_lowest}{logical. This option forces the original lowest ground point of each
cell (if it exists) to be chosen instead of the interpolated values.}

\item{full_raster}{logical. By default the interpolation is made only within the convex hull of
the point cloud. This prevents meaningless interpolations where there is no data. If TRUE,
each pixel of the raster is interpolated.}

\item{use_class}{integer vector. By default the terrain is computed by using ground points
(class 2) and water points (class 9).}

\item{Wdegenerated}{logical. The function always checks and removes degenerated ground points
for computing the DTM to avoid unexpected behaviours, such as infinite elevation. If
TRUE, a warning is thrown to alert users to the presence of degenerated ground points.}

\item{is_concave}{boolean. By default the function tries to compute a DTM that
has the same shape as the point cloud by interpolating only in the convex
hull of the points. If the point cloud is concave this may lead to weird values
where there are no points. Use \code{is_concave = TRUE} to use a concave hull.
This is more computationally demanding. It uses \link{concaveman} internally.}
}
\value{
A \code{RasterLayer} containing a numeric value in each cell. If the \code{RasterLayer}s
are written on disk when running the function with a \code{LAScatalog}, a virtual raster mosaic is
returned (see \href{https://gdal.org/programs/gdalbuildvrt.html}{gdalbuildvrt})
}
\description{
Interpolates the ground points and creates a rasterized digital terrain model. The algorithm uses
the points classified as "ground" and "water" (Classification = 2 and 9, respectively, according to
\href{https://www.asprs.org/wp-content/uploads/2019/07/LAS_1_4_r15.pdf}{LAS file format specifications})
to compute the interpolation.\cr\cr
How well the edges of the dataset are interpolated depends on the interpolation method used.
A buffer around the region of interest is always recommended to avoid edge effects.
}
\section{Working with a \code{LAScatalog}}{

This section appears in each function that supports a LAScatalog as input.\cr

In \code{lidR} when the input of a function is a \link[lidR:LAScatalog-class]{LAScatalog} the
function uses the LAScatalog processing engine. The user can modify the engine options using
the \link[lidR:catalog_options_tools]{available options}. A careful reading of the
\link[lidR:LAScatalog-class]{engine documentation} is recommended before processing \code{LAScatalogs}. Each
\code{lidR} function should come with a section that documents the supported engine options.\cr

The \code{LAScatalog} engine supports \code{.lax} files that \emph{significantly} improve the computation
speed of spatial queries using a spatial index. Users should really take advantage a \code{.lax} files,
but this is not mandatory.\cr
}

\section{Supported processing options}{

Supported processing options for a \code{LAScatalog} in \code{grid_*} functions (in bold). For
more details see the \link[lidR:LAScatalog-class]{LAScatalog engine documentation}:
\itemize{
\item \strong{chunk size}: How much data is loaded at once. The chunk size may be slightly modified
internally to ensure a strict continuous wall-to-wall output even when chunk size is equal to 0
(processing by file).
\item \strong{chunk buffer}: This function guarantees a strict continuous wall-to-wall output. The
\code{buffer} option is not considered.
\item \strong{chunk alignment}: Align the processed chunks. The alignment may be slightly modified
internally to ensure a strict continuous wall-to-wall output.
\item \strong{progress}: Displays a progress estimate.
\item \strong{output files}: Return the output in R or write each cluster's output in a file.
Supported templates are \code{\{XLEFT\}}, \code{\{XRIGHT\}}, \code{\{YBOTTOM\}}, \code{\{YTOP\}},
\code{\{XCENTER\}}, \code{\{YCENTER\}} \code{\{ID\}} and, if chunk size is equal to 0 (processing
by file), \code{\{ORIGINALFILENAME\}}.
\item select: The function 'knows' what should be loaded and this option is not considered.
\item \strong{filter}: Read only the points of interest.
}
}

\examples{

LASfile <- system.file("extdata", "Topography.laz", package="lidR")
las = readLAS(LASfile, filter = "-inside 273450 5274350 273550 5274450")
#plot(las)

dtm1 = grid_terrain(las, algorithm = knnidw(k = 6L, p = 2))
dtm2 = grid_terrain(las, algorithm = tin())

\dontrun{
dtm3 = grid_terrain(las, algorithm = kriging(k = 10L))

plot(dtm1)
plot(dtm2)
plot(dtm3)
plot_dtm3d(dtm1)
plot_dtm3d(dtm2)
plot_dtm3d(dtm3)
}
}
