% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/clip_roi.R
\name{clip}
\alias{clip}
\alias{clip_roi}
\alias{clip_rectangle}
\alias{clip_polygon}
\alias{clip_circle}
\alias{clip_transect}
\title{Clip points in regions of interest}
\usage{
clip_roi(las, geometry, ...)

clip_rectangle(las, xleft, ybottom, xright, ytop, ...)

clip_polygon(las, xpoly, ypoly, ...)

clip_circle(las, xcenter, ycenter, radius, ...)

clip_transect(las, p1, p2, width, xz = FALSE, ...)
}
\arguments{
\item{las}{An object of class \link[lidR:LAS-class]{LAS} or \link[lidR:LAScatalog-class]{LAScatalog}.}

\item{geometry}{a geometric object. Many types are supported, see section 'supported geometries'.}

\item{...}{in \code{clip_roi}: optional supplementary options (see supported geometries). Unused in
other functions}

\item{xleft}{numeric. left x coordinates of rectangles.}

\item{ybottom}{numeric. bottom y coordinates of rectangles.}

\item{xright}{numeric. right x coordinates of rectangles.}

\item{ytop}{numeric. top y coordinates of rectangles.}

\item{xpoly}{numeric. x coordinates of a polygon.}

\item{ypoly}{numeric. y coordinates of a polygon.}

\item{xcenter}{numeric. x coordinates of disc centers.}

\item{ycenter}{numeric. y coordinates of disc centers.}

\item{radius}{numeric. disc radius or radii.}

\item{p1, p2}{numeric vectors of length 2 that gives the coordinates of two points that define a
transect}

\item{width}{numeric. width of the transect.}

\item{xz}{bool. If \code{TRUE} the point cloud is reoriented to fit on XZ coordinates}
}
\value{
If the input is a \code{LAS} object: an object of class \code{LAS}, or a \code{list} of \code{LAS} objects if the query implies several regions of interest will be returned.\cr\cr
If the input is a \code{LAScatalog} object: an object of class \code{LAS}, or a \code{list} of \code{LAS}
objects if the query implies several regions of interest will be returned, or a \code{LAScatalog} if the
queries are immediately written into files without loading anything in R.
}
\description{
Clip points within a given region of interest (ROI) from a point cloud (\code{LAS} object) or a catalog
(\code{LAScatalog} object). With a \code{LAS} object, the user first reads and loads a point cloud
into memory and then can clip it to get a subset within a region of interest. With a \code{LAScatalog}
object, the user can extract any arbitrary ROI for a set of \code{las/laz} files, loading only the
points of interest. This is faster, easier and much more memory-efficient for extracting ROIs.
}
\section{Supported geometries}{

\itemize{
 \item \href{https://en.wikipedia.org/wiki/Well-known_text}{WKT string}: describing a POINT, a POLYGON or
 a MULTIPOLYGON. If points, a parameter 'radius' must be passed in \code{...}
 \item \link[sp:Polygon-class]{Polygon} or \link[sp:Polygons-class]{Polygons}
 \item \link[sp:SpatialPolygons-class]{SpatialPolygons} or \link[sp:SpatialPolygonsDataFrame-class]{SpatialPolygonsDataFrame}
 \item \link[sp:SpatialPoints-class]{SpatialPoints} or \link[sp:SpatialPointsDataFrame-class]{SpatialPointsDataFrame}
 in that case a parameter 'radius' must be passed in \code{...}
 \item \link[sf:sf]{SimpleFeature} that consistently contains \code{POINT} or \code{POLYGON/MULTIPOLYGON}.
 In case of \code{POINT} a parameter 'radius' must be passed in \code{...}
 \item \link[raster:Extent-class]{Extent}
 \item \link[base:matrix]{matrix} 2 x 2 describing a bounding box following this order:
 \preformatted{
  min     max
x 684816  684943
y 5017823 5017957}
 }
}

\section{Working with a \code{LAScatalog}}{

This section appears in each function that supports a LAScatalog as input.\cr

In \code{lidR} when the input of a function is a \link[lidR:LAScatalog-class]{LAScatalog} the
function uses the LAScatalog processing engine. The user can modify the engine options using
the \link[lidR:catalog_options_tools]{available options}. A careful reading of the
\link[lidR:LAScatalog-class]{engine documentation} is recommended before processing \code{LAScatalogs}. Each
\code{lidR} function should come with a section that documents the supported engine options.\cr

The \code{LAScatalog} engine supports \code{.lax} files that \emph{significantly} improve the computation
speed of spatial queries using a spatial index. Users should really take advantage a \code{.lax} files,
but this is not mandatory.\cr
}

\section{Supported processing options}{

Supported processing options for a \code{LAScatalog} (in bold). For more details see the
\link[=LAScatalog-class]{LAScatalog engine documentation}:
\itemize{
\item chunk_size: Does not make sense here.
\item buffer: Not supported yet.
\item alignment: Does not makes sense here.
\item \strong{progress}: Displays a progress estimation.
\item \strong{output_files}: If 'output_files' is set in the catalog, the ROIs will not be returned in R.
They will be written immediately in files. See \link{LAScatalog-class} and examples. The allowed templates in
\code{clip_*} are \code{{XLEFT}, {XRIGHT}, {YBOTTOM}, {YTOP}, {ID}, {XCENTER},
{YCENTER}}. In addition \code{clip_roi} supports any names from the table of attributes of a spatial object given as
input such as \code{{PLOTID}}, \code{{YEAR}}, \code{{SPECIES}}, for examples, if these attributes exist. If empty everything
is returned into R.
\item \strong{laz_compression}: write \code{las} or \code{laz} files
\item select: The function will write files equivalent to the originals. This option is not respected.
\item \strong{filter}: Read only the points of interest.
}
}

\examples{
LASfile <- system.file("extdata", "Megaplot.laz", package="lidR")

# Load the file and clip the region of interest
las = readLAS(LASfile, select = "xyz", filter = "-keep_first")
subset1 = clip_rectangle(las, 684850, 5017850, 684900, 5017900)

# Do not load the file(s), extract only the region of interest
# from a bigger dataset
ctg = readLAScatalog(LASfile, progress = FALSE, filter = "-keep_first")
subset2 = clip_rectangle(ctg, 684850, 5017850, 684900, 5017900)

# Extract all the polygons from a shapefile
f <- system.file("extdata", "lake_polygons_UTM17.shp", package = "lidR")
lakes <- sf::st_read(f, quiet = TRUE)
subset3 <- clip_roi(las, lakes)

# Extract the polygons for a catalog, write them in files named
# after the lake names, do not load anything in R
opt_output_files(ctg) <- paste0(tempfile(), "_{LAKENAME_1}")
new_ctg = clip_roi(ctg, lakes)
plot(new_ctg)

# Extract a transect
p1 <- c(684800, y = 5017800)
p2 <- c(684900, y = 5017900)
tr1 <- clip_transect(las, p1, p2, width = 4)

\dontrun{
plot(subset1)
plot(subset2)
plot(subset3)

plot(tr1, axis = TRUE, clear_artifacts = FALSE)
}
}
