% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/utils_metrics.r
\docType{data}
\name{stdmetrics}
\alias{stdmetrics}
\alias{.stdmetrics}
\alias{stdmetrics_z}
\alias{.stdmetrics_z}
\alias{stdmetrics_i}
\alias{.stdmetrics_i}
\alias{stdmetrics_rn}
\alias{.stdmetrics_rn}
\alias{stdmetrics_pulse}
\alias{.stdmetrics_pulse}
\alias{stdmetrics_ctrl}
\alias{.stdmetrics_ctrl}
\alias{stdtreemetrics}
\alias{.stdtreemetrics}
\title{Predefined standard metrics functions}
\format{An object of class \code{formula} of length 2.}
\usage{
stdmetrics(x, y, z, i, rn, class, dz = 1, th = 2)

.stdmetrics

stdmetrics_z(z, dz = 1, th = 2)

.stdmetrics_z

stdmetrics_i(i, z = NULL, class = NULL, rn = NULL)

.stdmetrics_i

stdmetrics_rn(rn, class = NULL)

.stdmetrics_rn

stdmetrics_pulse(pulseID, rn)

.stdmetrics_pulse

stdmetrics_ctrl(x, y, z)

.stdmetrics_ctrl

stdtreemetrics(x, y, z)

.stdtreemetrics
}
\arguments{
\item{x, y, z, i}{Coordinates of the points, Intensity}

\item{rn, class}{ReturnNumber, Classification}

\item{dz}{numeric. Layer thickness  metric \link[lidR:entropy]{entropy}}

\item{th}{numeric. Threshold for metrics pzabovex. Can be a vector to compute with several thresholds.}

\item{pulseID}{The number referencing each pulse}
}
\description{
Predefined functions usable in \link{grid_metrics}, \link{grid_hexametrics}, \link{lasmetrics}, \link{tree_metrics},
and their convenient shortcuts. The philosophy of the \code{lidR} package is to provide an easy way
to compute user-defined metrics rather than to provide them. However, for efficiency and to save time, a set of
standard metrics has been predefined. To use these functions please read the details and examples sections.
}
\details{
The function names, their parameters and the output names of the metrics rely on a nomenclature chosen for brevity:
\itemize{
\item{\code{z}: refers to the elevation}
\item{\code{i}: refers to the intensity}
\item{\code{rn}: refers to the return number}
\item{\code{q}: refers to quantile}
\item{\code{a}: refers to the ScanAngleRank or ScanAngle}
\item{\code{n}: refers to a number (a count)}
\item{\code{p}: refers to a percentage}
}
For example the metric named \code{zq60} refers to the elevation, quantile, 60 i.e. the 60th percentile of elevations.
The metric \code{pground} refers to a percentage. It is the percentage of points classified as ground.
The function \code{stdmetric_i} refers to metrics of intensity. A description of each existing metric can be found
on the \href{https://github.com/Jean-Romain/lidR/wiki/stdmetrics}{lidR wiki page}.\cr\cr
Some functions have optional parameters. If these parameters are not provided the function
computes only a subset of existing metrics. For example, \code{stdmetrics_i} requires the intensity
values, but if the elevation values are also provided it can compute additional metrics such as cumulative
intensity at a given percentile of height.\cr\cr
Each function has a convenient associated variable. It is the name of the function, with a
dot before the name. This enables the function to be used without writing parameters. The cost
of such a feature is inflexibility. It corresponds to a predefined behavior (see examples)\cr\cr
\code{stdtreemetrics} is a special function that works with \link{tree_metrics}. Actually,
it won't fail with other functions but the output makes more sense if computed at the
individual tree level.
}
\examples{
LASfile <- system.file("extdata", "Megaplot.laz", package="lidR")
las = readLAS(LASfile, select = "*")

# All the predefined metrics
m1 = grid_metrics(las, stdmetrics(X,Y,Z,Intensity,ReturnNumber,Classification,dz=1))

# Convenient shortcut
m2 = grid_metrics(las, .stdmetrics)

# Basic metrics from intensities
m3 = grid_metrics(las, stdmetrics_i(Intensity))

# All the metrics from intensities
m4 = grid_metrics(las, stdmetrics_i(Intensity, Z, Classification, ReturnNumber))

# Convenient shortcut for the previous example
m5 = grid_metrics(las, .stdmetrics_i)

# Compute the metrics only on first return
first = lasfilterfirst(las)
m6 = grid_metrics(first, .stdmetrics_z)

# Compute the metrics with a threshold at 2 meters
over2 = lasfilter(las, Z > 2)
m7 = grid_metrics(over2, .stdmetrics_z)

# Works also with lasmetrics and grid_hexametrics
m8 = lasmetrics(las, .stdmetrics)
m9 = grid_hexametrics(las, .stdmetrics)

# Combine some predefined function with your own new metrics
# Here convenient shortcuts are no longer usable.
myMetrics = function(z, i, rn)
{
  first  = rn == 1L
  zfirst = z[first]
  nfirst = length(zfirst)
  above2 = sum(z > 2)

  x = above2/nfirst*100

  # User's metrics
  metrics = list(
     above2aboven1st = x,       # Num of returns above 2 divided by num of 1st returns
     zimean  = mean(z*i),       # Mean products of z by intensity
     zsqmean = sqrt(mean(z^2))  # Quadratic mean of z
   )

  # Combined with standard metrics
  return( c(metrics, stdmetrics_z(z)) )
}

m10 = grid_metrics(las, myMetrics(Z, Intensity, ReturnNumber))

# Users can write their own convenient shorcuts like this:
.myMetrics = ~myMetrics(Z, Intensity, ReturnNumber)

m11 = grid_metrics(las, .myMetrics)
}
\seealso{
\link{grid_metrics}
\link{lasmetrics}
\link{grid_hexametrics}
\link{grid_metrics3d}
\link{tree_metrics}
}
\keyword{datasets}
