% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/texmex.R
\name{texmex}
\alias{texmex}
\alias{alogLik.evmOpt}
\title{Loglikelihood adjustment of texmex fits}
\usage{
\method{alogLik}{evmOpt}(x, cluster = NULL, use_vcov = TRUE, ...)
}
\arguments{
\item{x}{A fitted model object with certain associated S3 methods.
See \strong{Details}.}

\item{cluster}{A vector or factor indicating from which cluster the
respective loglikelihood contributions from \code{loglik} originate.
This must have the same length as the vector returned by the
\code{logLikVec} method for an object like \code{x}.
If \code{cluster} is not supplied (i.e. is \code{NULL}) then it is
assumed that each observation forms its own cluster.
See \strong{Details}.}

\item{use_vcov}{A logical scalar.  Should we use the \code{vcov} S3 method
for \code{x} (if this exists) to estimate the Hessian of the independence
loglikelihood to be passed as the argument \code{H} to
\code{\link[chandwich]{adjust_loglik}}?
Otherwise, \code{H} is estimated inside
\code{\link[chandwich]{adjust_loglik}} using
\code{\link[stats:optim]{optimHess}}.}

\item{...}{Further arguments to be passed to the functions in the
sandwich package \code{\link[sandwich]{meat}} (if \code{cluster = NULL}),
or \code{\link[sandwich:vcovCL]{meatCL}} (if \code{cluster} is not
\code{NULL}).}
}
\value{
An object inheriting from class \code{"chandwich"}.  See
  \code{\link[chandwich]{adjust_loglik}}.
  \code{class(x)} is a vector of length 5. The first 3 components are
  \code{c("lax", "chandwich", "texmex")}.
  The remaining 2 components depend on the model that was fitted.
  The 4th component is: \code{"gev"} if \code{x$family$name = "GEV"};
  \code{"gpd"} if \code{x$family$name = "GPD"};
  \code{"egp3"} if \code{x$family$name = "EGP3"}.
  The 5th component is
  \code{"stat"} if there are no covariates in the mode and
  \code{"nonstat"} otherwise.
}
\description{
S3 \code{alogLik} method to perform loglikelihood adjustment of fitted
extreme value model objects returned from the \code{\link[texmex]{evm}}
function in the \code{\link[texmex:texmex-package]{texmex}} package.
The model must have been fitted using maximum likelihood estimation.
}
\details{
Object \code{x} \emph{must} have the following S3
  methods:
  \itemize{
    \item{\code{logLikVec: }}{returns a vector of the contributions to the
      independence loglikelihood from individual observations;}
    \item{\code{coef: }}{returns a vector of model coefficients, see
      \code{\link[stats]{coef}};}
    \item{\code{nobs: }}{returns the number of (non-missing) observations
      used in a model fit, see \code{\link[stats]{nobs}}};
  }
  and \emph{may} have the following S3 methods
  \itemize{
    \item{\code{vcov: }}{returns the estimated variance-covariance matrix of
      the (main) parameters of a fitted model, see
      \code{\link[stats]{vcov}};}
    \item{\code{estfun: }}{returns an \eqn{n x k} matrix, in which each
      column gives the derivative of the loglikelihood at each of \eqn{n}
      observation with respect to the \eqn{k} parameters of the model, see
      \code{\link[sandwich]{estfun}}.}
  }
  Loglikelihood adjustment is performed using the
  \code{\link[chandwich]{adjust_loglik}} function in the
  \code{\link[chandwich]{chandwich}} package.
  The relevant arguments to \code{\link[chandwich]{adjust_loglik}}, namely
  \code{loglik, mle, H} and \code{V}, are created based on the class of
  the object \code{x}.

  If a \code{vcov} method is not available, or if \code{use_vcov = FALSE},
  then the variance-covariance matrix of the MLE (from which \code{H} is
  calculated) is estimated inside \code{\link[chandwich]{adjust_loglik}}
  using \code{\link[stats:optim]{optimHess}}.

  The \code{sandwich} package is used to estimate the variance matrix
  \code{V} of the score vector: \code{\link[sandwich]{meat}} is used if
  \code{cluster = NULL}; \code{\link[sandwich:vcovCL]{meatCL}} is used if
  \code{cluster} is not \code{NULL}.
  If \code{cluster} is \code{NULL} then any arguments of
  \code{\link[sandwich:vcovCL]{meatCL}} present in \dots will be ignored.
  Similarly, if \code{cluster} is not \code{NULL} then any arguments of
  \code{\link[sandwich]{meat}} present in \dots will be ignored.
  \code{\link[sandwich]{meat}} and \code{\link[sandwich:vcovCL]{meatCL}}
  require an \code{\link[sandwich]{estfun}} method to be available, which,
  in the current context, provides matrix of score contributions.
  If a bespoke \code{estfun} method is not provided then this is constructed
  by estimating the score contributions using \code{\link[numDeriv]{jacobian}}.
}
\examples{
# We need the texmex package, and ismev for the fremantle dataset
got_texmex <- requireNamespace("texmex", quietly = TRUE)
got_ismev <- requireNamespace("ismev", quietly = TRUE)
if (got_texmex) {
  library(texmex)
  # Examples from the texmex::evm documentation

  # GEV
  mod <- evm(SeaLevel, data = texmex::portpirie, family = gev)
  adj_mod <- alogLik(mod)
  summary(adj_mod)

  # GP
  mod <- evm(rain, th = 30)
  adj_mod <- alogLik(mod)
  summary(adj_mod)
  mod <- evm(rain, th = 30, cov = "sandwich")
  mod$se
  vcov(adj_mod)
  vcov(mod)

  # EGP3
  mod <- evm(rain, th = 30, family = egp3)
  adj_mod <- alogLik(mod)
  summary(adj_mod)

  # GP regression
  # An example from page 119 of Coles (2001)
  n_rain <- length(rain)
  rain_df <- data.frame(rain = rain, time = 1:n_rain / n_rain)
  evm_fit <- evm(y = rain, data = rain_df, family = gpd, th = 30,
                 phi = ~ time)
  adj_evm_fit <- alogLik(evm_fit)
  summary(adj_evm_fit)
  evm_fit <- evm(y = rain, data = rain_df, family = gpd, th = 30,
                 phi = ~ time, cov = "sandwich")
  evm_fit$se
  vcov(adj_evm_fit)
  vcov(evm_fit)

  # GEV regression
  # An example from page 113 of Coles (2001)
  if (got_ismev) {
    library(ismev)
    data(fremantle)
    new_fremantle <- fremantle
    # Set year 1897 to 1 for consistency with page 113 of Coles (2001)
    new_fremantle[, "Year"] <- new_fremantle[, "Year"] - 1896
    evm_fit <- evm(y = SeaLevel, data = new_fremantle, family = gev,
                   mu = ~ Year + SOI)
    adj_evm_fit <- alogLik(evm_fit)
    summary(adj_evm_fit)
  }

  # An example from Chandler and Bate (2007)
  # Note: evm uses phi = log(sigma)
  evm_fit <- evm(temp, ow, gev, mu = ~ loc, phi = ~ loc, xi = ~loc)
  adj_evm_fit <- alogLik(evm_fit, cluster = ow$year, cadjust = FALSE)
  summary(adj_evm_fit)
}
}
\references{
Chandler, R. E. and Bate, S. (2007). Inference for clustered
  data using the independence loglikelihood. \emph{Biometrika},
  \strong{94}(1), 167-183. \url{http://doi.org/10.1093/biomet/asm015}

Zeleis (2006) Object-Oriented Computation and Sandwich
  Estimators.  \emph{Journal of Statistical Software}, \strong{16}, 1-16.
  \url{http://doi.org/10.18637/jss.v016.i09}
}
\seealso{
\code{\link{alogLik}}: loglikelihood adjustment for model fits.
}
