% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/api.R, R/kibior.R
\docType{class}
\name{kibior}
\alias{kibior}
\alias{Kibior}
\title{KibioR, an Kibio and Elasticsearch data manipulation package.}
\format{
\code{\link{R6Class}} object.
}
\description{
KibioR is a lightweight package for data manipulation 
 with Elasticsearch. Its main features allow easy data import, export,
 download, upload, searching and sharing to any Elasticsearch-based open 
 architecture, scaling to billions of data and TB capability.

Kibior is a Kibio/Elasticsearch client written with R6 class. 
Instances of Kibior are object that allow to use Elasticsearch power and execute lots of predefined 
requests such as searching in massive amounts of data, joins between in-memory data and Elasticsearch 
indices, push and pull data to and from multiples Elasticsearch servers, and more. This little 
utilitary was built in the context of massive data invading biology and bioinformatics, but is completely
versatile and can be applied to other fields. By adding it to R-scripts, it can perform several 
useful tasks such as: saving intermediary results, sharing them with a collaborator, automating 
import and upload of lots of files directly, and much more.
}
\details{
A client to send, retrieve, search, join data in Elasticsearch.
}
\section{Constructor Arguments}{

 \tabular{llll}{
 \strong{Argument} \tab \strong{Type} \tab \strong{Details} \tab \strong{Default} \cr
 \code{host} \tab character \tab address or name of Elasticsearch server \tab "localhost" \cr
 \code{port} \tab numeric \tab port of Elasticsearch server \tab 9200 \cr
 \code{user} \tab character \tab if required by the server, the username for authentication \tab NULL \cr
 \code{pwd} \tab character \tab if required by the server, the password for authentication \tab NULL \cr
 \code{verbose} \tab logical \tab verbose mode \tab FALSE \cr
 }

 created
}

\examples{

## ------------------------------------------------
## Method `Kibior$new`
## ------------------------------------------------

\dontrun{
# default initiatlization, connect to "localhost:9200"
kc <- Kibior$new()
# connect to "192.168.2.145:9200"
kc <- Kibior$new("192.168.2.145")
# connect to "es:15005", verbose mode activated
kc <- Kibior$new(host = "elasticsearch", port = 15005, verbose = TRUE)
# connect to "192.168.2.145:9450" with credentials "foo:bar"
kc <- Kibior$new(host = "192.168.2.145", port = 9450, user = "foo", pwd = "bar")
# connect to "elasticsearch:9200"
kc <- Kibior$new("elasticsearch")

# get kibior var from env (".Renviron" file or local env) 
dd <- system.file("doc_env", "kibior_build.R", package = "kibior")
source(dd, local = TRUE)
kc <- .kibior_get_instance_from_env()
kc$quiet_progress <- TRUE

# preparing all examples (do not mind this for this method)
delete_if_exists <- function(index_names){
    tryCatch(
        expr = { kc$delete(index_names) },
        error = function(e){  }
    )
}
delete_if_exists(c(
    "aaa", 
    "bbb", 
    "ccc", 
    "ddd", 
    "sw", 
    "sw_naboo", 
    "sw_tatooine", 
    "sw_alderaan", 
    "sw_from_file", 
    "storms",
    "starwars"
))
}


## ------------------------------------------------
## Method `Kibior$print`
## ------------------------------------------------

\dontrun{
print(kc)
}



## ------------------------------------------------
## Method `Kibior$eq`
## ------------------------------------------------

\dontrun{
kc$eq(kc)
}



## ------------------------------------------------
## Method `Kibior$ne`
## ------------------------------------------------

\dontrun{
kc$ne(kc)
}


## ------------------------------------------------
## Method `Kibior$create`
## ------------------------------------------------

\dontrun{
kc$create("aaa")
kc$create(c("bbb", "ccc"))
}


## ------------------------------------------------
## Method `Kibior$list`
## ------------------------------------------------

\dontrun{
kc$list()
kc$list(get_specials = TRUE)
}


## ------------------------------------------------
## Method `Kibior$has`
## ------------------------------------------------

\dontrun{
kc$has("aaa")
kc$has(c("bbb", "ccc"))
}


## ------------------------------------------------
## Method `Kibior$delete`
## ------------------------------------------------

\dontrun{
kc$delete("aaa")
kc$delete(c("bbb", "ccc"))
}


## ------------------------------------------------
## Method `Kibior$add_description`
## ------------------------------------------------

\dontrun{
kc$add_description(
    index_name = "sw", 
    dataset_name = "starwars", 
    source_name = "Package dplyr", 
    index_description = "Description of starwars characters, the data comes from the Star 
     Wars API.", 
    version = "dplyr (1.0.0)", 
    link = "http://swapi.dev/", 
    direct_download_link = "http://swapi.dev/", 
    version_date = "2020-05-28", 
    license_link = "MIT", 
    columns = list(
        "name" = "Name of the character",
        "height" = "Height (cm)",
        "mass" = "Weight (kg)",
        "hair_color" = "Hair colors",
        "skin_color" = "Skin colors",
        "eye_color" = "Eye colors",
        "birth_year" = "Year born (BBY = Before Battle of Yavin)",
        "sex" = "The biological sex of the character, namely male, female, 
             hermaphroditic, or none (as in the case for Droids).",
        "gender" = "The gender role or gender identity of the character as determined by 
             their personality or the way they were progammed (as in the case for Droids
             ).",
        "homeworld" = "Name of homeworld",
        "species" = "Name of species",
        "films" = "List of films the character appeared in",
        "vehicles" = "List of vehicles the character has piloted",
        "starships" = "List of starships the character has piloted"
    )
)
}


## ------------------------------------------------
## Method `Kibior$has_description`
## ------------------------------------------------

\dontrun{
kc$has_description("s*")
kc$has_description(c("sw", "asdf"))
}


## ------------------------------------------------
## Method `Kibior$missing_descriptions`
## ------------------------------------------------

\dontrun{
kc$missing_descriptions()
}


## ------------------------------------------------
## Method `Kibior$remove_description`
## ------------------------------------------------

\dontrun{
# remove the description of 'test' index
kc$remove_description("test")
}


## ------------------------------------------------
## Method `Kibior$clean_descriptions`
## ------------------------------------------------

\dontrun{
# remove the description of 'test' index
kc$clean_descriptions()
}


## ------------------------------------------------
## Method `Kibior$describe`
## ------------------------------------------------

\dontrun{
kc$describe("s*")
kc$describe("sw", columns = c("name", "height"))
}


## ------------------------------------------------
## Method `Kibior$describe_index`
## ------------------------------------------------

\dontrun{
kc$describe_index("s*")
}


## ------------------------------------------------
## Method `Kibior$describe_columns`
## ------------------------------------------------

\dontrun{
kc$describe_columns("s*", c("name", "height"))
}


## ------------------------------------------------
## Method `Kibior$infos`
## ------------------------------------------------

\dontrun{
kc$infos()
}


## ------------------------------------------------
## Method `Kibior$ping`
## ------------------------------------------------

\dontrun{
kc$ping()
}


## ------------------------------------------------
## Method `Kibior$mappings`
## ------------------------------------------------

\dontrun{
kc$mappings()
kc$mappings("sw")
kc$mappings(c("sw", "sw_naboo"))
}


## ------------------------------------------------
## Method `Kibior$settings`
## ------------------------------------------------

\dontrun{
kc$settings()
kc$settings("sw")
kc$settings(c("sw", "sw_tatooine"))
}


## ------------------------------------------------
## Method `Kibior$aliases`
## ------------------------------------------------

\dontrun{
kc$aliases()
kc$aliases("sw")
kc$aliases(c("sw", "sw_alderaan"))
}


## ------------------------------------------------
## Method `Kibior$dim`
## ------------------------------------------------

\dontrun{
# Couple [<nb obs> <nb var>] in "sw"
kc$dim("sw")
# Couple [<nb obs> <nb var>] in indices "sw_naboo" and "sw_alderaan"
kc$dim(c("sw_naboo", "sw_alderaan"))
}


## ------------------------------------------------
## Method `Kibior$columns`
## ------------------------------------------------

\dontrun{
kc$columns("sw")          # direct search
kc$columns("sw_*")        # pattern search
}


## ------------------------------------------------
## Method `Kibior$count`
## ------------------------------------------------

\dontrun{
# Number of observations (nb of records) in "sw"
kc$count("sw")
# Number of observations in indices "sw_naboo" and "sw_tatooine"
kc$count(c("sw_naboo", "sw_tatooine"))
# Number of variables (nb of columns) in index "sw_naboo"
kc$count("sw_naboo", type = "variables")
}


## ------------------------------------------------
## Method `Kibior$avg`
## ------------------------------------------------

\dontrun{
# Avg of "sw" column "height"
kc$avg("sw", "height")
# if pattern
kc$avg("s*", "height")
# multiple indices, multiple columns
kc$avg(c("sw", "sw2"), c("height", "mass"), query = "homeworld:naboo")
}


## ------------------------------------------------
## Method `Kibior$mean`
## ------------------------------------------------

\dontrun{
# mean of "sw" column "height"
kc$mean("sw", "height")
# if pattern
kc$mean("s*", "height")
# multiple indices, multiple columns
kc$mean(c("sw", "sw2"), c("height", "mass"), query = "homeworld:naboo")
}


## ------------------------------------------------
## Method `Kibior$min`
## ------------------------------------------------

\dontrun{
# min of "sw" column "height"
kc$min("sw", "height")
# if pattern
kc$min("s*", "height")
# multiple indices, multiple columns
kc$min(c("sw", "sw2"), c("height", "mass"), query = "homeworld:naboo")
}


## ------------------------------------------------
## Method `Kibior$max`
## ------------------------------------------------

\dontrun{
# max of "sw" column "height"
kc$max("sw", "height")
# if pattern
kc$max("s*", "height")
# multiple indices, multiple columns
kc$max(c("sw", "sw2"), c("height", "mass"), query = "homeworld:naboo")
}


## ------------------------------------------------
## Method `Kibior$sum`
## ------------------------------------------------

\dontrun{
# sum of "sw" column "height"
kc$sum("sw", "height")
# if pattern
kc$sum("s*", "height")
# multiple indices, multiple columns
kc$sum(c("sw", "sw2"), c("height", "mass"), query = "homeworld:naboo")
}


## ------------------------------------------------
## Method `Kibior$stats`
## ------------------------------------------------

\dontrun{
# Stats of "sw" column "height"
kc$stats("sw", "height")
# if pattern
kc$stats("s*", "height")
# multiple indices and sigma definition
kc$stats(c("sw", "sw2"), "height", sigma = 2.5)
# multiple indices, multiple columns
kc$stats(c("sw", "sw2"), c("height", "mass"), query = "homeworld:naboo")
}


## ------------------------------------------------
## Method `Kibior$percentiles`
## ------------------------------------------------

\dontrun{
# percentiles of "sw" column "height", default is with q1, q2 and q3
kc$percentiles("sw", "height")
# if pattern
kc$percentiles("s*", "height")
# defining percents to get
kc$percentiles("s*", "height", percents = c(20, 25))
# multiple indices, multiple columns
kc$percentiles(c("sw", "sw2"), c("height", "mass"), query = "homeworld:naboo")
}


## ------------------------------------------------
## Method `Kibior$q1`
## ------------------------------------------------

\dontrun{
# Q1 of "sw" column "height"
kc$q1("sw", "height")
# if pattern
kc$q1("s*", "height")
# multiple indices, multiple columns
kc$q1(c("sw", "sw2"), c("height", "mass"), query = "homeworld:naboo")
}


## ------------------------------------------------
## Method `Kibior$q2`
## ------------------------------------------------

\dontrun{
# Q2 of "sw" column "height"
kc$q2("sw", "height")
# if pattern
kc$q2("s*", "height")
# multiple indices, multiple columns
kc$q2(c("sw", "sw2"), c("height", "mass"), query = "homeworld:naboo")
}


## ------------------------------------------------
## Method `Kibior$median`
## ------------------------------------------------

\dontrun{
# median of "sw" column "height"
kc$median("sw", "height")
# if pattern
kc$median("s*", "height")
# multiple indices, multiple columns
kc$median(c("sw", "sw2"), c("height", "mass"), query = "homeworld:naboo")
}


## ------------------------------------------------
## Method `Kibior$q3`
## ------------------------------------------------

\dontrun{
# Q3 of "sw" column "height"
kc$q3("sw", "height")
# if pattern
kc$q3("s*", "height")
# multiple indices, multiple columns
kc$q3(c("sw", "sw2"), c("height", "mass"), query = "homeworld:naboo")
}


## ------------------------------------------------
## Method `Kibior$summary`
## ------------------------------------------------

\dontrun{
# summary of "sw" column "height"
kc$summary("sw", "height")
# if pattern
kc$summary("s*", "height")
# multiple indices, multiple columns
kc$summary(c("sw", "sw2"), c("height", "mass"), query = "homeworld:naboo")
}


## ------------------------------------------------
## Method `Kibior$keys`
## ------------------------------------------------

\dontrun{
kc$keys("sw", "name")
kc$keys("sw", "eye_color")
}


## ------------------------------------------------
## Method `Kibior$bam_to_tibble`
## ------------------------------------------------

\dontrun{
dd_bai <- system.file("extdata", "test.bam.bai", package = "kibior")
bam_param <- Rsamtools::ScanBamParam(what = c("pos", "qwidth"))
bam_data <- Rsamtools::scanBam(dd_bai, param = bam_param)
kc$bam_to_tibble(bam_data)
}


## ------------------------------------------------
## Method `Kibior$soft_cast`
## ------------------------------------------------

\dontrun{
kc$soft_cast(datasets::iris)
}


## ------------------------------------------------
## Method `Kibior$get_resource`
## ------------------------------------------------

\dontrun{
kc$get_resource(system.file("R", "kibior.R", package = "kibior"))
kc$get_resource("https://ftp.ncbi.nlm.nih.gov/entrez/README")
}


## ------------------------------------------------
## Method `Kibior$export`
## ------------------------------------------------

\dontrun{
f <- tempfile(fileext=".csv")
# export and overwrite last file with the same data from Elasticsearch
kc$export(data = "sw", filepath = f)
# export from in-memory data to a file
kc$export(data = dplyr::starwars, filepath = f, force = TRUE)
}


## ------------------------------------------------
## Method `Kibior$import_tabular`
## ------------------------------------------------

\dontrun{
f <- tempfile(fileext = ".csv")
rio::export(ggplot2::diamonds, f)
# import to in-memory variable
kc$import_tabular(filepath = f)
# import raw data
kc$import_tabular(filepath = f, to_tibble = FALSE)
}


## ------------------------------------------------
## Method `Kibior$import_features`
## ------------------------------------------------

\dontrun{
# get sample files
f_gff <- system.file("extdata", "chr_y.gff3.gz", package = "kibior")
f_bed <- system.file("extdata", "cpg.bed", package = "kibior")
# import to in-memory variable
kc$import_features(filepath = f_bed)
kc$import_features(filepath = f_gff)
# import raw data
kc$import_features(filepath = f_bed, to_tibble = FALSE)
kc$import_features(filepath = f_gff, to_tibble = FALSE)
}


## ------------------------------------------------
## Method `Kibior$import_alignments`
## ------------------------------------------------

\dontrun{
# get sample file
f_bai <- system.file("extdata", "test.bam.bai", package = "kibior")
# import to in-memory variable
kc$import_alignments(filepath = f_bai)
# import raw data
kc$import_alignments(filepath = f_bai, to_tibble = FALSE)
}


## ------------------------------------------------
## Method `Kibior$import_json`
## ------------------------------------------------

\dontrun{
# get sample file
f_json <- system.file("extdata", "storms100.json", package = "kibior")
# import to in-memory variable
kc$import_json(f_json)
# import raw data
kc$import_json(f_json, to_tibble = FALSE)
}


## ------------------------------------------------
## Method `Kibior$import_sequences`
## ------------------------------------------------

\dontrun{
# get sample file
f_dna <- system.file("extdata", "dna_human_y.fa.gz", package = "kibior")
f_rna <- system.file("extdata", "ncrna_mus_musculus.fa.gz", package = "kibior")
f_aa <- system.file("extdata", "pep_mus_spretus.fa.gz", package = "kibior")
# import to in-memory variable
kc$import_sequences(filepath = f_dna, fasta_type = "dna")
# import raw data
kc$import_sequences(filepath = f_rna, to_tibble = FALSE, fasta_type = "rna")
# import auto
kc$import_sequences(filepath = f_aa)
}


## ------------------------------------------------
## Method `Kibior$guess_import`
## ------------------------------------------------

\dontrun{
# get sample file
f_dna <- system.file("extdata", "dna_human_y.fa.gz", package = "kibior")
f_rna <- system.file("extdata", "ncrna_mus_musculus.fa.gz", package = "kibior")
f_aa <- system.file("extdata", "pep_mus_spretus.fa.gz", package = "kibior")
f_bai <- system.file("extdata", "test.bam.bai", package = "kibior")
f_gff <- system.file("extdata", "chr_y.gff3.gz", package = "kibior")
f_bed <- system.file("extdata", "cpg.bed", package = "kibior")
# import 
kc$guess_import(f_dna)
kc$guess_import(f_rna)
kc$guess_import(f_aa)
kc$guess_import(f_bai)
kc$guess_import(f_gff)
kc$guess_import(f_bed)
}


## ------------------------------------------------
## Method `Kibior$import`
## ------------------------------------------------

\dontrun{
# get sample file
f_aa <- system.file("extdata", "pep_mus_spretus.fa.gz", package = "kibior")
f_gff <- system.file("extdata", "chr_y.gff3.gz", package = "kibior")
f_bai <- system.file("extdata", "test.bam.bai", package = "kibior")
# import 
kc$import(filepath = f_aa)
# import to Elasticsearch index ("sw_from_file") if not exists
kc$import(filepath = f_bai, push_index = "sw_from_file")
# import to index by recreating it, then pull indexed data
kc$import(filepath = f_gff, push_index = "sw_from_file",
 push_mode = "recreate")
}


## ------------------------------------------------
## Method `Kibior$push`
## ------------------------------------------------

\dontrun{
# erase the last push data by recreating the index and re-pushing data
kc$push(dplyr::starwars, index_name = "sw", mode = "recreate")
# characters names are unique, can be used as ID
kc$push(dplyr::starwars, index_name = "sw", mode = "recreate", id_col = "name")
# a bit more complicated: update some data of the dataset "starwars"
# 38 records on 87 filtered
some_new_data <- dplyr::filter(dplyr::starwars, height > 180)
# make them all "gender <- female"
some_new_data["gender"] <- "female"
# update that apply, based on cahracter names to match the right record
kc$push(some_new_data, "sw", mode = "update", id_col = "name")
# view result by querying
kc$pull("sw", query = "height:>180", columns = c("name", "gender"))
}


## ------------------------------------------------
## Method `Kibior$pull`
## ------------------------------------------------

\dontrun{
# push some data sample
kc$push(dplyr::storms, "storms")
# get the whole "sw" index
kc$pull("sw")
# get the whole "sw" index with all metadata
kc$pull("sw", keep_metadata = TRUE)
# get only "name" and "status" columns of indices starting with "s"
# columns not found will be ignored
kc$pull("s*", columns = c("name", "status"))
# limit the size of the result to 10
kc$pull("storms", max_size = 10, bulk_size = 10)
# use Elasticsearch query syntax to select and filter on all indices, for all data
# Here, we want to search for all records taht match the conditions:
# field "height" is strictly more than 180 AND field homeworld is "Tatooine" OR "Naboo"
r <- kc$pull("sw", query = "height:>180 && homeworld:(Tatooine || Naboo)")
# it can be used in conjunction with `columns` to select only columns that matter
r <- kc$pull("sw", query = "height:>180 && homeworld:(Tatooine || Naboo)", columns = 
 c("name", "hair_color", "homeworld"))
}


## ------------------------------------------------
## Method `Kibior$move`
## ------------------------------------------------

\dontrun{
kc$push(dplyr::starwars, "sw", mode = "recreate")
# move data from an index to another (change name, same instance)
r <- kc$move(from_index = "sw", to_index = "sw_new")
kc$pull("sw_new")
kc$list()
} 


## ------------------------------------------------
## Method `Kibior$copy`
## ------------------------------------------------

\dontrun{
# copy data from one index to another (same instance)
r <- kc$copy(from_index = "sw_new", to_index = "sw")
kc$pull(c("sw", "sw_new"))
kc$list()
} 


## ------------------------------------------------
## Method `Kibior$match`
## ------------------------------------------------

\dontrun{
# search "sw" index name
kc$match("sw")
# search all starting with an "s"
kc$match("s*")
# get all index name, identical to `$list()`
kc$match("*")
# search multiple names 
kc$match(c("sw", "sw_new", "nope"))
# search multiple names with pattern
kc$match(c("s*", "nope"))
}


## ------------------------------------------------
## Method `Kibior$search`
## ------------------------------------------------

\dontrun{
# search "sw" index, head mode on
kc$search("sw")
# search "sw" index with all metadata, head mode on
kc$search("sw", keep_metadata = TRUE)
# get only "name" field of the head of indices starting with "s"
# if an index does not have the "name" field, it will be empty
kc$search("s*", columns = "name")
# limit the size of the result to 50 to the whole index
kc$search("storms", max_size = 50, bulk_size = 50, head = FALSE)
# use Elasticsearch query syntax to select and filter on all indices, for all data
# Here, we want to search for all records taht match the conditions:
# field "height" is strictly more than 180 AND field homeworld is "Tatooine" OR "Naboo"
kc$search("*", query = "height:>180 && homeworld:(Tatooine || Naboo)")
# it can be used in conjunction with `columns` to select only columns that matter
kc$search("*", query = "height:>180 && homeworld:(Tatooine || Naboo)", columns = 
 c("name", "hair_color", "homeworld"))
}


## ------------------------------------------------
## Method `Kibior$inner_join`
## ------------------------------------------------

\dontrun{
# some data for joins examples
kc$push(ggplot2::diamonds, "diamonds")
# prepare join datasets, only big the biggest diamonds are selected (9)
sup_carat <- dplyr::filter(ggplot2::diamonds, carat > 3.5)
r <- kc$push(sup_carat, "diamonds_superior")
# execute a inner_join with one index and one in-memory dataset
kc$inner_join(ggplot2::diamonds, "diamonds_superior")
# execute a inner_join with one index queried, and one in-memory dataset
kc$inner_join(ggplot2::diamonds, "diamonds", right_query 
 = "carat:>3.5")
}


## ------------------------------------------------
## Method `Kibior$full_join`
## ------------------------------------------------

\dontrun{
# prepare join datasets, fair cuts 
fair_cut <- dplyr::filter(ggplot2::diamonds, cut == "Fair")  # 1605 lines
sup_carat <- kc$pull("diamonds_superior")$diamonds_superior
# execute a full_join with one index and one in-memory dataset
kc$full_join(fair_cut, "diamonds_superior")
# execute a full_join with one index queried, and one in-memory dataset
kc$full_join(sup_carat, "diamonds", right_query = "cut:fair")
}


## ------------------------------------------------
## Method `Kibior$left_join`
## ------------------------------------------------

\dontrun{
# prepare join datasets, fair cuts 
fair_cut <- dplyr::filter(ggplot2::diamonds, cut == "Fair")  # 1605 lines
sup_carat <- kc$pull("diamonds_superior")$diamonds_superior
# execute a left_join with one index and one in-memory dataset
kc$left_join(fair_cut, "diamonds_superior")
# execute a left_join with one index queried, and one in-memory dataset
kc$left_join(sup_carat, "diamonds", right_query 
 = "cut:fair")
}


## ------------------------------------------------
## Method `Kibior$right_join`
## ------------------------------------------------

\dontrun{
# prepare join datasets, fair cuts 
fair_cut <- dplyr::filter(ggplot2::diamonds, cut == "Fair")  # 1605 lines
sup_carat <- kc$pull("diamonds_superior")$diamonds_superior
# execute a right_join with one index and one in-memory dataset
kc$right_join(fair_cut, "diamonds_superior")
# execute a right_join with one index queried, and one in-memory dataset
kc$right_join(sup_carat, "diamonds", right_query 
 = "cut:fair")
}


## ------------------------------------------------
## Method `Kibior$semi_join`
## ------------------------------------------------

\dontrun{
# prepare join datasets, fair cuts 
fair_cut <- dplyr::filter(ggplot2::diamonds, cut == "Fair")  # 1605 lines
sup_carat <- kc$pull("diamonds_superior")$diamonds_superior
# execute a semi_join with one index and one in-memory dataset
kc$semi_join(fair_cut, "diamonds_superior")
# execute a semi_join with one index queried, and one in-memory dataset
kc$semi_join(sup_carat, "diamonds", right_query 
 = "cut:fair")
}


## ------------------------------------------------
## Method `Kibior$anti_join`
## ------------------------------------------------

\dontrun{
# prepare join datasets, fair cuts 
fair_cut <- dplyr::filter(ggplot2::diamonds, cut == "Fair")  # 1605 lines
sup_carat <- kc$pull("diamonds_superior")$diamonds_superior
# execute a anti_join with one index and one in-memory dataset
kc$anti_join(fair_cut, "diamonds_superior")
# execute a anti_join with one index queried, and one in-memory dataset
kc$anti_join(sup_carat, "diamonds", right_query 
 = "cut:fair")
# 
# Do not mind this, removing example indices
elastic::index_delete(kc$connection, "*")
kc <- NULL
}

}
\references{
Kibio.science: \url{http://kibio.science}, \cr
 Elasticsearch documentation: 
 \url{https://www.elastic.co/guide/en/elasticsearch/reference/current/index.html}
}
\seealso{
\code{\link{kibior}}

you should use \code{\link{count}} for more accurate count.

\url{https://www.elastic.co/guide/en/elasticsearch/reference/current/common-options.html#time-units} 
 for time-units and \url{https://www.elastic.co/guide/en/elasticsearch/reference/current/query-dsl-query-string-query.html#query-string-syntax}
 for the Elasticsearch query string syntax.

: \url{https://www.elastic.co/guide/en/elasticsearch/reference/current/docs-reindex.html}
 Elasticsearch reindex feature for more information.

: \url{https://www.elastic.co/guide/en/elasticsearch/reference/current/docs-reindex.html}
 Elasticsearch reindex feature for more information.

\url{https://www.elastic.co/guide/en/elasticsearch/reference/current/common-options.html#time-units} 
 for time-units and \url{https://www.elastic.co/guide/en/elasticsearch/reference/current/query-dsl-query-string-query.html#query-string-syntax}
 for the Elasticsearch query string syntax.
}
\author{
Régis Ongaro-Carcy,
 \email{regis.ongaro-carcy2@crchudequebec.ulaval.ca}
}
\concept{cluster-wealth}
\concept{crud-index}
\concept{crud-metadata}
\concept{data-manipulation}
\concept{joins}
\concept{kibior-metadata}
\concept{move-data}
\concept{search-data}
\concept{stats}
\keyword{data}
\keyword{dataset}
\keyword{integration}
\section{Public fields}{
\if{html}{\out{<div class="r6-fields">}}
\describe{
\item{\code{verbose}}{verbose mode, prints out more informations during execution}

\item{\code{quiet_progress}}{progressbar quiet mode, toggles progress bar}

\item{\code{quiet_results}}{results quiet mode, toggles results printing}
}
\if{html}{\out{</div>}}
}
\section{Active bindings}{
\if{html}{\out{<div class="r6-active-bindings">}}
\describe{
\item{\code{host}}{Access and change the Elasticsearch host}

\item{\code{port}}{Access and change the Elasticsearch port}

\item{\code{endpoint}}{Access the Elasticsearch main endpoint}

\item{\code{user}}{Access the Elasticsearch user.}

\item{\code{pwd}}{Access the Elasticsearch password.}

\item{\code{connection}}{Access the Elasticsearch connection object.}

\item{\code{head_search_size}}{Access and change the head size default value.}

\item{\code{cluster_name}}{Access the cluster name if and only if already connected.}

\item{\code{cluster_status}}{Access the cluster status if and only if already connected.}

\item{\code{nb_documents}}{Access the current cluster total number of documents if and only if 
already connected.}

\item{\code{version}}{Access the Elasticsearch version if and only if already connected.}

\item{\code{elastic_wait}}{Access and change the Elasticsearch wait time for update commands if 
and only if already connected.}

\item{\code{valid_joins}}{Access the valid joins available in Kibior.}

\item{\code{valid_count_types}}{Access the valid count types available (mainly observations = 
rows, variables = columns)}

\item{\code{valid_elastic_metadata_types}}{Access the valid Elasticsearch metadata types 
available.}

\item{\code{valid_push_modes}}{Access the valid push modes available.}

\item{\code{shard_number}}{Access and modify the number of allocated primary shards when 
creating an Elasticsearch index.}

\item{\code{shard_replicas_number}}{Access and modify the number of allocated replicas in an 
Elasticsearch index.}

\item{\code{default_id_col}}{Access and modify the default ID column/field created when pushing 
data to Elasticsearch.}
}
\if{html}{\out{</div>}}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-new}{\code{Kibior$new()}}
\item \href{#method-print}{\code{Kibior$print()}}
\item \href{#method-eq}{\code{Kibior$eq()}}
\item \href{#method-ne}{\code{Kibior$ne()}}
\item \href{#method-create}{\code{Kibior$create()}}
\item \href{#method-list}{\code{Kibior$list()}}
\item \href{#method-has}{\code{Kibior$has()}}
\item \href{#method-delete}{\code{Kibior$delete()}}
\item \href{#method-add_description}{\code{Kibior$add_description()}}
\item \href{#method-has_description}{\code{Kibior$has_description()}}
\item \href{#method-missing_descriptions}{\code{Kibior$missing_descriptions()}}
\item \href{#method-remove_description}{\code{Kibior$remove_description()}}
\item \href{#method-clean_descriptions}{\code{Kibior$clean_descriptions()}}
\item \href{#method-describe}{\code{Kibior$describe()}}
\item \href{#method-describe_index}{\code{Kibior$describe_index()}}
\item \href{#method-describe_columns}{\code{Kibior$describe_columns()}}
\item \href{#method-infos}{\code{Kibior$infos()}}
\item \href{#method-ping}{\code{Kibior$ping()}}
\item \href{#method-mappings}{\code{Kibior$mappings()}}
\item \href{#method-settings}{\code{Kibior$settings()}}
\item \href{#method-aliases}{\code{Kibior$aliases()}}
\item \href{#method-dim}{\code{Kibior$dim()}}
\item \href{#method-columns}{\code{Kibior$columns()}}
\item \href{#method-count}{\code{Kibior$count()}}
\item \href{#method-avg}{\code{Kibior$avg()}}
\item \href{#method-mean}{\code{Kibior$mean()}}
\item \href{#method-min}{\code{Kibior$min()}}
\item \href{#method-max}{\code{Kibior$max()}}
\item \href{#method-sum}{\code{Kibior$sum()}}
\item \href{#method-stats}{\code{Kibior$stats()}}
\item \href{#method-percentiles}{\code{Kibior$percentiles()}}
\item \href{#method-q1}{\code{Kibior$q1()}}
\item \href{#method-q2}{\code{Kibior$q2()}}
\item \href{#method-median}{\code{Kibior$median()}}
\item \href{#method-q3}{\code{Kibior$q3()}}
\item \href{#method-summary}{\code{Kibior$summary()}}
\item \href{#method-keys}{\code{Kibior$keys()}}
\item \href{#method-bam_to_tibble}{\code{Kibior$bam_to_tibble()}}
\item \href{#method-soft_cast}{\code{Kibior$soft_cast()}}
\item \href{#method-get_resource}{\code{Kibior$get_resource()}}
\item \href{#method-export}{\code{Kibior$export()}}
\item \href{#method-import_tabular}{\code{Kibior$import_tabular()}}
\item \href{#method-import_features}{\code{Kibior$import_features()}}
\item \href{#method-import_alignments}{\code{Kibior$import_alignments()}}
\item \href{#method-import_json}{\code{Kibior$import_json()}}
\item \href{#method-import_sequences}{\code{Kibior$import_sequences()}}
\item \href{#method-guess_import}{\code{Kibior$guess_import()}}
\item \href{#method-import}{\code{Kibior$import()}}
\item \href{#method-push}{\code{Kibior$push()}}
\item \href{#method-pull}{\code{Kibior$pull()}}
\item \href{#method-move}{\code{Kibior$move()}}
\item \href{#method-copy}{\code{Kibior$copy()}}
\item \href{#method-match}{\code{Kibior$match()}}
\item \href{#method-search}{\code{Kibior$search()}}
\item \href{#method-inner_join}{\code{Kibior$inner_join()}}
\item \href{#method-full_join}{\code{Kibior$full_join()}}
\item \href{#method-left_join}{\code{Kibior$left_join()}}
\item \href{#method-right_join}{\code{Kibior$right_join()}}
\item \href{#method-semi_join}{\code{Kibior$semi_join()}}
\item \href{#method-anti_join}{\code{Kibior$anti_join()}}
\item \href{#method-clone}{\code{Kibior$clone()}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-new"></a>}}
\if{latex}{\out{\hypertarget{method-new}{}}}
\subsection{Method \code{new()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$new(
  host = "localhost",
  port = 9200,
  user = NULL,
  pwd = NULL,
  verbose = getOption("verbose")
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{host}}{The target host to connect to Elasticsearch REST API (default: "localhost").}

\item{\code{port}}{The target port (default: 9200).}

\item{\code{user}}{If the server needs authentication, your username (default: NULL).}

\item{\code{pwd}}{If the server needs authentication, your password (default: NULL).}

\item{\code{verbose}}{The verbose mode (default: FALSE).}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Initialize a new object, automatically called when calling `Kibior$new()`
}

\subsection{Returns}{
a new instance/object of Kibior
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# default initiatlization, connect to "localhost:9200"
kc <- Kibior$new()
# connect to "192.168.2.145:9200"
kc <- Kibior$new("192.168.2.145")
# connect to "es:15005", verbose mode activated
kc <- Kibior$new(host = "elasticsearch", port = 15005, verbose = TRUE)
# connect to "192.168.2.145:9450" with credentials "foo:bar"
kc <- Kibior$new(host = "192.168.2.145", port = 9450, user = "foo", pwd = "bar")
# connect to "elasticsearch:9200"
kc <- Kibior$new("elasticsearch")

# get kibior var from env (".Renviron" file or local env) 
dd <- system.file("doc_env", "kibior_build.R", package = "kibior")
source(dd, local = TRUE)
kc <- .kibior_get_instance_from_env()
kc$quiet_progress <- TRUE

# preparing all examples (do not mind this for this method)
delete_if_exists <- function(index_names){
    tryCatch(
        expr = { kc$delete(index_names) },
        error = function(e){  }
    )
}
delete_if_exists(c(
    "aaa", 
    "bbb", 
    "ccc", 
    "ddd", 
    "sw", 
    "sw_naboo", 
    "sw_tatooine", 
    "sw_alderaan", 
    "sw_from_file", 
    "storms",
    "starwars"
))
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-print"></a>}}
\if{latex}{\out{\hypertarget{method-print}{}}}
\subsection{Method \code{print()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$print()}\if{html}{\out{</div>}}
}

\subsection{Details}{
Print simple informations of the current object.
}

\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
print(kc)
}


}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-eq"></a>}}
\if{latex}{\out{\hypertarget{method-eq}{}}}
\subsection{Method \code{eq()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$eq(other = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{other}}{Another instance/object of Kibior (default: NULL).}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Tells if another instance of Kibior has the same `host:port` couple.
}

\subsection{Returns}{
TRUE if hosts and ports are identical, else FALSE
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
kc$eq(kc)
}


}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ne"></a>}}
\if{latex}{\out{\hypertarget{method-ne}{}}}
\subsection{Method \code{ne()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$ne(other = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{other}}{Another instance/object of Kibior (default: NULL).}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Tells if another instance of Kibior has a different `host:port` couple.
}

\subsection{Returns}{
TRUE if hosts and ports are differents, else FALSE
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
kc$ne(kc)
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-create"></a>}}
\if{latex}{\out{\hypertarget{method-create}{}}}
\subsection{Method \code{create()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$create(index_name, force = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{index_name}}{a vector of index names to create (default: NULL).}

\item{\code{force}}{Erase already existing identical index names? (default: FALSE).}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Create one or several indices in Elasticsearch.
}

\subsection{Returns}{
a list containing results of creation per index
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
kc$create("aaa")
kc$create(c("bbb", "ccc"))
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-list"></a>}}
\if{latex}{\out{\hypertarget{method-list}{}}}
\subsection{Method \code{list()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$list(get_specials = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{get_specials}}{a boolean to get special indices (default: FALSE).}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
List indices in Elasticsearch.
}

\subsection{Returns}{
a list of index names, NULL if no index found
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
kc$list()
kc$list(get_specials = TRUE)
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-has"></a>}}
\if{latex}{\out{\hypertarget{method-has}{}}}
\subsection{Method \code{has()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$has(index_name)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{index_name}}{a vector of index names to check.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Does Elasticsearch has one or several indices?
}

\subsection{Returns}{
a list with TRUE for found index, else FALSE
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
kc$has("aaa")
kc$has(c("bbb", "ccc"))
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-delete"></a>}}
\if{latex}{\out{\hypertarget{method-delete}{}}}
\subsection{Method \code{delete()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$delete(index_name)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{index_name}}{a vector of index names to delete.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Delete one or several indices in Elasticsearch.
}

\subsection{Returns}{
a list containing results of deletion per index, or NULL if no index name match
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
kc$delete("aaa")
kc$delete(c("bbb", "ccc"))
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-add_description"></a>}}
\if{latex}{\out{\hypertarget{method-add_description}{}}}
\subsection{Method \code{add_description()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$add_description(
  index_name,
  dataset_name,
  source_name,
  index_description,
  version,
  change_log,
  website,
  direct_download,
  version_date,
  license,
  contact,
  references,
  columns = list(),
  force = FALSE
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{index_name}}{the index name to describe}

\item{\code{dataset_name}}{the full length dataset name}

\item{\code{source_name}}{the source/website/entity full length name}

\item{\code{index_description}}{the index description, should be explicit}

\item{\code{version}}{the version of the source dataset}

\item{\code{change_log}}{what have been done from the last version}

\item{\code{website}}{the website to the source dataset website}

\item{\code{direct_download}}{the direct download url of the dataset source}

\item{\code{version_date}}{the version or build date}

\item{\code{license}}{the license attached to this dataset, could be a url}

\item{\code{contact}}{a mailto/contact}

\item{\code{references}}{some paper and other references (e.g. doi, url)}

\item{\code{columns}}{a list of (column_name = column_description) to register (default: list())}

\item{\code{force}}{if FALSE, raise an error if the description already exists, else TRUE to 
overwrite (default: FALSE)}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Add a description of a pushed dataset.
}

\subsection{Returns}{
the index name if complete, else an error
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
kc$add_description(
    index_name = "sw", 
    dataset_name = "starwars", 
    source_name = "Package dplyr", 
    index_description = "Description of starwars characters, the data comes from the Star 
     Wars API.", 
    version = "dplyr (1.0.0)", 
    link = "http://swapi.dev/", 
    direct_download_link = "http://swapi.dev/", 
    version_date = "2020-05-28", 
    license_link = "MIT", 
    columns = list(
        "name" = "Name of the character",
        "height" = "Height (cm)",
        "mass" = "Weight (kg)",
        "hair_color" = "Hair colors",
        "skin_color" = "Skin colors",
        "eye_color" = "Eye colors",
        "birth_year" = "Year born (BBY = Before Battle of Yavin)",
        "sex" = "The biological sex of the character, namely male, female, 
             hermaphroditic, or none (as in the case for Droids).",
        "gender" = "The gender role or gender identity of the character as determined by 
             their personality or the way they were progammed (as in the case for Droids
             ).",
        "homeworld" = "Name of homeworld",
        "species" = "Name of species",
        "films" = "List of films the character appeared in",
        "vehicles" = "List of vehicles the character has piloted",
        "starships" = "List of starships the character has piloted"
    )
)
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-has_description"></a>}}
\if{latex}{\out{\hypertarget{method-has_description}{}}}
\subsection{Method \code{has_description()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$has_description(index_name)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{index_name}}{the index name to describe}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Does the description exists?
}

\subsection{Returns}{
a list splitted by index, with TRUE if the description is 
 found, else FALSE. Removes unknown index names.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
kc$has_description("s*")
kc$has_description(c("sw", "asdf"))
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-missing_descriptions"></a>}}
\if{latex}{\out{\hypertarget{method-missing_descriptions}{}}}
\subsection{Method \code{missing_descriptions()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$missing_descriptions()}\if{html}{\out{</div>}}
}

\subsection{Details}{
List indices that do no have descriptions.
}

\subsection{Returns}{
a vector of indices not present in description.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
kc$missing_descriptions()
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-remove_description"></a>}}
\if{latex}{\out{\hypertarget{method-remove_description}{}}}
\subsection{Method \code{remove_description()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$remove_description(index_name)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{index_name}}{the index name to describe}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Remove a description.
}

\subsection{Returns}{
a vector of indices not present in description.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# remove the description of 'test' index
kc$remove_description("test")
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-clean_descriptions"></a>}}
\if{latex}{\out{\hypertarget{method-clean_descriptions}{}}}
\subsection{Method \code{clean_descriptions()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$clean_descriptions()}\if{html}{\out{</div>}}
}

\subsection{Details}{
Remove all descriptions that do not have a index associated.
}

\subsection{Returns}{
a list of index names which have been removed from descriptions.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# remove the description of 'test' index
kc$clean_descriptions()
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-describe"></a>}}
\if{latex}{\out{\hypertarget{method-describe}{}}}
\subsection{Method \code{describe()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$describe(index_name, columns = NULL, pretty = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{index_name}}{the index name to describe}

\item{\code{columns}}{a vector of column names to describe (default: NULL)}

\item{\code{pretty}}{pretty-print the result (default: FALSE)}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Get the description of indices and their columns.
}

\subsection{Returns}{
all description, grouped by indices
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
kc$describe("s*")
kc$describe("sw", columns = c("name", "height"))
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-describe_index"></a>}}
\if{latex}{\out{\hypertarget{method-describe_index}{}}}
\subsection{Method \code{describe_index()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$describe_index(index_name)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{index_name}}{the index name to describe}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Get the description text of indices.
}

\subsection{Returns}{
a list of description text, grouped by indices
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
kc$describe_index("s*")
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-describe_columns"></a>}}
\if{latex}{\out{\hypertarget{method-describe_columns}{}}}
\subsection{Method \code{describe_columns()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$describe_columns(index_name, columns)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{index_name}}{the index name to describe}

\item{\code{columns}}{a vector of column names to describe}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Get the description text of index columns.
}

\subsection{Returns}{
a list of description text, grouped by indices
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
kc$describe_columns("s*", c("name", "height"))
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-infos"></a>}}
\if{latex}{\out{\hypertarget{method-infos}{}}}
\subsection{Method \code{infos()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$infos()}\if{html}{\out{</div>}}
}

\subsection{Details}{
Get informations about Elasticsearch cluster
}

\subsection{Returns}{
a list of statistics about the cluster
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
kc$infos()
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ping"></a>}}
\if{latex}{\out{\hypertarget{method-ping}{}}}
\subsection{Method \code{ping()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$ping()}\if{html}{\out{</div>}}
}

\subsection{Details}{
Ping cluster connection
}

\subsection{Returns}{
the ping result with some basic infos
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
kc$ping()
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-mappings"></a>}}
\if{latex}{\out{\hypertarget{method-mappings}{}}}
\subsection{Method \code{mappings()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$mappings(index_name)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{index_name}}{a vector of index names to get mappings.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Get mappings of indices
}

\subsection{Returns}{
the list of indices, containing their mapping
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
kc$mappings()
kc$mappings("sw")
kc$mappings(c("sw", "sw_naboo"))
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-settings"></a>}}
\if{latex}{\out{\hypertarget{method-settings}{}}}
\subsection{Method \code{settings()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$settings(index_name)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{index_name}}{a vector of index names to get settings.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Get settings of indices
}

\subsection{Returns}{
the list of indices, containing their settings
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
kc$settings()
kc$settings("sw")
kc$settings(c("sw", "sw_tatooine"))
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-aliases"></a>}}
\if{latex}{\out{\hypertarget{method-aliases}{}}}
\subsection{Method \code{aliases()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$aliases(index_name)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{index_name}}{a vector of index names to get aliases.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Get aliases of indices
}

\subsection{Returns}{
the list of indices, containing their aliases
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
kc$aliases()
kc$aliases("sw")
kc$aliases(c("sw", "sw_alderaan"))
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-dim"></a>}}
\if{latex}{\out{\hypertarget{method-dim}{}}}
\subsection{Method \code{dim()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$dim(index_name)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{index_name}}{a vector of index names to get aliases.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Shortcut to `$count()` to match the classical `dim()` function pattern `[line col]`
}

\subsection{Returns}{
the list of indices, containing their number of observations and variables.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# Couple [<nb obs> <nb var>] in "sw"
kc$dim("sw")
# Couple [<nb obs> <nb var>] in indices "sw_naboo" and "sw_alderaan"
kc$dim(c("sw_naboo", "sw_alderaan"))
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-columns"></a>}}
\if{latex}{\out{\hypertarget{method-columns}{}}}
\subsection{Method \code{columns()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$columns(index_name)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{index_name}}{a vector of index names, can be a pattern.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Get fields/columns of indices.
}

\subsection{Returns}{
a list of indices, each containing their fields/columns.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
kc$columns("sw")          # direct search
kc$columns("sw_*")        # pattern search
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-count"></a>}}
\if{latex}{\out{\hypertarget{method-count}{}}}
\subsection{Method \code{count()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$count(index_name, type = "observations", query = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{index_name}}{a vector of index names to get aliases.}

\item{\code{type}}{a string representing the type to count: "observations" (lines) or 
"variables" (columns) (default: "observations").}

\item{\code{query}}{a string as a query string syntax (default: NULL).}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Count observations or variables in Elasticsearch data
}

\subsection{Returns}{
the list of indices, containing their number of observations or variables. 
 Use `$dim()` for both
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# Number of observations (nb of records) in "sw"
kc$count("sw")
# Number of observations in indices "sw_naboo" and "sw_tatooine"
kc$count(c("sw_naboo", "sw_tatooine"))
# Number of variables (nb of columns) in index "sw_naboo"
kc$count("sw_naboo", type = "variables")
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-avg"></a>}}
\if{latex}{\out{\hypertarget{method-avg}{}}}
\subsection{Method \code{avg()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$avg(index_name, columns, query = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{index_name}}{a vector of index names.}

\item{\code{columns}}{a vector of column names.}

\item{\code{query}}{a string as a query string syntax (default: NULL).}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Get the average of numeric columns.
}

\subsection{Returns}{
a tibble with avg, one line by matching index and column.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# Avg of "sw" column "height"
kc$avg("sw", "height")
# if pattern
kc$avg("s*", "height")
# multiple indices, multiple columns
kc$avg(c("sw", "sw2"), c("height", "mass"), query = "homeworld:naboo")
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-mean"></a>}}
\if{latex}{\out{\hypertarget{method-mean}{}}}
\subsection{Method \code{mean()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$mean(index_name, columns, query = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{index_name}}{a vector of index names.}

\item{\code{columns}}{a vector of column names.}

\item{\code{query}}{a string as a query string syntax (default: NULL).}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Get the mean of numeric columns.
}

\subsection{Returns}{
a tibble with mean, one line by matching index and column.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# mean of "sw" column "height"
kc$mean("sw", "height")
# if pattern
kc$mean("s*", "height")
# multiple indices, multiple columns
kc$mean(c("sw", "sw2"), c("height", "mass"), query = "homeworld:naboo")
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-min"></a>}}
\if{latex}{\out{\hypertarget{method-min}{}}}
\subsection{Method \code{min()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$min(index_name, columns, query = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{index_name}}{a vector of index names.}

\item{\code{columns}}{a vector of column names.}

\item{\code{query}}{a string as a query string syntax (default: NULL).}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Get the minimum of numeric columns.
}

\subsection{Returns}{
a tibble with min, one line by matching index and column.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# min of "sw" column "height"
kc$min("sw", "height")
# if pattern
kc$min("s*", "height")
# multiple indices, multiple columns
kc$min(c("sw", "sw2"), c("height", "mass"), query = "homeworld:naboo")
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-max"></a>}}
\if{latex}{\out{\hypertarget{method-max}{}}}
\subsection{Method \code{max()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$max(index_name, columns, query = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{index_name}}{a vector of index names.}

\item{\code{columns}}{a vector of column names.}

\item{\code{query}}{a string as a query string syntax (default: NULL).}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Get the maximum of numeric columns.
}

\subsection{Returns}{
a tibble with max, one line by matching index and column.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# max of "sw" column "height"
kc$max("sw", "height")
# if pattern
kc$max("s*", "height")
# multiple indices, multiple columns
kc$max(c("sw", "sw2"), c("height", "mass"), query = "homeworld:naboo")
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-sum"></a>}}
\if{latex}{\out{\hypertarget{method-sum}{}}}
\subsection{Method \code{sum()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$sum(index_name, columns, query = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{index_name}}{a vector of index names.}

\item{\code{columns}}{a vector of column names.}

\item{\code{query}}{a string as a query string syntax (default: NULL).}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Get the sum of numeric columns.
}

\subsection{Returns}{
a tibble with sum, one line by matching index and column.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# sum of "sw" column "height"
kc$sum("sw", "height")
# if pattern
kc$sum("s*", "height")
# multiple indices, multiple columns
kc$sum(c("sw", "sw2"), c("height", "mass"), query = "homeworld:naboo")
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-stats"></a>}}
\if{latex}{\out{\hypertarget{method-stats}{}}}
\subsection{Method \code{stats()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$stats(index_name, columns, sigma = NULL, query = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{index_name}}{a vector of index names.}

\item{\code{columns}}{a vector of column names.}

\item{\code{sigma}}{(default: NULL).}

\item{\code{query}}{a string as a query string syntax (default: NULL).}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Produces descriptive statistics of a column.
Returns a tibble composed of: count, min, max, avg, sum, 
 sum_of_squares, variance, std_deviation (+ upper and lower bounds).
Multiple warnings here. One for the count and one for the std_dev.
1/ Counts: they are approximate, see vignette.
2/ Std dev: as stated in ES documentation: "The standard deviation 
and its bounds are displayed by default, but they are not always 
applicable to all data-sets. Your data must be normally distributed 
for the metrics to make sense. The statistics behind standard 
deviations assumes normally distributed data, so if your data is 
skewed heavily left or right, the value returned will be misleading."
}

\subsection{Returns}{
a tibble with descriptive stats, one line by matching index.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# Stats of "sw" column "height"
kc$stats("sw", "height")
# if pattern
kc$stats("s*", "height")
# multiple indices and sigma definition
kc$stats(c("sw", "sw2"), "height", sigma = 2.5)
# multiple indices, multiple columns
kc$stats(c("sw", "sw2"), c("height", "mass"), query = "homeworld:naboo")
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-percentiles"></a>}}
\if{latex}{\out{\hypertarget{method-percentiles}{}}}
\subsection{Method \code{percentiles()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$percentiles(index_name, columns, percents = NULL, query = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{index_name}}{a vector of index names.}

\item{\code{columns}}{a vector of column names.}

\item{\code{percents}}{a numeric vector of pecents to use (default: NULL).}

\item{\code{query}}{a string as a query string syntax (default: NULL).}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Get percentiles of numeric columns.
}

\subsection{Returns}{
a list of tibble, splitted by indices with percentiles inside tibble columns.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# percentiles of "sw" column "height", default is with q1, q2 and q3
kc$percentiles("sw", "height")
# if pattern
kc$percentiles("s*", "height")
# defining percents to get
kc$percentiles("s*", "height", percents = c(20, 25))
# multiple indices, multiple columns
kc$percentiles(c("sw", "sw2"), c("height", "mass"), query = "homeworld:naboo")
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-q1"></a>}}
\if{latex}{\out{\hypertarget{method-q1}{}}}
\subsection{Method \code{q1()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$q1(index_name, columns, query = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{index_name}}{a vector of index names.}

\item{\code{columns}}{a vector of column names.}

\item{\code{query}}{a string as a query string syntax (default: NULL).}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Get Q1 percentiles from numeric columns.
}

\subsection{Returns}{
a list of tibble, splitted by indices with Q1 inside tibble columns.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# Q1 of "sw" column "height"
kc$q1("sw", "height")
# if pattern
kc$q1("s*", "height")
# multiple indices, multiple columns
kc$q1(c("sw", "sw2"), c("height", "mass"), query = "homeworld:naboo")
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-q2"></a>}}
\if{latex}{\out{\hypertarget{method-q2}{}}}
\subsection{Method \code{q2()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$q2(index_name, columns, query = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{index_name}}{a vector of index names.}

\item{\code{columns}}{a vector of column names.}

\item{\code{query}}{a string as a query string syntax (default: NULL).}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Get Q2 percentiles from numeric columns.
}

\subsection{Returns}{
a list of tibble, splitted by indices with Q2 inside tibble columns.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# Q2 of "sw" column "height"
kc$q2("sw", "height")
# if pattern
kc$q2("s*", "height")
# multiple indices, multiple columns
kc$q2(c("sw", "sw2"), c("height", "mass"), query = "homeworld:naboo")
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-median"></a>}}
\if{latex}{\out{\hypertarget{method-median}{}}}
\subsection{Method \code{median()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$median(index_name, columns, query = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{index_name}}{a vector of index names.}

\item{\code{columns}}{a vector of column names.}

\item{\code{query}}{a string as a query string syntax (default: NULL).}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Get median from numeric columns.
Basically a wrapper around `$q2()`.
}

\subsection{Returns}{
a list of tibble, splitted by indices with median inside tibble columns.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# median of "sw" column "height"
kc$median("sw", "height")
# if pattern
kc$median("s*", "height")
# multiple indices, multiple columns
kc$median(c("sw", "sw2"), c("height", "mass"), query = "homeworld:naboo")
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-q3"></a>}}
\if{latex}{\out{\hypertarget{method-q3}{}}}
\subsection{Method \code{q3()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$q3(index_name, columns, query = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{index_name}}{a vector of index names.}

\item{\code{columns}}{a vector of column names.}

\item{\code{query}}{a string as a query string syntax (default: NULL).}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Get Q3 percentiles from numeric columns.
}

\subsection{Returns}{
a list of tibble, splitted by indices with Q3 inside tibble columns.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# Q3 of "sw" column "height"
kc$q3("sw", "height")
# if pattern
kc$q3("s*", "height")
# multiple indices, multiple columns
kc$q3(c("sw", "sw2"), c("height", "mass"), query = "homeworld:naboo")
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-summary"></a>}}
\if{latex}{\out{\hypertarget{method-summary}{}}}
\subsection{Method \code{summary()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$summary(index_name, columns, query = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{index_name}}{a vector of index names.}

\item{\code{columns}}{a vector of column names.}

\item{\code{query}}{a string as a query string syntax (default: NULL).}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Summary for numeric columns.
Cumulates `$min()`, `$max()`, `$q1()`, `$q2()`, `$q3()`.
}

\subsection{Returns}{
a list of tibble, splitted by indices.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# summary of "sw" column "height"
kc$summary("sw", "height")
# if pattern
kc$summary("s*", "height")
# multiple indices, multiple columns
kc$summary(c("sw", "sw2"), c("height", "mass"), query = "homeworld:naboo")
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-keys"></a>}}
\if{latex}{\out{\hypertarget{method-keys}{}}}
\subsection{Method \code{keys()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$keys(index_name, column, max_size = 1000)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{index_name}}{an index name.}

\item{\code{column}}{a field name of this index (default: NULL).}

\item{\code{max_size}}{the maximum result to return (default: 1000).}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Get distinct keys elements of a specific column.
}

\subsection{Returns}{
a vector of keys values from this field/column
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
kc$keys("sw", "name")
kc$keys("sw", "eye_color")
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-bam_to_tibble"></a>}}
\if{latex}{\out{\hypertarget{method-bam_to_tibble}{}}}
\subsection{Method \code{bam_to_tibble()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$bam_to_tibble(bam_data = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{bam_data}}{data from a BAM file (default: NULL).}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Transformation function for collapsing the BAM list of lists format 
 into a single list as per the Rsamtools vignette
}

\subsection{Returns}{
a tibble of BAM data
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
dd_bai <- system.file("extdata", "test.bam.bai", package = "kibior")
bam_param <- Rsamtools::ScanBamParam(what = c("pos", "qwidth"))
bam_data <- Rsamtools::scanBam(dd_bai, param = bam_param)
kc$bam_to_tibble(bam_data)
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-soft_cast"></a>}}
\if{latex}{\out{\hypertarget{method-soft_cast}{}}}
\subsection{Method \code{soft_cast()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$soft_cast(
  data,
  caster = getFromNamespace("as_tibble", "tibble"),
  caster_args = list(.name_repair = "unique"),
  warn = TRUE
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{data}}{data to cast.}

\item{\code{caster}}{the caster closure/function (default: tibble::as_tibble)}

\item{\code{caster_args}}{others caster args (default: list(.name_repair = "unique"))}

\item{\code{warn}}{do print warning if error? (default: TRUE)}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Casting function that tries to cast a transformation closure. Uses 
 tibble::as_tibble() by default.
}

\subsection{Returns}{
a cast or the unchanged data.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
kc$soft_cast(datasets::iris)
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-get_resource"></a>}}
\if{latex}{\out{\hypertarget{method-get_resource}{}}}
\subsection{Method \code{get_resource()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$get_resource(url_or_filepath, fileext = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{url_or_filepath}}{a filepath or an URL.}

\item{\code{fileext}}{the file extension (default: NULL).}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Get a local filepath or an URL data through a tempfile. If the file
 exists locally, the filepath will be returned, if not, it will tries
 to download the data and return the temp filepath.
}

\subsection{Returns}{
a filepath.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
kc$get_resource(system.file("R", "kibior.R", package = "kibior"))
kc$get_resource("https://ftp.ncbi.nlm.nih.gov/entrez/README")
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-export"></a>}}
\if{latex}{\out{\hypertarget{method-export}{}}}
\subsection{Method \code{export()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$export(data, filepath, format = "csv", force = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{data}}{an index name or in-memory data to be extracted to a file.}

\item{\code{filepath}}{the filepath to use as export, must contain the file extention.}

\item{\code{format}}{the file format to use (default: "csv").}

\item{\code{force}}{overwrite the file? (default: FALSE).}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Export data to a file.
Some data formats are not installed by default.
Use `rio::install_formats()` to be able to parse them.
}

\subsection{Returns}{
the filepath if correctly exported, else an error
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
f <- tempfile(fileext=".csv")
# export and overwrite last file with the same data from Elasticsearch
kc$export(data = "sw", filepath = f)
# export from in-memory data to a file
kc$export(data = dplyr::starwars, filepath = f, force = TRUE)
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-import_tabular"></a>}}
\if{latex}{\out{\hypertarget{method-import_tabular}{}}}
\subsection{Method \code{import_tabular()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$import_tabular(filepath, to_tibble = TRUE, fileext = ".csv")}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{filepath}}{the filepath to use as import, must contain the file extention.}

\item{\code{to_tibble}}{returns the result as tibble? If FALSE, the raw default rio::import() 
format will be used (default: TRUE).}

\item{\code{fileext}}{the file extension (default: ".csv").}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Import method for tabular data.
Works mainly with CSV, TSV, TAB, TXT and ZIPped formats.
}

\subsection{Returns}{
data contained in the file as a tibble, or NULL.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
f <- tempfile(fileext = ".csv")
rio::export(ggplot2::diamonds, f)
# import to in-memory variable
kc$import_tabular(filepath = f)
# import raw data
kc$import_tabular(filepath = f, to_tibble = FALSE)
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-import_features"></a>}}
\if{latex}{\out{\hypertarget{method-import_features}{}}}
\subsection{Method \code{import_features()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$import_features(filepath, to_tibble = TRUE, fileext = ".gtf")}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{filepath}}{the filepath to use as import, must contain the file extention.}

\item{\code{to_tibble}}{returns the result as tibble? If FALSE, the raw default 
rtracklayer::import() format will be used (default: TRUE).}

\item{\code{fileext}}{the file extension (default: ".gtf").}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Import method for features data.
Works with BED, GTF, GFFx, and GZIPped formats.
}

\subsection{Returns}{
data contained in the file as a tibble, or NULL.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# get sample files
f_gff <- system.file("extdata", "chr_y.gff3.gz", package = "kibior")
f_bed <- system.file("extdata", "cpg.bed", package = "kibior")
# import to in-memory variable
kc$import_features(filepath = f_bed)
kc$import_features(filepath = f_gff)
# import raw data
kc$import_features(filepath = f_bed, to_tibble = FALSE)
kc$import_features(filepath = f_gff, to_tibble = FALSE)
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-import_alignments"></a>}}
\if{latex}{\out{\hypertarget{method-import_alignments}{}}}
\subsection{Method \code{import_alignments()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$import_alignments(filepath, to_tibble = TRUE, fileext = ".bam")}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{filepath}}{the filepath to use as import, should contain the file extention.}

\item{\code{to_tibble}}{returns the result as tibble? If FALSE, the raw default 
Rsamtools::scanBam() format will be used (default: TRUE).}

\item{\code{fileext}}{the file extension (default: ".bam").}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Import method for alignments data.
Works with BAM format.
}

\subsection{Returns}{
data contained in the file as a tibble, or NULL.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# get sample file
f_bai <- system.file("extdata", "test.bam.bai", package = "kibior")
# import to in-memory variable
kc$import_alignments(filepath = f_bai)
# import raw data
kc$import_alignments(filepath = f_bai, to_tibble = FALSE)
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-import_json"></a>}}
\if{latex}{\out{\hypertarget{method-import_json}{}}}
\subsection{Method \code{import_json()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$import_json(filepath, to_tibble = TRUE, fileext = ".json")}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{filepath}}{the filepath to use as import, should contain the file extention.}

\item{\code{to_tibble}}{returns the result as tibble? If FALSE, the raw dataframe format 
will be used (default: TRUE).}

\item{\code{fileext}}{the file extension (default: ".json").}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Import method for JSON format.
}

\subsection{Returns}{
data contained in the file as a tibble, dataframe or NULL.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# get sample file
f_json <- system.file("extdata", "storms100.json", package = "kibior")
# import to in-memory variable
kc$import_json(f_json)
# import raw data
kc$import_json(f_json, to_tibble = FALSE)
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-import_sequences"></a>}}
\if{latex}{\out{\hypertarget{method-import_sequences}{}}}
\subsection{Method \code{import_sequences()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$import_sequences(filepath, to_tibble = TRUE, fasta_type = "auto")}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{filepath}}{the filepath to use as import, should contain the file extention.}

\item{\code{to_tibble}}{returns the result as tibble? If FALSE, the raw default 
Rsamtools::scanBam() format will be used (default: TRUE).}

\item{\code{fasta_type}}{type of parsing. It can be "dna", "rna", "aa" ou "auto" (default: 
"auto")}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Import method for sequences data.
Works with FASTA formats.
}

\subsection{Returns}{
data contained in the file as a tibble, or NULL.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# get sample file
f_dna <- system.file("extdata", "dna_human_y.fa.gz", package = "kibior")
f_rna <- system.file("extdata", "ncrna_mus_musculus.fa.gz", package = "kibior")
f_aa <- system.file("extdata", "pep_mus_spretus.fa.gz", package = "kibior")
# import to in-memory variable
kc$import_sequences(filepath = f_dna, fasta_type = "dna")
# import raw data
kc$import_sequences(filepath = f_rna, to_tibble = FALSE, fasta_type = "rna")
# import auto
kc$import_sequences(filepath = f_aa)
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-guess_import"></a>}}
\if{latex}{\out{\hypertarget{method-guess_import}{}}}
\subsection{Method \code{guess_import()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$guess_import(filepath, to_tibble = TRUE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{filepath}}{the filepath to use as import, must contain the file extention.}

\item{\code{to_tibble}}{returns the result as tibble? (default: TRUE).}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Import method that will try to guess importation method.
Will also try to read from compressed data if they are.
This method will call other import_* methods when trying.
Some data formats are not installed by default.
Use `rio::install_formats()` to be able to parse them.
}

\subsection{Returns}{
data contained in the file, or NULL.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# get sample file
f_dna <- system.file("extdata", "dna_human_y.fa.gz", package = "kibior")
f_rna <- system.file("extdata", "ncrna_mus_musculus.fa.gz", package = "kibior")
f_aa <- system.file("extdata", "pep_mus_spretus.fa.gz", package = "kibior")
f_bai <- system.file("extdata", "test.bam.bai", package = "kibior")
f_gff <- system.file("extdata", "chr_y.gff3.gz", package = "kibior")
f_bed <- system.file("extdata", "cpg.bed", package = "kibior")
# import 
kc$guess_import(f_dna)
kc$guess_import(f_rna)
kc$guess_import(f_aa)
kc$guess_import(f_bai)
kc$guess_import(f_gff)
kc$guess_import(f_bed)
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-import"></a>}}
\if{latex}{\out{\hypertarget{method-import}{}}}
\subsection{Method \code{import()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$import(
  filepath,
  import_type = "auto",
  push_index = NULL,
  push_mode = "check",
  id_col = NULL,
  to_tibble = TRUE
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{filepath}}{the filepath to use as import, must contain the file extention.}

\item{\code{import_type}}{can be one of "auto", "tabular", "features", "alignments", "sequences" 
(default: "auto").}

\item{\code{push_index}}{the name of the index where to push data (default: NULL).}

\item{\code{push_mode}}{the push mode (default: "check").}

\item{\code{id_col}}{the column name of unique IDs (default: NULL).}

\item{\code{to_tibble}}{returns the result as tibble? (default: TRUE).}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Generic import method.
This method will call other import_* methods when trying.
Some data formats are not installed by default.
}

\subsection{Returns}{
data contained in the file, or NULL.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# get sample file
f_aa <- system.file("extdata", "pep_mus_spretus.fa.gz", package = "kibior")
f_gff <- system.file("extdata", "chr_y.gff3.gz", package = "kibior")
f_bai <- system.file("extdata", "test.bam.bai", package = "kibior")
# import 
kc$import(filepath = f_aa)
# import to Elasticsearch index ("sw_from_file") if not exists
kc$import(filepath = f_bai, push_index = "sw_from_file")
# import to index by recreating it, then pull indexed data
kc$import(filepath = f_gff, push_index = "sw_from_file",
 push_mode = "recreate")
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-push"></a>}}
\if{latex}{\out{\hypertarget{method-push}{}}}
\subsection{Method \code{push()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$push(data, index_name, bulk_size = 1000, mode = "check", id_col = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{data}}{the data to push.}

\item{\code{index_name}}{the index name to use in Elasticsearch.}

\item{\code{bulk_size}}{the number of record to send to Elasticsearch in a row (default: 1000).}

\item{\code{mode}}{the push mode, could be "check", "recreate" or "update" (default: "check").}

\item{\code{id_col}}{an column anme to use as ID, must be composed of unique elements (default: 
NULL).}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Push data from in-memory to Elasticsearch.
Everything is done by bulk.
}

\subsection{Returns}{
the index_name given if the push ended well, else an error.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# erase the last push data by recreating the index and re-pushing data
kc$push(dplyr::starwars, index_name = "sw", mode = "recreate")
# characters names are unique, can be used as ID
kc$push(dplyr::starwars, index_name = "sw", mode = "recreate", id_col = "name")
# a bit more complicated: update some data of the dataset "starwars"
# 38 records on 87 filtered
some_new_data <- dplyr::filter(dplyr::starwars, height > 180)
# make them all "gender <- female"
some_new_data["gender"] <- "female"
# update that apply, based on cahracter names to match the right record
kc$push(some_new_data, "sw", mode = "update", id_col = "name")
# view result by querying
kc$pull("sw", query = "height:>180", columns = c("name", "gender"))
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-pull"></a>}}
\if{latex}{\out{\hypertarget{method-pull}{}}}
\subsection{Method \code{pull()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$pull(
  index_name,
  bulk_size = 500,
  max_size = NULL,
  scroll_timer = "3m",
  keep_metadata = FALSE,
  columns = NULL,
  query = NULL
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{index_name}}{the index name to use in Elasticsearch.}

\item{\code{bulk_size}}{the number of record to send to Elasticsearch in a row (default: 500).}

\item{\code{max_size}}{the number of record Elasticsearch will send (default: NULL (all data)).}

\item{\code{scroll_timer}}{the time the scroll API will let the request alive to scroll on the 
result (default: "3m" (3 minute)).}

\item{\code{keep_metadata}}{does Elasticsearch needs to sent metadata? Data columns will be 
prefixed by "_source." (default: FALSE).}

\item{\code{columns}}{a vector of columns to select (default: NULL (all columns)).}

\item{\code{query}}{a string formatted to Elasticsearch query syntax, see links for the syntax 
 details (default: NULL)

# Simple syntax details:}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Pull data from Elasticsearch.
Everything is done by bulk.
This method is essentially a wrapper around `$search()` with parameter `head = FALSE`
}

\subsection{Returns}{
a list of datasets corresponding to the pull request, else an error. Keys of the 
 list are index names matching the request, value are the associated tibbles
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# push some data sample
kc$push(dplyr::storms, "storms")
# get the whole "sw" index
kc$pull("sw")
# get the whole "sw" index with all metadata
kc$pull("sw", keep_metadata = TRUE)
# get only "name" and "status" columns of indices starting with "s"
# columns not found will be ignored
kc$pull("s*", columns = c("name", "status"))
# limit the size of the result to 10
kc$pull("storms", max_size = 10, bulk_size = 10)
# use Elasticsearch query syntax to select and filter on all indices, for all data
# Here, we want to search for all records taht match the conditions:
# field "height" is strictly more than 180 AND field homeworld is "Tatooine" OR "Naboo"
r <- kc$pull("sw", query = "height:>180 && homeworld:(Tatooine || Naboo)")
# it can be used in conjunction with `columns` to select only columns that matter
r <- kc$pull("sw", query = "height:>180 && homeworld:(Tatooine || Naboo)", columns = 
 c("name", "hair_color", "homeworld"))
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-move"></a>}}
\if{latex}{\out{\hypertarget{method-move}{}}}
\subsection{Method \code{move()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$move(
  from_index,
  to_index,
  from_instance = NULL,
  force = FALSE,
  copy = FALSE
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{from_index}}{The source index name (default: NULL).}

\item{\code{to_index}}{The destination index name (default: NULL).}

\item{\code{from_instance}}{If not NULL, the Kibior object of another instance. if NULL 
(default), this instance will be used. (default: NULL).}

\item{\code{force}}{Does the destination index need to be erase? (default: FALSE)}

\item{\code{copy}}{Does the destination have to be a copy of the source? FALSE (default) will 
delete source index, TRUE will keep it. (default: FALSE).}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Move data from one index to another.
It needs to be configured in the `config/elasticsearch.yml` file to actually work.
}

\subsection{Returns}{
the reindex result
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
kc$push(dplyr::starwars, "sw", mode = "recreate")
# move data from an index to another (change name, same instance)
r <- kc$move(from_index = "sw", to_index = "sw_new")
kc$pull("sw_new")
kc$list()
} 

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-copy"></a>}}
\if{latex}{\out{\hypertarget{method-copy}{}}}
\subsection{Method \code{copy()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$copy(from_index, to_index, from_instance = NULL, force = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{from_index}}{The source index name (default: NULL).}

\item{\code{to_index}}{The destination index name (default: NULL).}

\item{\code{from_instance}}{If not NULL, the Kibior object of another instance. if NULL 
(default), this instance will be used. (default: NULL).}

\item{\code{force}}{Does the destination index need to be erase? (default: FALSE)}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Copy data from one index to another.
It needs to be configured in the `config/elasticsearch.yml` file to actually work.
This method is a wrapper around `$move(copy = TRUE)`.
}

\subsection{Returns}{
the reindex result
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# copy data from one index to another (same instance)
r <- kc$copy(from_index = "sw_new", to_index = "sw")
kc$pull(c("sw", "sw_new"))
kc$list()
} 

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-match"></a>}}
\if{latex}{\out{\hypertarget{method-match}{}}}
\subsection{Method \code{match()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$match(index_name)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{index_name}}{the index name to use in Elasticsearch, can be a pattern with '*'.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Match requested index names against Elasticsearch indices list.
}

\subsection{Returns}{
a vector of matching index names, NULL if nothing matches.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# search "sw" index name
kc$match("sw")
# search all starting with an "s"
kc$match("s*")
# get all index name, identical to `$list()`
kc$match("*")
# search multiple names 
kc$match(c("sw", "sw_new", "nope"))
# search multiple names with pattern
kc$match(c("s*", "nope"))
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-search"></a>}}
\if{latex}{\out{\hypertarget{method-search}{}}}
\subsection{Method \code{search()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$search(
  index_name = "_all",
  keep_metadata = FALSE,
  columns = NULL,
  bulk_size = 500,
  max_size = NULL,
  scroll_timer = "3m",
  head = TRUE,
  query = NULL
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{index_name}}{the index name to use in Elasticsearch (default: NULL).}

\item{\code{keep_metadata}}{does Elasticsearch needs to sent metadata? Data columns will be 
prefixed by "_source." (default: FALSE).}

\item{\code{columns}}{a vector of columns to select (default: NULL (all columns)).}

\item{\code{bulk_size}}{the number of record to send to Elasticsearch in a row (default: 500).}

\item{\code{max_size}}{the number of record Elasticsearch will send (default: NULL (all data)).}

\item{\code{scroll_timer}}{the time the scroll API will let the request alive to scroll on the 
result (default: "3m" (3 minutes)).}

\item{\code{head}}{a boolean limiting the search result and time (default: TRUE)}

\item{\code{query}}{a string formatted to Elasticsearch query syntax, see links for the syntax 
details (default: NULL)}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Search data from Elasticsearch.
The goal of this method is to discover quickly what data are interesting, thus 
 `head = TRUE` by default.
If you want to get all data, use `head = FALSE` or `$pull()`.
Everything is done by bulk.
}

\subsection{Returns}{
a list of datasets corresponding to the pull request, else an error. Keys of the 
 list are index names matching the request, value are the associated tibbles
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# search "sw" index, head mode on
kc$search("sw")
# search "sw" index with all metadata, head mode on
kc$search("sw", keep_metadata = TRUE)
# get only "name" field of the head of indices starting with "s"
# if an index does not have the "name" field, it will be empty
kc$search("s*", columns = "name")
# limit the size of the result to 50 to the whole index
kc$search("storms", max_size = 50, bulk_size = 50, head = FALSE)
# use Elasticsearch query syntax to select and filter on all indices, for all data
# Here, we want to search for all records taht match the conditions:
# field "height" is strictly more than 180 AND field homeworld is "Tatooine" OR "Naboo"
kc$search("*", query = "height:>180 && homeworld:(Tatooine || Naboo)")
# it can be used in conjunction with `columns` to select only columns that matter
kc$search("*", query = "height:>180 && homeworld:(Tatooine || Naboo)", columns = 
 c("name", "hair_color", "homeworld"))
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-inner_join"></a>}}
\if{latex}{\out{\hypertarget{method-inner_join}{}}}
\subsection{Method \code{inner_join()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$inner_join(...)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{...}}{see `join()` params.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Execute a inner join between two datasets using `dplyr` joins.
The datasets can be in-memory (variable name) or the name of an currently stored 
 Elasticsearch index. Joins cannot be done on column of type "list" ("by" 
 argument).
}

\subsection{Returns}{
a tibble
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# some data for joins examples
kc$push(ggplot2::diamonds, "diamonds")
# prepare join datasets, only big the biggest diamonds are selected (9)
sup_carat <- dplyr::filter(ggplot2::diamonds, carat > 3.5)
r <- kc$push(sup_carat, "diamonds_superior")
# execute a inner_join with one index and one in-memory dataset
kc$inner_join(ggplot2::diamonds, "diamonds_superior")
# execute a inner_join with one index queried, and one in-memory dataset
kc$inner_join(ggplot2::diamonds, "diamonds", right_query 
 = "carat:>3.5")
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-full_join"></a>}}
\if{latex}{\out{\hypertarget{method-full_join}{}}}
\subsection{Method \code{full_join()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$full_join(...)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{...}}{see `join()` params.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Execute a full join between two datasets using `dplyr` joins.
The datasets can be in-memory (variable name) or the name of an currently stored 
 Elasticsearch index. Joins cannot be done on column of type "list" ("by" 
 argument).
}

\subsection{Returns}{
a tibble
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# prepare join datasets, fair cuts 
fair_cut <- dplyr::filter(ggplot2::diamonds, cut == "Fair")  # 1605 lines
sup_carat <- kc$pull("diamonds_superior")$diamonds_superior
# execute a full_join with one index and one in-memory dataset
kc$full_join(fair_cut, "diamonds_superior")
# execute a full_join with one index queried, and one in-memory dataset
kc$full_join(sup_carat, "diamonds", right_query = "cut:fair")
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-left_join"></a>}}
\if{latex}{\out{\hypertarget{method-left_join}{}}}
\subsection{Method \code{left_join()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$left_join(...)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{...}}{see `join()` params.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Execute a left join between two datasets using `dplyr` joins.
The datasets can be in-memory (variable name) or the name of an currently stored 
 Elasticsearch index. Joins cannot be done on column of type "list" ("by" 
 argument).
}

\subsection{Returns}{
a tibble
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# prepare join datasets, fair cuts 
fair_cut <- dplyr::filter(ggplot2::diamonds, cut == "Fair")  # 1605 lines
sup_carat <- kc$pull("diamonds_superior")$diamonds_superior
# execute a left_join with one index and one in-memory dataset
kc$left_join(fair_cut, "diamonds_superior")
# execute a left_join with one index queried, and one in-memory dataset
kc$left_join(sup_carat, "diamonds", right_query 
 = "cut:fair")
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-right_join"></a>}}
\if{latex}{\out{\hypertarget{method-right_join}{}}}
\subsection{Method \code{right_join()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$right_join(...)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{...}}{see `join()` params.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Execute a right join between two datasets using `dplyr` joins.
The datasets can be in-memory (variable name) or the name of an currently stored 
 Elasticsearch index. Joins cannot be done on column of type "list" ("by" 
 argument).
}

\subsection{Returns}{
a tibble
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# prepare join datasets, fair cuts 
fair_cut <- dplyr::filter(ggplot2::diamonds, cut == "Fair")  # 1605 lines
sup_carat <- kc$pull("diamonds_superior")$diamonds_superior
# execute a right_join with one index and one in-memory dataset
kc$right_join(fair_cut, "diamonds_superior")
# execute a right_join with one index queried, and one in-memory dataset
kc$right_join(sup_carat, "diamonds", right_query 
 = "cut:fair")
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-semi_join"></a>}}
\if{latex}{\out{\hypertarget{method-semi_join}{}}}
\subsection{Method \code{semi_join()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$semi_join(...)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{...}}{see `join()` params.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Execute a semi join between two datasets using `dplyr` joins.
The datasets can be in-memory (variable name) or the name of an currently stored 
 Elasticsearch index. Joins cannot be done on column of type "list" ("by" 
 argument).
}

\subsection{Returns}{
a tibble
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# prepare join datasets, fair cuts 
fair_cut <- dplyr::filter(ggplot2::diamonds, cut == "Fair")  # 1605 lines
sup_carat <- kc$pull("diamonds_superior")$diamonds_superior
# execute a semi_join with one index and one in-memory dataset
kc$semi_join(fair_cut, "diamonds_superior")
# execute a semi_join with one index queried, and one in-memory dataset
kc$semi_join(sup_carat, "diamonds", right_query 
 = "cut:fair")
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-anti_join"></a>}}
\if{latex}{\out{\hypertarget{method-anti_join}{}}}
\subsection{Method \code{anti_join()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$anti_join(...)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{...}}{see `join()` params.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Execute a anti join between two datasets using `dplyr` joins.
The datasets can be in-memory (variable name) or the name of an currently stored 
 Elasticsearch index. Joins cannot be done on column of type "list" ("by" 
 argument).
}

\subsection{Returns}{
a tibble
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# prepare join datasets, fair cuts 
fair_cut <- dplyr::filter(ggplot2::diamonds, cut == "Fair")  # 1605 lines
sup_carat <- kc$pull("diamonds_superior")$diamonds_superior
# execute a anti_join with one index and one in-memory dataset
kc$anti_join(fair_cut, "diamonds_superior")
# execute a anti_join with one index queried, and one in-memory dataset
kc$anti_join(sup_carat, "diamonds", right_query 
 = "cut:fair")
# 
# Do not mind this, removing example indices
elastic::index_delete(kc$connection, "*")
kc <- NULL
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-clone"></a>}}
\if{latex}{\out{\hypertarget{method-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Kibior$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
