% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bcmixmeta.R
\name{bcmixmeta}
\alias{bcmixmeta}
\title{Bias Corrected Meta-Analysis with Dirichlet Mixture Process Priors for the biased component}
\usage{
bcmixmeta(
  data,
  x = NULL,
  mean.mu.0 = 0,
  sd.mu.0 = 10,
  scale.sigma.between = 0.5,
  df.scale.between = 1,
  scale.sigma.beta = 0.5,
  df.scale.beta = 1,
  B.lower = -15,
  B.upper = 15,
  a.0 = 0.5,
  a.1 = 1,
  alpha.0 = 0.03,
  alpha.1 = 2,
  K = 10,
  bilateral.bias = FALSE,
  nr.chains = 2,
  nr.iterations = 10000,
  nr.adapt = 1000,
  nr.burnin = 1000,
  nr.thin = 1,
  be.quiet = FALSE
)
}
\arguments{
\item{data}{A data frame with at least two columns with the following names:
1) TE = treatment effect,
2) seTE = the standard error of the treatment effect.}

\item{x}{a covariate to perform meta-regression.}

\item{mean.mu.0}{Prior mean of the mean of the base distribution default value is mean.mu.0 = 0.}

\item{sd.mu.0}{Prior standard deviation of the base distribution, the default value is 10^-6.}

\item{scale.sigma.between}{Prior scale parameter for scale gamma distribution for the
precision between studies. The default value is 0.5.}

\item{df.scale.between}{Degrees of freedom of the scale gamma distribution for the precision between studies.
The default value is 1, which results in a Half Cauchy distribution for the standard
deviation between studies. Larger values e.g. 30 corresponds to a Half Normal distribution.}

\item{scale.sigma.beta}{Prior scale parameter for the scale.gamma distribution for the
precision between study biases.}

\item{df.scale.beta}{Degrees of freedom of the scale gamma distribution for the precision between
study biases. The default value is 1, which results in a Half Cauchy distribution
for the standard deviation between biases.}

\item{B.lower}{Lower bound of the bias parameter B, the default value is -15.}

\item{B.upper}{Upper bound of the bias parameter B, the default value is 15.}

\item{a.0}{Parameter for the prior Beta distribution for the probability of bias. Default value is a0 = 0.5.}

\item{a.1}{Parameter for the prior Beta distribution for the probability of bias. Default value is a1 = 1.}

\item{alpha.0}{Lower bound of the uniform prior for the concentration parameter for the DP,
the default value is 0.5.}

\item{alpha.1}{Upper bound of the uniform prior for the concentration parameter for the DP,
the default value depends on the sample size, see the example below. We give as
working value alpha.1 = 2}

\item{K}{Maximum number of clusters in the DP, the default value depends on alpha.1, see the
example below. We give as working value K = 10.}

\item{bilateral.bias}{Experimental option, which indicates if bias could be to the left and
to the right of the model of interest. If bilateral.bias==TRUE,
then the function generates three mean and sorts the means in two
groups: mean_bias_left, mean_theta, mean_bias_right.}

\item{nr.chains}{Number of chains for the MCMC computations, default 2.}

\item{nr.iterations}{Number of iterations after adapting the MCMC, default is 10000. Some models may need more iterations.}

\item{nr.adapt}{Number of iterations in the adaptation process, default is 1000. Some models may need more iterations during adptation.}

\item{nr.burnin}{Number of iteration discard for burn-in period, default is 1000. Some models may need a longer burnin period.}

\item{nr.thin}{Thinning rate, it must be a positive integer, the default value 1.}

\item{be.quiet}{Do not print warning message if the model does not adapt. The default value is FALSE. If you are not sure about the adaptation period choose be.quiet=TRUE.}
}
\value{
This function returns an object of the class "bcmixmeta". This object contains the MCMC
                           output of each parameter and hyper-parameter in the model and
                           the data frame used for fitting the model.
}
\description{
This function performers a Bayesian meta-analysis with DPM as random effects
}
\details{
The results of the object of the class bcmixmeta can be extracted with R2jags or with rjags. In addition a summary, a print and a plot functions are
implemented for this type of object.
}
\examples{
\dontrun{
library(jarbes)


# Example: Stemcells

data("stemcells")
stemcells$TE = stemcells$effect.size
stemcells$seTE = stemcells$se.effect

# Beta(0.5, 1)
a.0 = 0.5
a.1 = 1

# alpha.max
 N = dim(stemcells)[1]
 alpha.max = 1/5 *((N-1)*a.0 - a.1)/(a.0 + a.1)

alpha.max

# K.max
K.max = 1 + 5*alpha.max
K.max = round(K.max)

K.max

set.seed(20233)

bcmix.2.stemcell = bcmixmeta(stemcells,
                            mean.mu.0=0, sd.mu.0=100,
                            B.lower = -15,
                            B.upper = 15,
                            alpha.0 = 0.5,
                            alpha.1 = alpha.max,
                            a.0 = a.0,
                            a.1 = a.1,
                            K = K.max,
                            sort.priors = FALSE,
                            df.scale.between = 1,
                            scale.sigma.between = 0.5,
                            nr.chains = 4,
                            nr.iterations = 50000,
                            nr.adapt = 1000,
                            nr.burnin = 10000,
                            nr.thin = 4)


 diagnostic(bcmix.2.stemcell, y.lim = c(-1, 15), title.plot = "Default priors")


 bcmix.2.stemcell.mcmc <- as.mcmc(bcmix.1.stemcell$BUGSoutput$sims.matrix)


theta.names <- paste(paste("theta[",1:31, sep=""),"]", sep="")
theta.b.names <- paste(paste("theta.bias[",1:31, sep=""),"]", sep="")

theta.b.greek.names <- paste(paste("theta[",1:31, sep=""),"]^B", sep="")
theta.greek.names <- paste(paste("theta[",1:31, sep=""),"]", sep="")


caterplot(bcmix.2.stemcell.mcmc,
         parms = theta.names,               # theta
         labels = theta.greek.names,
         greek = T,
         labels.loc="axis", cex =0.7,
         col = "black",
         style = "plain",
         reorder = F,
         val.lim =c(-6, 16),
         quantiles = list(outer=c(0.05,0.95),inner=c(0.16,0.84)),
         x.lab = "Effect: mean difference"
)
title( "95\% posterior intervals of studies' effects")
caterplot(bcmix.2.stemcell.mcmc,
         parms = theta.b.names,             # theta.bias
         labels = theta.greek.names,
         greek = T,
         labels.loc="no",
         cex = 0.7,
         col = "grey",
         style = "plain", reorder = F,
         val.lim =c(-6, 16),
         quantiles = list(outer=c(0.025,0.975),inner=c(0.16,0.84)),
         add = TRUE,
         collapse=TRUE, cat.shift= -0.5,
)

attach.jags(bcmix.2.stemcell, overwrite = TRUE)
abline(v=mean(mu.0), lwd =2, lty =2)

legend(9, 20, legend = c("bias corrected", "biased"),
     lty = c(1,1), lwd = c(2,2), col = c("black", "grey"))


}

}
\references{
Verde, P.E. and Rosner, G. L. (2024) A Bias-Corrected Bayesian Nonparamteric Model for Combining
             Studies with Varying Quality in Meta-Analysis. Biometrical Journal; (under revision).
}
