#' Generic summary function for bmeta object in jarbes
#' @param object The object generated by the bmeta function.
#'
#' @param digits The number of significant digits printed. The default value is 3.
#' @param ... \dots
#'
#' @export
summary.b3lmeta = function(object, digits = 3, ...) {

  bugs.output = object$BUGSoutput
  bugs.summary = bugs.output$summary

  summary.m = list()

  # Model specifications ....
  #
  model.spec = list()
  model.spec$link = "Normal approximation"

  # Hyper-priors parameters............................................
  model.spec$mean.mu.0     = object$prior$mean.mu.0
  model.spec$sd.mu.0       = object$prior$sd.mu.0
  model.spec$scale.sigma.between = object$prior$scale.sigma.between
  model.spec$df.scale.between    = object$prior$df.scale.between
  model.spec$scale.sigma.within  = object$prior$scale.sigma.within
  model.spec$df.scale.within     = object$prior$df.scale.within

  summary.m$model.specification  = model.spec

  # Posterior of the model parameters
  #
  # The list of parameters will include more complex models, e.g. estimation of
  # the parameters nu from the Beta ...
  #

  Ndesign = object$Ndesign

  row.names.list = c("mu.0", "mu.0.new",
                     "tau.theta.between",
                     paste0("tau.theta.within","[",1:Ndesign,"]"),
                     "tau.theta.total")

  var.names.list = c("Mean (Pooled mean)",
                     "Predictive effect",
                     "Tau (between studies sd)",
                     paste0("Tau (within)","[",1:Ndesign,"]"),
                     "Tau (total)")

  summary.m$summary.par = bugs.summary[row.names.list, ]

  row.names(summary.m$summary.par) = var.names.list

  # DIC
  summary.m$DIC = bugs.output$DIC
  summary.m$pD = bugs.output$pD

  # MCMC setup ...
  mcmc.setup = list()
  mcmc.setup$n.chains = bugs.output$n.chains
  mcmc.setup$n.iter = bugs.output$n.iter
  mcmc.setup$n.burnin = bugs.output$n.burnin
  summary.m$mcmc.setup = mcmc.setup

  class(summary.m) = "summary.b3lmeta"

  print(summary.m, digits, ...)

}

print.summary.b3lmeta = function(x, digits, ...) {
  cat('Model specifications:\n')
  model.spec = x$model.specification

  cat(paste('  Link function: ', model.spec$link, sep = ''))
  cat('\n')
  cat('\n')
  cat('  Hyper-priors parameters: \n')
  cat(paste('  Prior for mu: Normal', '[', model.spec$mean.mu.0,', ' ,model.spec$sd.mu.0^2,']', sep = ''))
  cat('\n')

  cat(paste('  Prior for 1/tau.between^2: Scale.Gamma', '[', model.spec$scale.sigma.between,', ' ,
            model.spec$df.scale.between,']', sep = ''))
  cat('\n')

  cat(paste('  Prior for 1/tau.within^2: Scale.Gamma', '[', model.spec$scale.sigma.within,', ' ,
            model.spec$df.scale.within,']', sep = ''))
  cat('\n')

  cat('Posterior distributions: \n')
  print(round(x$summary.par, digits))

  cat('\n-------------------\n')


  mcmc = x$mcmc.setup
  cat(paste('MCMC setup (fit using jags): ', mcmc$n.chains, ' chains, each with ', mcmc$n.iter, ' iterations (first ', mcmc$n.burnin, ' discarded)', sep = ''))
  cat('\n')
  cat(paste('DIC: ', round(x$DIC, digits), sep = ''))
  cat('\n')
  cat(paste('pD: ', round(x$pD, digits), sep = ''))
  cat('\n')
}

