% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/detect_envi_change.R
\name{detect_envi_change}
\alias{detect_envi_change}
\title{Detect areas influenced by a changing environment variable.}
\usage{
detect_envi_change(
  model,
  var_occ,
  variables,
  target_var,
  bins = NULL,
  shap_nsim = 10,
  seed = 10,
  var_future = NULL,
  variables_future = NULL,
  pfun = .pfun_shap,
  method = "gam",
  formula = y ~ s(x)
)
}
\arguments{
\item{model}{(\code{isolation_forest} or other model). It could
be the item \code{model} of \code{POIsotree} made by function \code{\link{isotree_po}}.
It also could be other user-fitted models as long as the \code{pfun} can work on it.}

\item{var_occ}{(\code{data.frame}, \code{tibble}) The \code{data.frame} style table that
include values of environmental variables at occurrence locations.}

\item{variables}{(\code{stars}) The \code{stars} of environmental variables.
It should have multiple \code{attributes} instead of \code{dims}.
If you have \code{raster} object instead, you
could use \code{\link{st_as_stars}} to convert it to \code{stars} or use
\code{\link{read_stars}} directly read source data as a \code{stars}.
You also could use item \code{variables} of \code{POIsotree} made by function
\code{\link{isotree_po}}.}

\item{target_var}{(\code{character}) The selected variable to process.}

\item{bins}{(\code{integer}) The bin to cut the target variable for the analysis.
If it is \code{NULL}, no cut to apply. The default is \code{NULL}.}

\item{shap_nsim}{(\code{integer}) The number of Monte Carlo repetitions in SHAP
method to use for estimating each Shapley value. See details in documentation
of function \code{\link{explain}} in package \code{fastshap}.
When the number of variables is large, a smaller shap_nsim could be used.
Be cautious that making SHAP-based spatial dependence will be slow
because of Monte-Carlo computation for all pixels.
But it is worth the time because it is much more
informative. See details in documentation of function \code{\link{explain}}
in package \code{fastshap}. The default is 10. Usually a value 10 - 20 is enough.}

\item{seed}{(\code{integer}) The seed for any random progress. The default is \code{10L}.}

\item{var_future}{(\code{numeric} or \code{stars}) A number to apply to the current
variable or a \code{stars} layer as the future variable. It can be \code{NULL} if
\code{variables_future} is set.}

\item{variables_future}{(\code{stars}) A \code{stars} raster stack for future variables.
It could be \code{NULL} if \code{var_future} is set.}

\item{pfun}{(\code{function}) The predict function that requires two arguments,
\code{object} and \code{newdata}.
It is only required when \code{model} is not \code{isolation_forest}.
The default is the wrapper function designed for iForest model in \code{itsdm}.}

\item{method}{Argument passed on to \code{\link{geom_smooth}} to fit the line.
Note that the same arguments will be used for all target variables.
User could set variable one by one to set the arguments separately.
Default value is "gam".}

\item{formula}{Argument passed on to \code{\link{geom_smooth}} to fit the line.
Note that the same arguments will be used for all target variables.
User could set variable one by one to set the arguments separately.
The default is y ~ s(x).}
}
\value{
(\code{EnviChange}) A list of
\itemize{
\item{A figure of fitted variable curve}
\item{A map of variable contribiution change}
\item{Tipping points of variable contribution}
\item{A \code{stars} of variable contribution under current and future condition,
and the detected changes}
}
}
\description{
Use shapley values to detect the potential areas that will
impact the species distribution. It only works on continuous variables.
}
\details{
The values show how changes in environmental variable affects the modeling
prediction in space. These maps could help to answer questions of where will
be affected by a changing variable.
}
\examples{
# Using a pseudo presence-only occurrence dataset of
# virtual species provided in this package
library(dplyr)
library(sf)
library(stars)
library(itsdm)
#'
# Prepare data
data("occ_virtual_species")
obs_df <- occ_virtual_species \%>\% filter(usage == "train")
eval_df <- occ_virtual_species \%>\% filter(usage == "eval")
x_col <- "x"
y_col <- "y"
obs_col <- "observation"
#'
# Format the observations
obs_train_eval <- format_observation(
  obs_df = obs_df, eval_df = eval_df,
  x_col = x_col, y_col = y_col, obs_col = obs_col,
  obs_type = "presence_only")
#'
env_vars <- system.file(
  'extdata/bioclim_tanzania_10min.tif',
  package = 'itsdm') \%>\% read_stars() \%>\%
  slice('band', c(1, 12))
#'
# With imperfect_presence mode,
mod <- isotree_po(
  obs_mode = "imperfect_presence",
  obs = obs_train_eval$obs,
  obs_ind_eval = obs_train_eval$eval,
  variables = env_vars, ntrees = 5,
  sample_size = 0.8, ndim = 1L,
  nthreads = 1,
  seed = 123L, response = FALSE,
  spatial_response = FALSE,
  check_variable = FALSE)

# Use a fixed value
bio1_changes <- detect_envi_change(
  model = mod$model,
  var_occ = mod$vars_train,
  variables = mod$variables,
  shap_nsim = 1,
  target_var = "bio1",
  var_future = 5)

\dontrun{
# Use a future layer
## Read the future Worldclim variables
future_vars <- system.file(
  'extdata/future_bioclim_tanzania_10min.tif',
  package = 'itsdm') \%>\% read_stars() \%>\%
  split() \%>\% select(bioc1, bioc12)
# Rename the bands
names(future_vars) <- paste0("bio", c(1, 12))

## Just use the target future variable
climate_changes <- detect_envi_change(
  model = mod$model,
  var_occ = mod$vars_train,
  variables = mod$variables,
  shap_nsim = 1,
  target_var = "bio1",
  var_future = future_vars \%>\% select("bio1"))

## Use the whole future variable tack
bio12_changes <- detect_envi_change(
  model = mod$model,
  var_occ = mod$vars_train,
  variables = mod$variables,
  shap_nsim = 1,
  target_var = "bio12",
  variables_future = future_vars)

print(bio12_changes)

##### Use Random Forest model as an external model ########
library(randomForest)

# Prepare data
data("occ_virtual_species")
obs_df <- occ_virtual_species \%>\%
  filter(usage == "train")

env_vars <- system.file(
  'extdata/bioclim_tanzania_10min.tif',
  package = 'itsdm') \%>\% read_stars() \%>\%
  slice('band', c(1, 5, 12)) \%>\%
  split()

model_data <- stars::st_extract(
  env_vars, at = as.matrix(obs_df \%>\% select(x, y))) \%>\%
  as.data.frame()
names(model_data) <- names(env_vars)
model_data <- model_data \%>\%
  mutate(occ = obs_df[['observation']])
model_data$occ <- as.factor(model_data$occ)

mod_rf <- randomForest(
  occ ~ .,
  data = model_data,
  ntree = 200)

pfun <- function(X.model, newdata) {
  # for data.frame
  predict(X.model, newdata, type = "prob")[, "1"]
}

# Use a fixed value
bio5_changes <- detect_envi_change(
  model = mod_rf,
  var_occ = model_data \%>\% select(-occ),
  variables = env_vars,
  target_var = "bio5",
  bins = 20,
  var_future = 5,
  pfun = pfun)

plot(bio5_changes)
}

}
\seealso{
\code{\link{shap_spatial_response}}
}
