% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/isoforest.R
\name{predict.isolation_forest}
\alias{predict.isolation_forest}
\title{Predict method for Isolation Forest}
\usage{
\method{predict}{isolation_forest}(
  object,
  newdata,
  type = "score",
  square_mat = FALSE,
  refdata = NULL,
  ...
)
}
\arguments{
\item{object}{An Isolation Forest object as returned by \link{isolation.forest}.}

\item{newdata}{A `data.frame`, `data.table`, `tibble`, `matrix`, or sparse matrix (from package `Matrix` or `SparseM`,
CSC/dgCMatrix supported for distance and outlierness, CSR/dgRMatrix supported for outlierness and imputations)
for which to predict outlierness, distance, or imputations of missing values.

Note that, if using sparse matrices from package `Matrix`, converting to `dgRMatrix` (when required) might require using
`as(m, "RsparseMatrix")` instead of `dgRMatrix` directly.
Nevertheless, if `newdata` is sparse and one wants to obtain the outlier score or average depth or tree
numbers, it's highly recommended to pass it in CSC (`dgCMatrix`) format as it will be much faster
when the number of trees or rows is large.}

\item{type}{Type of prediction to output. Options are:
\itemize{
  \item `"score"` for the standardized outlier score, where values closer to 1 indicate more outlierness, while values
  closer to 0.5 indicate average outlierness, and close to 0 more averageness (harder to isolate).
  \item `"avg_depth"` for  the non-standardized average isolation depth.
  \item `"dist"` for approximate pairwise or between-points distances (must pass more than 1 row) - these are
  standardized in the same way as outlierness, values closer to zero indicate nearer points,
  closer to one further away points, and closer to 0.5 average distance.
  \item `"avg_sep"` for the non-standardized average separation depth.
  \item `"tree_num"` for the terminal node number for each tree - if choosing this option,
  will return a list containing both the average isolation depth and the terminal node numbers, under entries
  `avg_depth` and `tree_num`, respectively.
  \item `"impute"` for imputation of missing values in `newdata`.
}}

\item{square_mat}{When passing `type` = `"dist` or `"avg_sep"` with no `refdata`, whether to return a
just the upper-triangular part (returned as a `dist` object and compatible with functions such as `hclust`),
in which the entry for pair (i,j) with 1 <= i < j <= n is located at position
p(i, j) = ((i - 1) * (n - i/2) + j - i).
Ignored when not predicting distance/separation or when passing `refdata`.}

\item{refdata}{If passing this and calculating distance or average separation depth, will calculate distances
between each point in `newdata` and each point in `refdata`, outputing a matrix in which points in `newdata`
correspond to rows and points in `refdata` correspond to columns. Must be of the same type as `newdata` (e.g.
`data.frame`, `matrix`, `dgCMatrix`, etc.). If this is not passed, and type is `"dist"`
or `"avg_sep"`, will calculate pairwise distances/separation between the points in `newdata`.}

\item{...}{Not used.}
}
\value{
The requested prediction type, which can be: \itemize{
\item A numeric vector with one entry per row in `newdata` (for output types `"score"`, `"avg_depth"`).
\item A list with entries `avg_depth` (numeric vector)
and `tree_num` (integer matrix indicating the terminal node number under each tree for each
observation, with trees as columns), for output type
`"tree_num"`.
\item A numeric square matrix or `dist` object containing a vector with the upper triangular
part of a square matrix
(for output types `"dist"`, `"avg_sep"`, with no `refdata`).
\item A numeric matrix with points in `newdata` as rows and points in `refdata` as columns
(for output types `"dist"`, `"avg_sep"`, with `refdata`).
\item The same type as the input `newdata` (for output type `"impute"`).}
}
\description{
Predict method for Isolation Forest
}
\details{
The standardized outlier score is calculated according to the original paper's formula:
\eqn{  2^{ - \frac{\bar{d}}{c(n)}  }  }{2^(-avg(depth)/c(nobs))}, where
\eqn{\bar{d}}{avg(depth)} is the average depth under each tree at which an observation
becomes isolated (a remainder is extrapolated if the actual terminal node is not isolated),
and \eqn{c(n)}{c(nobs)} is the expected isolation depth if observations were uniformly random
(see references under \link{isolation.forest} for details). The actual calculation
of \eqn{c(n)}{c(nobs)} however differs from the paper as this package uses more exact procedures
for calculation of harmonic numbers.

The distribution of outlier scores should be centered around 0.5, unless using non-random splits (parameters
`prob_pick_avg_gain`, `prob_pick_pooled_gain`, `prob_split_avg_gain`, `prob_split_pooled_gain`)
and/or range penalizations, or having distributions which are too skewed.

The more threads that are set for the model, the higher the memory requirement will be as each
thread will allocate an array with one entry per row (outlierness) or combination (distance).

Outlierness predictions for sparse data will be much slower than for dense data. Not recommended to pass
sparse matrices unless they are too big to fit in memory.

Note that after loading a serialized object from `isolation.forest` through `readRDS` or `load`,
it will only de-serialize the underlying C++ object upon running `predict`, `print`, or `summary`, so the
first run will  be slower, while subsequent runs will be faster as the C++ object will already be in-memory.

In order to save memory when fitting and serializing models, the functionality for outputting
terminal node numbers will generate index mappings on the fly for all tree nodes, even if passing only
1 row, so it's only recommended for batch predictions.

The outlier scores/depth predict functionality is optimized for making predictions on one or a
few rows at a time - for making large batches of predictions, it might be faster to use the
option `output_score=TRUE` in `isolation.forest`.

When making predictions on CSC matrices with many rows using multiple threads, there
can be small differences between runs due to roundoff error.

When imputing missing values, the input may contain new columns (i.e. not present when the model was fitted),
which will be output as-is.
}
\seealso{
\link{isolation.forest} \link{isotree.restore.handle}
}
