#' Output the result of Interval level intensity analysis as csv.
#' @details Gets the output of \code{IIA} function and the path variable and generate a csv report called "IntervalLevelIntensityAnalysis.csv". The output will be stored in "CSVOutput" directory in path direction.
#' @param IIA.output Output list generated by \code{IIA} function.
#' @param time.points a charachter vector showing the time point of each raster layer in chronological order.
#' @param filename A charachter variable including an optional path and a required filename to where the user wants to store the csv output. If only the name of the file is provided, it will be stored in the working directory.
#' @return The output is a CSV file.
#' @importFrom utils write.table
#' @importFrom stats na.omit
#' @export
#' @examples
#' raster_2005 <- raster::raster(system.file("external/RASTER_2005.RST", package="intensity.analysis"))
#' raster_2010 <- raster::raster(system.file("external/RASTER_2010.RST", package="intensity.analysis"))
#' raster_2012 <- raster::raster(system.file("external/RASTER_2012.RST", package="intensity.analysis"))
#' raster.layers <- list(raster_2005, raster_2010, raster_2012)
#' time.points <- c("2005","2010","2012")
#' categories <- c("Water","Trees","Impervious")
#' crosstabulation <- multicrosstab(raster.layers, time.points, categories)
#' IIA.output <- IIA(crosstabulation, time.points)
#' filename <- file.path(normalizePath(tempdir(), winslash = "/"), "IIA.csv")
#' IIA2csv(IIA.output,time.points, filename)


IIA2csv <- function(IIA.output, time.points, filename){

  parameters <- reqpar(time.points)

  output <- chkfilename(filename, expand = TRUE)

  if (file.exists(output)){
    suppressWarnings(file.remove(output, showWarnings = FALSE ))
  }

  out.dataframe.3 <- cbind(IIA.output$Change.Size, IIA.output$Uniform.Intensity.of.Change, IIA.output$Uniform.Change.all.Intervals, IIA.output$Interval.Behavior)
  colnames(out.dataframe.3) <- c( "Change Size", "Annual change", "Uniform Change across Intervals", "Interval Behavior")
  rownames(out.dataframe.3) <- paste(parameters$interval.name)
  write.table(out.dataframe.3, file = output, append = FALSE, col.names=NA ,quote = TRUE, sep = ",")

}
