% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/PartialDependence.R
\name{PartialDependence}
\alias{PartialDependence}
\title{Partial Dependence Plot}
\format{\code{\link{R6Class}} object.}
\description{
\code{PartialDependence} computes and plots partial dependence functions of prediction models.
}
\section{Usage}{

\preformatted{
pdp = PartialDependence$new(predictor, feature, grid.size = 20, run = TRUE)

plot(pdp)
pdp$results
print(pdp)
pdp$set.feature(2)
}
}

\section{Arguments}{


For PartialDependence$new():
\describe{
\item{predictor}{Object of type \code{Predictor}. See \link{Predictor}.}
\item{feature}{The feature name or index for which to compute the partial dependencies. 
Either a single number or vector of two numbers.}
\item{grid.size}{The size of the grid for evaluating the predictions}
\item{run}{logical. Should the Interpretation method be run?}
}
}

\section{Details}{

The partial dependence plot calculates and plots the dependence of f(X) on a single or two features.
To learn more about partial dependence plot, read the Interpretable Machine Learning book: 
https://christophm.github.io/interpretable-ml-book/pdp.html
}

\section{Fields}{

\describe{
\item{feature.index}{The index of the features for which the partial dependence was computed.}
\item{feature.name}{The names of the features for which the partial dependence was computed.}
\item{feature.type}{The detected types of the features, either "categorical" or "numerical".}
\item{grid.size}{The size of the grid.}
\item{n.features}{The number of features (either 1 or 2)}
\item{predictor}{The prediction model that was analysed.}
\item{results}{data.frame with the grid of feature of interest and the predicted \eqn{\hat{y}}. 
Can be used for creating custom partial dependence plots.}
}
}

\section{Methods}{

\describe{
\item{set.feature}{method to get/set feature(s) (by index) fpr  which to compute pdp. See examples for usage.}
\item{plot()}{method to plot the partial dependence function. See \link{plot.PartialDependence}}
\item{\code{run()}}{[internal] method to run the interpretability method. Use \code{obj$run(force = TRUE)} to force a rerun.}
\item{\code{clone()}}{[internal] method to clone the R6 object.}
\item{\code{initialize()}}{[internal] method to initialize the R6 object.}
}
}

\examples{
# We train a random forest on the Boston dataset:
if (require("randomForest")) {
data("Boston", package  = "MASS")
rf = randomForest(medv ~ ., data = Boston, ntree = 50)
mod = Predictor$new(rf, data = Boston)

# Compute the partial dependence for the first feature
pdp.obj = PartialDependence$new(mod, feature = "crim")

# Plot the results directly
plot(pdp.obj)

# Since the result is a ggplot object, you can extend it: 
if (require("ggplot2")) {
 plot(pdp.obj) + theme_bw()
}

# If you want to do your own thing, just extract the data: 
pdp.dat = pdp.obj$results
head(pdp.dat)

# You can reuse the pdp object for other features: 
pdp.obj$set.feature("lstat")
plot(pdp.obj)

# Partial dependence plots support up to two features: 
pdp.obj = PartialDependence$new(mod, feature = c("crim", "lstat"))  
plot(pdp.obj)

# Partial dependence plots also works with multiclass classification
rf = randomForest(Species ~ ., data = iris, ntree=50)
predict.fun = function(object, newdata) predict(object, newdata, type = "prob")
mod = Predictor$new(rf, data = iris, predict.fun = predict.fun)

# For some models we have to specify additional arguments for the predict function
plot(PartialDependence$new(mod, feature = "Sepal.Length"))

# Partial dependence plots support up to two features: 
pdp.obj = PartialDependence$new(mod, feature = c("Sepal.Length", "Petal.Length"))
pdp.obj$plot()   

# For multiclass classification models, you can choose to only show one class:
mod = Predictor$new(rf, data = iris, predict.fun = predict.fun, class = 1)
plot(PartialDependence$new(mod, feature = "Sepal.Length"))
}
}
\references{
Friedman, J.H. 2001. "Greedy Function Approximation: A Gradient Boosting Machine." Annals of Statistics 29: 1189-1232.
}
\seealso{
\link{plot.PartialDependence}

\code{\link{Ice}} for individual conditional expectation plots.
}
