
#   IGraph R package
#   Copyright (C) 2005  Gabor Csardi <csardi@rmki.kfki.hu>
#   MTA RMKI, Konkoly-Thege Miklos st. 29-33, Budapest 1121, Hungary
#   
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#   
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc.,  51 Franklin Street, Fifth Floor, Boston, MA
#   02110-1301 USA
#
###################################################################

graph <- function( edges, n=max(edges)+1, directed=TRUE ) {
  .Call("R_igraph_create", as.numeric(edges), as.numeric(n),
        as.logical(directed),
        PACKAGE="igraph")
}

graph.adjacency <- function( adjmatrix, mode="directed", weighted=NULL ) {
  mode <- switch(mode,
                 "directed"=0,
                 "undirected"=1,
                 "max"=1,
                 "upper"=2,
                 "lower"=3,
                 "min"=4,
                 "plus"=5)
  if (!is.null(weighted)) {
    if (is.logical(weighted) && weighted) {
      weighted <- "weight"
    }
    if (!is.character(weighted)) {
      stop("invalid value supplied for `weighted' argument, please see docs.")
    }

    if (nrow(adjmatrix) != ncol(adjmatrix)) {
      stop("not a square matrix")
    }
    
    no.edges <- sum(adjmatrix > 0)
    edges <- numeric(2*no.edges)
    weight <- numeric(no.edges)
    ptr <- 1
    if (no.edges == 0) {
      res <- graph.empty(directed=(mode==0))
      res <- set.edge.attribute(res, weighted, value=1)
      res
    } else {
      for (i in 1:nrow(adjmatrix)) {
        for (j in 1:ncol(adjmatrix)) {
          if (adjmatrix[i,j] != 0) {
            edges[2*ptr-1] <- i-1
            edges[2*ptr] <- j-1
            weight[ptr] <- adjmatrix[i,j]
            ptr <- ptr + 1
          }          
        }
      }
      res <- graph.empty(n=nrow(adjmatrix), directed=(mode==0))
      weight <- list(weight)
      names(weight) <- weighted
      res <- add.edges(res, edges, attr=weight)
      res
    }
    
  } else {
  
    adjmatrix <- as.matrix(adjmatrix)
    attrs <- attributes(adjmatrix)
    adjmatrix <- as.numeric(adjmatrix)
    attributes(adjmatrix) <- attrs
    
    .Call("R_igraph_graph_adjacency", adjmatrix, as.numeric(mode),
          PACKAGE="igraph")
  }
}
  

graph.star <- function(n, mode="in", center=0 ) {

  if (is.character(mode)) {
    mode <- switch(mode, "out"=0, "in"=1, "undirected"=2)
  }
  .Call("R_igraph_star", as.numeric(n), as.numeric(mode),
        as.numeric(center),
        PACKAGE="igraph")
}

graph.full <- function(n, directed=FALSE, loops=FALSE) {
  .Call("R_igraph_full", as.numeric(n), as.logical(directed),
        as.logical(loops),
        PACKAGE="igraph")
}

###################################################################
# Lattices, every kind
###################################################################

graph.lattice <- function(dimvector=NULL,length=NULL, dim=NULL, nei=1,
                          directed=FALSE, mutual=FALSE, circular=FALSE, ...) {

##   # Check
##   if (is.null(dimvector) && (is.null(length) || is.null(dim))) {
##     stop("Either `length' and `dim' or 'dimvector' must be set. See docs.")
##   }
##   if (!is.null(length) && length < 1) {
##     stop("Invalid `length' argument, should be at least one")
##   }
##   if (!is.null(length) && dim < 1) {
##     stop("Invalid `dim' argument, should be at least one")
##   }
##   if (!is.null(length) && any(dimvector < 1)) {
##     stop("Invalid `dimvector', has negative or smaller than one elements")
##   }
##   if (mutual && !directed) {
##     warning("`mutual' specified for undirected graph, proceeding with multiplex edges...")
##   }
##   if (nei < 1) {
##     stop("`nei' should be at least one")
##   }
  
##   if (!is.null(length)) {
##     length <- as.numeric(length)
##     dim <- as.numeric(dim)
##     dimvector <- rep(length, times=dim)
##   } else {
##     dimvector <- as.numeric(dimvector)
##   }
##   nei <- as.numeric(nei)

##   n <- prod(dimvector)
##   res <- graph.empty(n=n, directed=directed, ...)
##   res <- add.edges(res, .Call("REST_create_lattice", dimvector, n,
##                               circular, mutual, PACKAGE="igraph"))

##   # Connect also to local neighborhood
##   if (nei >= 2) {
##     neighbors <- lapply(1:length(res), function(a) get.neighborhood(res, a))
##     res <- add.edges(res, .Call("REST_connect_neighborhood", neighbors, nei,
##                                 mutual, PACKAGE="igraph"))
##   }
  
##   res

  if (is.null(dimvector)) {
    dimvector <- rep(length, dim)
  }
  
  .Call("R_igraph_lattice", as.numeric(dimvector), as.numeric(nei),
        as.logical(directed), as.logical(mutual),
        as.logical(circular),
        PACKAGE="igraph")
}

graph.ring <- function(n, directed=FALSE, mutual=FALSE, circular=TRUE) {
  .Call("R_igraph_ring", as.numeric(n), as.logical(directed),
        as.logical(mutual), as.logical(circular),
        PACKAGE="igraph")
}

###################################################################
# Trees, regular
###################################################################

graph.tree <- function(n, children=2, mode="out") {
  if (is.character(mode)) {
    mode <- switch(mode, "out"=0, "in"=1, "undirected"=2);
  }

  .Call("R_igraph_tree", as.numeric(n), as.numeric(children),
        as.numeric(mode),
        PACKAGE="igraph")
}

###################################################################
# The graph atlas
###################################################################

graph.atlas <- function(n) {

  .Call("R_igraph_atlas", as.numeric(n),
        PACKAGE="igraph")
}

###################################################################
# Create a graph from a data frame
###################################################################

graph.data.frame <- function(d, directed=TRUE) {

  if (ncol(d) < 2) {
    stop("the data frame should contain at least two columns")
  }

  # assign vertex ids
  names <- unique( c(as.character(d[,1]), as.character(d[,2])) )
  ids <- seq(along=names)-1
  names(ids) <- names

  # create graph
  g <- graph.empty(n=0, directed=directed)
  g <- add.vertices(g, length(ids), name=names)

  # create edge list
  from <- as.character(d[,1])
  to <- as.character(d[,2])
  edges <- t(matrix(c(ids[from], ids[to]), nc=2))

  # edge attributes
  attrs <- list()
  if (ncol(d) > 2) {
    for (i in 3:ncol(d)) {
      newval <- d[,i]
      if (class(newval) == "factor") {
        newval <- as.character(newval)
      }
      attrs[[ names(d)[i] ]] <- newval
    }
  }
  
  # add the edges
  g <- add.edges(g, edges, attr=attrs)
  g
}

graph.edgelist <- function(el, directed=TRUE) {

  if (!is.matrix(el) || ncol(el) != 2) {
    stop("graph.edgelist expects a matrix with two columns")
  }

  if (nrow(el) == 0) {
    res <- graph.empty(directed=directed)
  } else {  
    if (is.character(el)) {
      ## symbolic edge list
      names <- unique(as.character(t(el)))
      ids <- seq(names)-1
      names(ids) <- names
      res <- graph( unname(ids[t(el)]), directed=directed)
      rm(ids)
      V(res)$name <- names
    } else {
      ## normal edge list
      res <- graph( t(el), directed=directed )
    }
  }

  res
}

graph.extended.chordal.ring <- function(n, w) {
  
  .Call("R_igraph_extended_chordal_ring", as.numeric(n),
        as.matrix(w),
        PACKAGE="igraph")
}
