% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/estimateCoeffs.R
\name{estimateCoeffs}
\alias{estimateCoeffs}
\alias{estimateTwoLocusInbreeding}
\alias{estimateInbreeding}
\alias{estimateTwoLocusKappa}
\alias{estimateKappa}
\alias{estimateTwoLocusIdentity}
\alias{estimateIdentity}
\title{Estimation of one- and two-locus relatedness coefficients}
\usage{
estimateTwoLocusInbreeding(
  x,
  id,
  rho = NULL,
  cM = NULL,
  Nsim,
  Xchrom = FALSE,
  verbose = FALSE,
  ...
)

estimateInbreeding(x, id, Nsim, Xchrom = FALSE, verbose = FALSE, ...)

estimateTwoLocusKappa(
  x,
  ids,
  rho = NULL,
  cM = NULL,
  Nsim,
  Xchrom = FALSE,
  verbose = FALSE,
  ...
)

estimateKappa(x, ids, Nsim, Xchrom = FALSE, verbose = FALSE, ...)

estimateTwoLocusIdentity(
  x,
  ids,
  rho = NULL,
  cM = NULL,
  Nsim,
  Xchrom = FALSE,
  verbose = FALSE,
  ...
)

estimateIdentity(x, ids, Nsim, Xchrom = FALSE, verbose = FALSE, ...)
}
\arguments{
\item{x}{A pedigree in the form of a \code{\link[pedtools:ped]{pedtools::ped()}} object.}

\item{id, ids}{A vector of one or two ID labels.}

\item{rho}{A scalar in the interval \verb{[0, 0.5]}: the recombination fraction
between the two loci, converted to centiMorgan using Haldanes map function:
cM = -50 * log(1 - 2*rho). Either \code{rho} or \code{cM} (but not both) must be
non-NULL.}

\item{cM}{A non-negative number: the genetic distance between the two loci,
given in centiMorgans. Either \code{rho} or \code{cM} (but not both) must be
non-NULL.}

\item{Nsim}{The number of simulations.}

\item{Xchrom}{A logical indicating if the loci are X-linked (if TRUE) or
autosomal (FALSE).}

\item{verbose}{A logical.}

\item{...}{Further arguments passed on to \code{\link[=ibdsim]{ibdsim()}}, e.g. \code{seed}.}
}
\value{
\code{estimateInbreeding()}: a single probability.

\code{estimateTwoLocusInbreeding()}: a single probability.

\code{estimateKappa()}: a numeric vector of length 3, with the estimated
\eqn{\kappa} coefficients.

\code{estimateTwoLocusKappa()}: a symmetric, numerical 3*3 matrix, with the
estimated values of \eqn{\kappa_{ij}}, for \eqn{i,j = 0,1,2}.

\code{estimateIdentity()}: a numeric vector of length 9, with the
estimated identity coefficients.

\code{estimateTwoLocusIdentity()}: symmetric, numerical 9*9 matrix, with the
estimated two-locus identity coefficients.
}
\description{
Estimate by simulation various relatedness coefficients, and two-locus
versions of the same coefficients, for a given recombination rate. The
current implementation covers inbreeding coefficients, pairwise IBD (kappa)
coefficients and pairwise condensed identity coefficients. These functions
are primarily meant as tools for validating exact algorithms, e.g., as
implemented in the \code{ribd} package.
}
\details{
In the following, let L1 and L2 denote two arbitrary autosomal loci with
recombination rate \eqn{\rho}.

The \emph{two-locus inbreeding coefficient} \eqn{f_2(\rho)} of a pedigree member
is defined as the probability of being autozygous at both L1 and L2
simultaneously.

The \emph{two-locus kappa coefficients} \eqn{\kappa_{ij}(\rho)}, for \eqn{i,j =
0,1,2}, of individuals A and B are defined by as the probability that A and B
share i alleles IBD at L1, and j alleles IBD at L2.

The \emph{two-locus identity coefficients} \eqn{\Delta_{ij}}, \eqn{i,j = 1,...,9}
are defined similarly to the two-locus kappa above. For a description of the
identity states, see e.g., \url{https://github.com/magnusdv/ribd}.
}
\examples{

### Two-locus inbreeding ###

x = cousinPed(0, child = TRUE)
rho = 0.25
Nsim = 10 # Increase!
estimateTwoLocusInbreeding(x, id = 5, rho = rho, Nsim = Nsim)


### Two-locus kappa: Grandparent vs half sib vs uncle ###

# These are indistinguishable with unlinked loci, see e.g.
# pages 182-183 in Egeland, Kling and Mostad (2016).
# Each simulations followed by exact counterpart.

rho = 0.25; R = .5 * (rho^2 + (1-rho)^2)
Nsim = 10 # Should be increased to at least 10000

# Grandparent/grandchild
G = linearPed(2); G.ids = c(1,5); #plot(G, shaded = G.ids)
estimateTwoLocusKappa(G, G.ids, rho = rho, Nsim = Nsim)[2,2]
.5*(1-rho) # exact

# Half sibs
H = halfSibPed(); H.ids = c(4,5); # plot(H, shaded = H.ids)
estimateTwoLocusKappa(H, H.ids, rho = rho, Nsim = Nsim)[2,2]
R # exact

# Uncle
U = cousinPed(0, removal = 1); U.ids = c(3,6); # plot(U, shaded = U.ids)
estimateTwoLocusKappa(U, U.ids, rho = rho, Nsim = Nsim)[2,2]
(1-rho) * R + rho/4 # exact


### Two-locus Jacquard ###
x = fullSibMating(1)
rho = 0.25
Nsim = 10 # (increase for more accurate estimates!)

estimateTwoLocusIdentity(x, ids = 5:6, rho = rho, Nsim = Nsim)

}
