% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/model_def.R
\name{define_model}
\alias{define_model}
\alias{eval_model}
\title{Define and evaluate model expression}
\usage{
define_model(tparams_def, rng_def, params = NULL, n_states = NULL)

eval_model(x, input_data)
}
\arguments{
\item{tparams_def}{A \link[=define_tparams]{tparams_def} object or a list of
\link[=define_tparams]{tparams_def} objects. A list might be considered if time intervals
specified with the \code{times} argument in \code{\link[=define_tparams]{define_tparams()}} vary across parameters.
Parameters for a transition probability matrix (\code{tpmatrix}), utilities (\code{utility}),
and/or cost categories (\code{costs}) are returned as a named list (see \code{\link[=define_tparams]{define_tparams()}}
for more details).}

\item{rng_def}{A \link[=define_rng]{rng_def} object used to randomly draw samples
of the parameters from suitable probability distributions.}

\item{params}{A list containing the values of parameters for random
number generation.}

\item{n_states}{The number of health states (inclusive of all health states
including the the death state) in the model. If \code{tpmatrix} is
an element returned by \code{tparams_def}, then it will be equal to the number of states
in the transition probability matrix; otherwise it must be specified as an argument.}

\item{x}{An object of class \code{model_def} created with \code{define_model()}.}

\item{input_data}{An object of class \link[=expand.hesim_data]{expanded_hesim_data}
expanded by patients and treatment strategies.}
}
\value{
\code{define_model()} returns an object of class \code{model_def},
which is a list containing the arguments to the function. \code{eval_model()} returns
a list containing \link[=id_attributes]{ID} variables
identifying parameter samples, treatment strategies, patient cohorts, and time
intervals; the values of parameters of the transition probability matrix,
utilities, and/or cost categories; the number of health states; and the number
of random number generation samples for the PSA.
}
\description{
A model expression is defined by specifying random number generation functions
for a probabilistic sensitivity analysis (PSA) and transformations of the sampled
parameters as a function of \code{input_data}. The unevaluated expressions
are evaluated with \code{eval_model()} and used to generate the model inputs needed to
create an economic model.
}
\details{
\code{eval_model()} evaluates the expressions in an object of class
\code{model_def} returned by \code{define_model()} and is, in turn, used within
functions that instantiate economic models (e.g., \code{\link[=create_CohortDtstm]{create_CohortDtstm()}}).
The direct output of \code{eval_model()} can also be useful for understanding and debugging
model definitions, but it is not used directly for simulation.

Economic models are constructed as a function of input data and parameters:
\enumerate{
\item \emph{Input data}: Objects of class \link[=expand.hesim_data]{expanded_hesim_data}
consisting of the treatment strategies and patient population.
\item \emph{Parameters}: The underlying parameter estimates from the literature
are first stored in a list (\code{params} argument). Random number generation
is then used to sample the parameters from suitable probability distributions
for the PSA (\code{rng_def} argument). Finally, the sampled parameters are
transformed as a function of the input data into values (e.g., elements of a
transition probability matrix) used for the simulation (\code{tparams_def} argument).
The \code{params} argument can be omitted if the underlying parameters values are
defined inside a \code{define_rng()} block.
}
}
\examples{

# Data
library("data.table")
strategies <- data.table(strategy_id = 1:2,
                         strategy_name = c("Monotherapy", "Combination therapy"))
patients <- data.table(patient_id = 1)
hesim_dat <- hesim_data(strategies = strategies,
                       patients = patients)
data <- expand(hesim_dat)

# Define the model
rng_def <- define_rng({
  alpha <- matrix(c(1251, 350, 116, 17,
                    0, 731, 512, 15,
                    0, 0, 1312, 437,
                    0, 0, 0, 469),
                  nrow = 4, byrow = TRUE)
  rownames(alpha) <- colnames(alpha) <- c("A", "B", "C", "D")
  lrr_mean <- log(.509)
  lrr_se <- (log(.710) - log(.365))/(2 * qnorm(.975))
  list(
    p_mono = dirichlet_rng(alpha),
    rr_comb = lognormal_rng(lrr_mean, lrr_se),
    u = 1,
    c_zido = 2278,
    c_lam = 2086.50,
    c_med = gamma_rng(mean = c(A = 2756, B = 3052, C = 9007),
                        sd = c(A = 2756, B = 3052, C = 9007))
  )
}, n = 2)

tparams_def <- define_tparams({
  rr = ifelse(strategy_name == "Monotherapy", 1, rr_comb)
  list(
    tpmatrix = tpmatrix(
      C, p_mono$A_B * rr, p_mono$A_C * rr, p_mono$A_D * rr,
      0, C, p_mono$B_C * rr, p_mono$B_D * rr,
      0, 0, C, p_mono$C_D * rr,
      0, 0, 0, 1),
    utility = u,
    costs = list(
      drug = ifelse(strategy_name == "Monotherapy",
                    c_zido, c_zido + c_lam),
      medical = c_med
    ) 
  )
})

model_def <- define_model(
  tparams_def = tparams_def,
  rng_def = rng_def)

# Evaluate the model expression to generate model inputs
# This can be useful for understanding the output of a model expression
eval_model(model_def, data)

# Create an economic model with a factory function
econmod <- create_CohortDtstm(model_def, data)

}
\seealso{
\code{\link[=define_tparams]{define_tparams()}}, \code{\link[=define_rng]{define_rng()}}
}
