% Generated by roxygen2 (4.1.1): do not edit by hand
% Please edit documentation in R/hdnom.validate.R
\name{hdnom.validate}
\alias{hdnom.validate}
\title{Validate High-Dimensional Cox models with time-dependent AUC}
\usage{
hdnom.validate(x, time, event, model.type = c("lasso", "alasso", "flasso",
  "enet", "aenet", "mcp", "mnet", "scad", "snet"), alpha, lambda,
  pen.factor = NULL, gamma, method = c("bootstrap", "cv", "repeated.cv"),
  boot.times = NULL, nfolds = NULL, rep.times = NULL,
  tauc.type = c("CD", "SZ", "UNO"), tauc.time, trace = TRUE)
}
\arguments{
\item{x}{Matrix of training data used for fitting the model;
on which to run the validation.}

\item{time}{Survival time.
Must be of the same length with the number of rows as \code{x}.}

\item{event}{Status indicator, normally 0 = alive, 1 = dead.
Must be of the same length with the number of rows as \code{x}.}

\item{model.type}{Model type to validate. Could be one of \code{"lasso"},
\code{"alasso"}, \code{"flasso"}, \code{"enet"}, \code{"aenet"},
\code{"mcp"}, \code{"mnet"}, \code{"scad"}, or \code{"snet"}.}

\item{alpha}{Value of the elastic-net mixing parameter alpha for
\code{enet}, \code{aenet}, \code{mnet}, and \code{snet} models.
\code{alpha=1}: lasso (l1) penalty; \code{alpha=0}: ridge (l2) penalty.
Note that for \code{mnet} and \code{snet} models,
\code{alpha} can be set to very close to 0 but not 0 exactly.}

\item{lambda}{Value of the penalty parameter lambda to use in the
model fits on the resampled data. From the built Cox model.}

\item{pen.factor}{Penalty factors to apply to each coefficient.
From the built \emph{adaptive lasso} or \emph{adaptive elastic-net} model.}

\item{gamma}{Value of the model parameter gamma for MCP/SCAD/Mnet/Snet models.}

\item{method}{Validation method.
Could be \code{"bootstrap"}, \code{"cv"}, or \code{"repeated.cv"}.}

\item{boot.times}{Number of repetitions for bootstrap.}

\item{nfolds}{Number of folds for cross-validation and
repeated cross-validation.}

\item{rep.times}{Number of repeated times for repeated cross-validation.}

\item{tauc.type}{Type of time-dependent AUC.
Including \code{"CD"} proposed by Chambless and Diao (2006).,
\code{"SZ"} proposed by Song and Zhou (2008).,
\code{"UNO"} proposed by Uno et al. (2007).}

\item{tauc.time}{Numeric vector. Time points at which to evaluate
the time-dependent AUC.}

\item{trace}{Logical. Output the validation progress or not.
Default is \code{TRUE}.}
}
\description{
Validate High-Dimensional Cox models with time-dependent AUC
}
\examples{
library("glmnet")
library("survival")
library("rms")

# Load imputed SMART data
data(smart)
x = as.matrix(smart[, -c(1, 2)])
time = smart$TEVENT
event = smart$EVENT

# Fit penalized Cox model (lasso penalty) with glmnet
set.seed(1010)
cvfit = cv.glmnet(x, Surv(time, event), family = "cox", nfolds = 5)

# Model validation by bootstrap with time-dependent AUC
# Normally boot.times should be set to 200 or more,
# we set it to 3 here only to save example running time.
val.boot = hdnom.validate(x, time, event, model.type = 'lasso',
                          alpha = 1, lambda = cvfit$lambda.1se,
                          method = "bootstrap", boot.times = 3,
                          tauc.type = "UNO", tauc.time = seq(0.25, 2, 0.25) * 365)

# Model validation by 10-fold cross-validation with time-dependent AUC
val.cv = hdnom.validate(x, time, event, model.type = 'lasso',
                        alpha = 1, lambda = cvfit$lambda.1se,
                        method = "cv", nfolds = 5,
                        tauc.type = "UNO", tauc.time = seq(0.25, 2, 0.25) * 365)

# Model validation by repeated cross-validation with time-dependent AUC
val.repcv = hdnom.validate(x, time, event, model.type = 'lasso',
                           alpha = 1, lambda = cvfit$lambda.1se,
                           method = "repeated.cv", nfolds = 5, rep.times = 3,
                           tauc.type = "UNO", tauc.time = seq(0.25, 2, 0.25) * 365)

# bootstrap-based discrimination curves has a very narrow band
print(val.boot)
summary(val.boot)
plot(val.boot, ylim = c(0.4, 0.8))

# k-fold cv provides a more strict evaluation than bootstrap
print(val.cv)
summary(val.cv)
plot(val.cv, ylim = c(0.4, 0.8))

# repeated cv provides similar results as k-fold cv
# but more stable than k-fold cv
print(val.repcv)
summary(val.repcv)
plot(val.repcv, ylim = c(0.4, 0.8))
}
\references{
Chambless, L. E. and G. Diao (2006).
Estimation of time-dependent area under the ROC curve for long-term
risk prediction.
\emph{Statistics in Medicine} 25, 3474--3486.

Song, X. and X.-H. Zhou (2008).
A semiparametric approach for the covariate specific ROC curve with
survival outcome.
\emph{Statistica Sinica} 18, 947--965.

Uno, H., T. Cai, L. Tian, and L. J. Wei (2007).
Evaluating prediction rules for t-year survivors with censored
regression models.
\emph{Journal of the American Statistical Association} 102, 527--537.
}

