% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rlasso.R
\name{rlasso}
\alias{rlasso}
\alias{rlasso.character}
\alias{rlasso.default}
\alias{rlasso.formula}
\title{rlasso: Function for Lasso estimation under homoscedastic and heteroscedastic non-Gaussian
disturbances}
\usage{
rlasso(x, ...)

\method{rlasso}{formula}(formula, data = NULL, post = TRUE,
  intercept = TRUE, model = TRUE, penalty = list(homoscedastic = FALSE,
  X.dependent.lambda = FALSE, lambda.start = NULL, c = 1.1, gamma = 0.1/log(n)),
  control = list(numIter = 15, tol = 10^-5, threshold = NULL), ...)

\method{rlasso}{character}(x, data = NULL, post = TRUE, intercept = TRUE,
  model = TRUE, penalty = list(homoscedastic = FALSE, X.dependent.lambda =
  FALSE, lambda.start = NULL, c = 1.1, gamma = 0.1/log(n)),
  control = list(numIter = 15, tol = 10^-5, threshold = NULL), ...)

\method{rlasso}{default}(x, y, post = TRUE, intercept = TRUE,
  model = TRUE, penalty = list(homoscedastic = FALSE, X.dependent.lambda =
  FALSE, lambda.start = NULL, c = 1.1, gamma = 0.1/log(n)),
  control = list(numIter = 15, tol = 10^-5, threshold = NULL), ...)
}
\arguments{
\item{x}{regressors (vector, matrix or object can be coerced to matrix)}

\item{...}{further arguments (only for consistent defintion of methods)}

\item{formula}{an object of class "formula" (or one that can be coerced to
that class): a symbolic description of the model to be fitted in the form
\code{y~x}}

\item{data}{an optional data frame, list or environment (or object coercible
by as.data.frame to a data frame) containing the variables in the model. If
not found in data, the variables are taken from environment(formula),
typically the environment from which \code{rlasso} is called.}

\item{post}{logical. If \code{TRUE}, post-Lasso estimation is conducted.}

\item{intercept}{logical. If \code{TRUE}, intercept is included which is not
penalized.}

\item{model}{logical. If \code{TRUE} (default), model matrix is returned.}

\item{penalty}{list with options for the calculation of the penalty. 
\itemize{
\item{\code{c} and \code{gamma}}{ constants for the penalty with default \code{c=1.1} and \code{gamma=0.1}}
\item{\code{homoscedastic}}{ logical, if homoscedastic errors are considered (default \code{FALSE}). Option \code{none} is described below.}
\item{\code{X.dependent.lambda}}{ logical,  \code{TRUE}, if the penalization parameter depends on the the design of the matrix \code{x}. \code{FALSE}, if independent of the design matrix  (default).}
\item{\code{numSim}}{ number of simulations for the dependent methods, default=5000}
\item{\code{lambda.start}}{ initial penalization value, compulsory for method "none"}
}}

\item{control}{list with control values.
\code{numIter} number of iterations for the algorithm for
the estimation of the variance and data-driven penalty, ie. loadings,
\code{tol} tolerance for improvement of the estimated variances.
\code{threshold} is applied to the final estimated lasso
coefficients. Absolute values below the threshold are set to zero.}

\item{y}{dependent variable (vector, matrix or object can be coerced to matrix)}
}
\value{
\code{rlasso} returns an object of class \code{rlasso}. An object of
class "rlasso" is a list containing at least the following components:
\item{coefficients}{parameter estimates}
\item{beta}{parameter estimates (named vector of coefficients without intercept)}
\item{intercept}{value of the intercept}
\item{index}{index of selected
variables (logical vector)} \item{lambda}{data-driven penalty term for each
variable, product of lambda0 (the penalization parameter) and the loadings}
\item{lambda0}{penalty term} \item{loadings}{loading for each regressor}
\item{residuals}{residuals, response minus fitted values} \item{sigma}{root of the variance of
the residuals} \item{iter}{number of iterations} \item{call}{function call}
\item{options}{options}
\item{model}{model matrix (if \code{model = TRUE} in function call)}
}
\description{
The function estimates the coefficients of a Lasso regression with
data-driven penalty under homoscedasticity and heteroscedasticity with non-Gaussian noise and X-dependent or X-independent design. The
method of the data-driven penalty can be chosen. The object which is
returned is of the S3 class \code{rlasso}.
}
\details{
The function estimates the coefficients of a Lasso regression with
data-driven penalty under homoscedasticity / heteroscedasticity and non-Gaussian noise. The options \code{homoscedastic} is a logical with \code{FALSE} by default.
Moreover, for the calculation of the penalty parameter it can be chosen, if the penalization parameter depends on the  design matrix (\code{X.dependent.lambda=TRUE}) or \code{independent} (default, \code{X.dependent.lambda=FALSE}).
The default value of the constant \code{c} is \code{1.1} in the post-Lasso case and \code{0.5} in the Lasso case. 
 A \emph{special} option is to set \code{homoscedastic} to \code{none} and to supply a values \code{lambda.start}. Then this value is used as penalty parameter with independent design and heteroscedastic errors to weight the regressors.
For details of the
implementation of the Algorithm for estimation of the data-driven penalty,
in particular the regressor-independent loadings, we refer to Appendix A in
Belloni et al. (2012). When the option "none" is chosen for \code{homoscedastic} (together with
\code{lambda.start}), lambda is set to \code{lambda.start} and the
regressor-independent loadings und heteroscedasticity are used. The options "X-dependent" and
"X-independent" under homoscedasticity are described in Belloni et al. (2013).

The option \code{post=TRUE} conducts post-lasso estimation, i.e. a refit of
the model with the selected variables.
}
\examples{
set.seed(1)
n = 100 #sample size
p = 100 # number of variables
s = 3 # nubmer of variables with non-zero coefficients
X = Xnames = matrix(rnorm(n*p), ncol=p)
colnames(Xnames) <- paste("V", 1:p, sep="")
beta = c(rep(5,s), rep(0,p-s))
Y = X\%*\%beta + rnorm(n)
reg.lasso <- rlasso(Y~Xnames)
Xnew = matrix(rnorm(n*p), ncol=p)  # new X
colnames(Xnew) <- paste("V", 1:p, sep="")
Ynew =  Xnew\%*\%beta + rnorm(n)  #new Y
yhat = predict(reg.lasso, newdata = Xnew)
}
\references{
A. Belloni, D. Chen, V. Chernozhukov and C. Hansen (2012).
Sparse models and methods for optimal instruments with an application to
eminent domain. \emph{Econometrica} 80 (6), 2369-2429.

A. Belloni, V. Chernozhukov and C. Hansen (2013). Inference for
high-dimensional sparse econometric models. In Advances in Economics and
Econometrics: 10th World Congress, Vol. 3: Econometrics, Cambirdge
University Press: Cambridge, 245-295.
}

